/*
 * mweather-details.c: Maemo details widget
 *
 * Copyright (C) 2006 Philip Langdale
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *     Philip Langdale <philipl@mail.utexas.edu>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>

#include "mweather-details.h"


struct _MWeatherDetailsPrivate {
   MWeatherLocation *location;
   gulong updateConnection;

   GtkWidget *radarMap;
   GtkTextBuffer *forecastText;

   GtkWidget *city;
   GtkWidget *lastUpdate;
   GtkWidget *conditions;
   GtkWidget *sky;
   GtkWidget *temp;
   GtkWidget *feelsLike;
   GtkWidget *dewPoint;
   GtkWidget *humidity;
   GtkWidget *wind;
   GtkWidget *pressure;
   GtkWidget *visibility;
   GtkWidget *sunrise;
   GtkWidget *sunset;
};

enum
{
   PROP_0,
   PROP_LOCATION,
};

enum
{
   PAGE_MAP,
   PAGE_DETAILS,
   PAGE_FORECAST,
   NUM_PAGES,
};

G_DEFINE_TYPE(MWeatherDetails, mweather_details, GTK_TYPE_NOTEBOOK)
#define MWEATHER_DETAILS_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), MWEATHER_TYPE_DETAILS, MWeatherDetailsPrivate)) 


static void
populate_window(MWeatherDetails *details)
{
   GtkWidget *mapPage;
   GtkWidget *detailsTable;
   GtkWidget *forecastPage;
   GtkWidget *forecast;
   GtkWidget *label;

   mapPage = gtk_scrolled_window_new(NULL, NULL);
   gtk_widget_show(mapPage);
   gtk_notebook_insert_page(GTK_NOTEBOOK(details),
                            mapPage,
                            gtk_label_new(_("Map")),
                            PAGE_MAP);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mapPage),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(mapPage),
                                       GTK_SHADOW_ETCHED_IN);

   details->priv->radarMap = gtk_image_new();
   gtk_widget_show(details->priv->radarMap);
   gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(mapPage),
                                         details->priv->radarMap);

   detailsTable = gtk_table_new(13, 2, FALSE);
   gtk_widget_show(detailsTable);
   gtk_notebook_insert_page(GTK_NOTEBOOK(details),
                            detailsTable,
                            gtk_label_new(_("Details")),
                            PAGE_DETAILS);
   gtk_container_set_border_width(GTK_CONTAINER(detailsTable), 6);
   gtk_table_set_row_spacings(GTK_TABLE(detailsTable), 6);
   gtk_table_set_col_spacings(GTK_TABLE(detailsTable), 6);

   label = gtk_label_new(_("City:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 0, 1,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Last update:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 1, 2,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Conditions:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 2, 3,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Sky:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 3, 4,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Temperature:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 4, 5,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Feels like:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 5, 6,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Dew point:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 6, 7,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Relative humidity:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 7, 8,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Wind:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 8, 9,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Pressure:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 9, 10,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Visibility:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 10, 11,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Sunrise:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 11, 12,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   label = gtk_label_new(_("Sunset:"));
   gtk_widget_show(label);
   gtk_table_attach(GTK_TABLE(detailsTable), label, 0, 1, 12, 13,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

   details->priv->city = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->city);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->city, 1, 2, 0, 1,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->city), 0, 0.5);

   details->priv->lastUpdate = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->lastUpdate);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->lastUpdate, 1, 2, 1, 2,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->lastUpdate), 0, 0.5);

   details->priv->conditions = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->conditions);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->conditions, 1, 2, 2, 3,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->conditions), 0, 0.5);

   details->priv->sky = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->sky);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->sky, 1, 2, 3, 4,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->sky), 0, 0.5);

   details->priv->temp = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->temp);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->temp, 1, 2, 4, 5,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->temp), 0, 0.5);

   details->priv->feelsLike = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->feelsLike);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->feelsLike, 1, 2, 5, 6,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->feelsLike), 0, 0.5);

   details->priv->dewPoint = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->dewPoint);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->dewPoint, 1, 2, 6, 7,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->dewPoint), 0, 0.5);

   details->priv->humidity = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->humidity);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->humidity, 1, 2, 7, 8,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->humidity), 0, 0.5);

   details->priv->wind = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->wind);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->wind, 1, 2, 8, 9,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->wind), 0, 0.5);

   details->priv->pressure = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->pressure);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->pressure, 1, 2, 9, 10,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->pressure), 0, 0.5);

   details->priv->visibility = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->visibility);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->visibility, 1, 2, 10, 11,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->visibility), 0, 0.5);

   details->priv->sunrise = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->sunrise);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->sunrise, 1, 2, 11, 12,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->sunrise), 0, 0.5);

   details->priv->sunset = gtk_label_new(_("XXX"));
   gtk_widget_show(details->priv->sunset);
   gtk_table_attach(GTK_TABLE(detailsTable), details->priv->sunset, 1, 2, 12, 13,
                    0, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(details->priv->sunset), 0, 0.5);

   forecastPage = gtk_scrolled_window_new(NULL, NULL);
   gtk_widget_show(forecastPage);
   gtk_notebook_insert_page(GTK_NOTEBOOK(details),
                            forecastPage,
                            gtk_label_new(_("Forecast")),
                            PAGE_FORECAST);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(forecastPage),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(forecastPage),
                                       GTK_SHADOW_ETCHED_IN);
   forecast = gtk_text_view_new();
   gtk_widget_show(forecast);
   gtk_container_add(GTK_CONTAINER(forecastPage), forecast);

   details->priv->forecastText = gtk_text_view_get_buffer(GTK_TEXT_VIEW(forecast));
}


static void
location_updated_cb(MWeatherDetails *details,
                    MWeatherLocation *location)
{
   GdkPixbufAnimation *radar = NULL;
   const gchar *forecast = NULL;
   const gchar *name = NULL;
   const gchar *update = NULL;
   const gchar *conditions = NULL;
   const gchar *sky = NULL;
   const gchar *temp = NULL;
   const gchar *feel = NULL;
   const gchar *dewpoint = NULL;
   const gchar *humidity = NULL;
   const gchar *wind = NULL;
   const gchar *pressure = NULL;
   const gchar *visibility = NULL;
   const gchar *sunrise = NULL;
   const gchar *sunset = NULL;

   if (location && mweather_location_is_valid(location)) {
      radar = mweather_location_get_radar(location);
      forecast = mweather_location_get_forecast(location);
      name = mweather_location_get_name(location);
      update = mweather_location_get_update(location);
      conditions = mweather_location_get_cond(location);
      sky = mweather_location_get_sky(location);
      temp = mweather_location_get_temp(location);
      feel = mweather_location_get_feel(location);
      dewpoint = mweather_location_get_dewpoint(location);
      humidity = mweather_location_get_humidity(location);
      wind = mweather_location_get_wind(location);
      pressure = mweather_location_get_pressure(location);
      visibility = mweather_location_get_visi(location);
      sunrise = mweather_location_get_sunrise(location);
      sunset = mweather_location_get_sunset(location);
   }

   gtk_image_set_from_animation(GTK_IMAGE(details->priv->radarMap), radar);
   gtk_text_buffer_set_text(details->priv->forecastText, forecast ? forecast : "", -1);
   gtk_label_set_text(GTK_LABEL(details->priv->city), name);
   gtk_label_set_text(GTK_LABEL(details->priv->lastUpdate), update);
   gtk_label_set_text(GTK_LABEL(details->priv->conditions), conditions);
   gtk_label_set_text(GTK_LABEL(details->priv->sky), sky);
   gtk_label_set_text(GTK_LABEL(details->priv->temp), temp);
   gtk_label_set_text(GTK_LABEL(details->priv->feelsLike), feel);
   gtk_label_set_text(GTK_LABEL(details->priv->dewPoint), dewpoint);
   gtk_label_set_text(GTK_LABEL(details->priv->humidity), humidity);
   gtk_label_set_text(GTK_LABEL(details->priv->wind), wind);
   gtk_label_set_text(GTK_LABEL(details->priv->pressure), pressure);
   gtk_label_set_text(GTK_LABEL(details->priv->visibility), visibility);
   gtk_label_set_text(GTK_LABEL(details->priv->sunrise), sunrise);
   gtk_label_set_text(GTK_LABEL(details->priv->sunset), sunset);
}


static void
mweather_details_set_property(GObject *object,
                             guint prop_id,
                             const GValue *value,
                             GParamSpec *pspec)
{
   MWeatherDetails *details = MWEATHER_DETAILS(object);

   switch (prop_id) {
      case PROP_LOCATION:
         mweather_details_set_location(details,
                                       MWEATHER_LOCATION(g_value_get_object(value)));
         break;
   }
}


static void
mweather_details_get_property(GObject *object,
                             guint prop_id,
                             GValue *value,
                             GParamSpec *pspec)
{
   MWeatherDetails *details = MWEATHER_DETAILS(object);

   switch (prop_id) {
      case PROP_LOCATION:
         g_value_set_object(value, details->priv->location);
         break;
   }
}


static void
mweather_details_init (MWeatherDetails *self)
{
   self->priv = MWEATHER_DETAILS_GET_PRIVATE(self);

   populate_window(self);
}


static GObject *
mweather_details_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
   GObject *object;
   MWeatherDetails *self;

   object = G_OBJECT_CLASS(mweather_details_parent_class)->
      constructor(type, n_construct_params, construct_params);
   self = MWEATHER_DETAILS(object);

   return object;
}


static void
mweather_details_finalize (GObject *object)
{
   MWeatherDetails *self = MWEATHER_DETAILS(object);

   if (self->priv->location) {
      if (g_signal_handler_is_connected(self->priv->location,
                                        self->priv->updateConnection)) {
         g_signal_handler_disconnect(self->priv->location,
                                     self->priv->updateConnection);
      }
      g_object_unref(self->priv->location);
   }

   G_OBJECT_CLASS(mweather_details_parent_class)->finalize(object);
}


GtkWidget *
mweather_details_new (void)
{
    return g_object_new(MWEATHER_TYPE_DETAILS,
                        NULL);
}


static void
mweather_details_class_init (MWeatherDetailsClass *klass)
{
   GObjectClass *object_class = G_OBJECT_CLASS(klass);

   mweather_details_parent_class = g_type_class_peek_parent(klass);

   object_class->set_property = mweather_details_set_property;
   object_class->get_property = mweather_details_get_property;
   object_class->constructor = mweather_details_constructor;
   object_class->finalize = mweather_details_finalize;

   g_object_class_install_property(object_class,
                                   PROP_LOCATION,
                                   g_param_spec_object("location",
                                                       "Location",
                                                       "The MWeather Location",
                                                       MWEATHER_TYPE_LOCATION,
                                                       G_PARAM_READWRITE));

   g_type_class_add_private(klass, sizeof(MWeatherDetailsPrivate));
}


void
mweather_details_set_location(MWeatherDetails *details,
                              MWeatherLocation *location)
{
   MWeatherLocation *oldLocation = details->priv->location;

   details->priv->location = location;
   if (details->priv->location) {
      g_object_ref(details->priv->location);
   }

   if (oldLocation) {
      if (g_signal_handler_is_connected(oldLocation,
                                        details->priv->updateConnection)) {
         g_signal_handler_disconnect(oldLocation,
                                     details->priv->updateConnection);
         details->priv->updateConnection = 0;
      }
      g_object_unref(oldLocation);
   }

   if (details->priv->location) {
      details->priv->updateConnection =
         g_signal_connect_swapped(details->priv->location, "updated",
                                  G_CALLBACK(location_updated_cb), details);
   }
   location_updated_cb(details, details->priv->location);
}
