/*
 * mweather-home.c: Hildon homepage applet using gweather backend.
 *
 * Copyright (C) 2005 Philip Langdale
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *     Philip Langdale <philipl@mail.utexas.edu>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
//#include <hildon-home-plugin/hildon-home-plugin-interface.h>
#include <hildon-widgets/hildon-banner.h>
#include <libosso.h>

#include <libgweather/weather.h>
#include <libgweather/gweather-prefs.h>

#define APPLET_X_SIZE 290
#define APPLET_Y_SIZE 110

typedef struct
{
   osso_context_t *ctxt;

   WeatherInfo *info;

   GWeatherGConf *gconf;
   GWeatherPrefs prefs;

   guint updateTimer;
   gboolean updating;
   gboolean newLocationPending;

   GtkWidget *frame;
   GtkWidget *hbox;
   GtkWidget *ev;
   GtkWidget *vbox;
   GtkWidget *icon;
   GtkWidget *temp;
   GtkWidget *location;
   GtkWidget *aspect;
   GtkWidget *alignment;
   GtkWidget *button;
   GtkWidget *refresh;

   GtkWidget *banner;

} MWeatherHomeCtx;

static void
on_weather_info_finished(WeatherInfo *info, MWeatherHomeCtx *ctx)
{
   fprintf(stderr, "on_weather_info_finished\n");
   ctx->updating = FALSE;
   gtk_widget_destroy(ctx->banner);
   ctx->banner = NULL;

   gchar *temp = NULL;
   if (weather_info_is_valid(info)) {
      GdkPixbuf *buf;
      temp = g_strdup(weather_info_get_temp_summary(info));

      gtk_label_set_text(GTK_LABEL(ctx->location),
                         weather_info_get_location_name(info));

      weather_info_get_pixbuf(info, &buf);
      gtk_image_set_from_pixbuf(GTK_IMAGE(ctx->icon), buf);
   } else {
      temp = g_strdup(_("Failed"));
   }
   gtk_label_set_text(GTK_LABEL(ctx->temp), temp);
   g_free(temp);

   if (ctx->newLocationPending) {
      weather_info_free(ctx->info);
      ctx->info = NULL;
      ctx->newLocationPending = FALSE;
   }
}

static gboolean
get_weather_info(MWeatherHomeCtx *ctx)
{
   gboolean started;
   WeatherPrefs prefs;

   if (ctx->updating) {
      return TRUE;
   }

   prefs.type = FORECAST_ZONE;
   prefs.radar = FALSE;
   prefs.radar_custom_url = NULL;
   prefs.temperature_unit = ctx->prefs.temperature_unit;
   prefs.speed_unit = SPEED_UNIT_DEFAULT;
   prefs.pressure_unit = PRESSURE_UNIT_DEFAULT;
   prefs.distance_unit = DISTANCE_UNIT_DEFAULT;

   if (!ctx->info) {
      WeatherLocation *location = gweather_gconf_get_location(ctx->gconf);
      ctx->info = weather_info_new(location, &prefs,
                                   (WeatherInfoFunc)on_weather_info_finished, ctx);
      weather_location_free(location);
      started = ctx->info != NULL;
   } else {
      WeatherInfo *info = weather_info_update(ctx->info, &prefs,
                                              (WeatherInfoFunc)on_weather_info_finished, ctx);
      started = info != NULL;
   }

   if (started) {
      ctx->updating = TRUE;
      ctx->banner = hildon_banner_show_animation(GTK_WIDGET(gtk_widget_get_toplevel(ctx->frame)),
                                                 NULL, _("Updating"));
   }

   return TRUE; /* When called as a timeout, keep the timer */
}

static void
on_gconf_value_changed(GConfClient *gclient,
                       const char *key,
                       GConfValue *value,
                       MWeatherHomeCtx *ctx)
{
   gweather_prefs_load(&ctx->prefs, ctx->gconf);

   fprintf(stderr, "on-gconf-value-changed %s\n", key);

   if (strstr(key, "auto_update")) {
      if (ctx->updateTimer) {
         g_source_remove(ctx->updateTimer);
         ctx->updateTimer = 0;
      }

      if (ctx->prefs.update_enabled) {
         ctx->updateTimer =
            g_timeout_add(ctx->prefs.update_interval * 1000,
                          (GSourceFunc)get_weather_info, ctx);
      }
   } else if (strstr(key, "location") ||
              strstr(key, "coordinates")) {
      if (ctx->updating) {
         ctx->newLocationPending = TRUE;
      } else {
         if (ctx->info) {
            weather_info_free(ctx->info);
            ctx->info = NULL;
         }
      }
   }
}

static gboolean
gweather_home_on_mouse_press(GtkWidget *widget,
                             GdkEvent *event,
                             MWeatherHomeCtx *ctx)
{
    osso_application_top(ctx->ctxt, "mweather", NULL);

   return TRUE;
}

void *
hildon_home_applet_lib_initialize (void *state_data, 
				   int *state_size,
				   GtkWidget **applet_return)
{
   GConfClient *gclient;
   GtkIconInfo *iconInfo;
   WeatherLocation *location;

   MWeatherHomeCtx *ctx = g_new0(MWeatherHomeCtx, 1);

   ctx->frame = gtk_frame_new(NULL);
   gtk_widget_set_size_request(GTK_WIDGET (ctx->frame), APPLET_X_SIZE, APPLET_Y_SIZE);
   gtk_widget_show(ctx->frame);
   *applet_return = ctx->frame;
//   gtk_frame_set_shadow_type(GTK_FRAME(ctx->frame), GTK_SHADOW_ETCHED_IN);
   gtk_widget_set_name(GTK_WIDGET(ctx->frame), "osso-rss-feed-reader");

   ctx->ev = gtk_event_box_new();
   gtk_widget_show(ctx->ev);
   gtk_container_add(GTK_CONTAINER(ctx->frame), ctx->ev);
   gtk_container_set_border_width(GTK_CONTAINER(ctx->ev), 6);
   g_signal_connect (ctx->ev, "button-press-event",
                     G_CALLBACK(gweather_home_on_mouse_press), ctx);

   ctx->vbox = gtk_vbox_new(FALSE, 6);
   gtk_widget_show(ctx->vbox);
   gtk_container_add(GTK_CONTAINER(ctx->ev), ctx->vbox);

   ctx->hbox = gtk_hbox_new(FALSE, 6);
   gtk_widget_show(ctx->hbox);
   gtk_box_pack_start(GTK_BOX(ctx->vbox), ctx->hbox, FALSE, TRUE, 0);

   ctx->icon = gtk_image_new_from_icon_name("stock_weather-storm",
                                            GTK_ICON_SIZE_DIALOG);
   gtk_widget_show(ctx->icon);
   gtk_box_pack_start(GTK_BOX(ctx->hbox), ctx->icon, FALSE, TRUE, 0);

   ctx->temp = gtk_label_new("XXX");
   gtk_widget_show(ctx->temp);
   gtk_box_pack_start(GTK_BOX(ctx->hbox), ctx->temp, TRUE, TRUE, 0);
   gtk_misc_set_alignment(GTK_MISC(ctx->temp), 0, 0.5);

   ctx->location = gtk_label_new("XXX");
   gtk_widget_show(ctx->location);
   gtk_box_pack_start(GTK_BOX(ctx->vbox), ctx->location, FALSE, TRUE, 0);
   gtk_misc_set_alignment(GTK_MISC(ctx->location), 0, 0.5);
   gtk_label_set_line_wrap(GTK_LABEL(ctx->location), TRUE);

   ctx->aspect = gtk_aspect_frame_new(NULL, 0.5, 0.5, 1.0, FALSE);
   gtk_widget_show(ctx->aspect);
   gtk_box_pack_start(GTK_BOX(ctx->hbox), ctx->aspect, FALSE, FALSE, 0);
   gtk_frame_set_shadow_type(GTK_FRAME(ctx->aspect), GTK_SHADOW_NONE);

   ctx->alignment = gtk_alignment_new(0.5, 0.5, 1, 1);
   gtk_widget_show(ctx->alignment);
   gtk_container_add(GTK_CONTAINER(ctx->aspect), ctx->alignment);

   ctx->button = gtk_button_new();
   gtk_widget_show(ctx->button);
   gtk_container_add(GTK_CONTAINER(ctx->alignment), ctx->button);
   g_signal_connect_swapped (ctx->button, "clicked",
                             G_CALLBACK(get_weather_info), ctx);

   iconInfo = gtk_icon_theme_lookup_icon(gtk_icon_theme_get_default(),
                                         "qgn_toolb_gene_refresh",
                                         26, 0);

   ctx->refresh = gtk_image_new_from_file(gtk_icon_info_get_filename(iconInfo));
   gtk_widget_show(ctx->refresh);
   gtk_container_add(GTK_CONTAINER(ctx->button), ctx->refresh);

   ctx->ctxt = osso_initialize ("mweather_home", PACKAGE_VERSION, FALSE, NULL);

   ctx->gconf = gweather_gconf_new("/apps/gweather/0");
   gclient = gweather_gconf_get_client(ctx->gconf);

   gweather_prefs_load(&ctx->prefs, ctx->gconf);

   location = gweather_gconf_get_location(ctx->gconf);
   gtk_label_set_label(GTK_LABEL(ctx->location), location->name);

   if (state_data && state_size) {
      gtk_label_set_label(GTK_LABEL(ctx->temp), (gchar *)state_data);
   } else {
      get_weather_info(ctx);
   }

   g_signal_connect(gclient, "value-changed",
                    G_CALLBACK(on_gconf_value_changed), ctx);
   gconf_client_add_dir(gclient, "/apps/gweather/prefs",
                        GCONF_CLIENT_PRELOAD_NONE, NULL);

   if (ctx->prefs.update_enabled) {
      ctx->updateTimer =
         g_timeout_add(ctx->prefs.update_interval * 1000,
                       (GSourceFunc)get_weather_info, ctx);
   }

   weather_location_free(location);

   fprintf(stderr, "mweather initialize %p %d\n", state_data, *state_size);
   return ctx;
}

int
hildon_home_applet_lib_save_state (MWeatherHomeCtx *ctx,
				   void **state_data, 
				   int *state_size)
{
  fprintf (stderr, "mweather save_state\n");

  const char *temp = gtk_label_get_label(GTK_LABEL(ctx->temp));
  *state_data = g_strdup(temp);
  *state_size = strlen(temp) + 1;
  return 0;
}

void
hildon_home_applet_lib_background (MWeatherHomeCtx *ctx)
{
   if (ctx->updateTimer) {
      g_source_remove(ctx->updateTimer);
   }

   fprintf(stderr, "mweather background\n");
}

void
hildon_home_applet_lib_foreground (MWeatherHomeCtx *ctx)
{
   if (ctx->prefs.update_enabled) {
      ctx->updateTimer =
         g_timeout_add(ctx->prefs.update_interval * 1000,
                       (GSourceFunc)get_weather_info, ctx);
   }

   fprintf (stderr, "mweather foreground\n");
}

GtkWidget *
hildon_home_applet_lib_settings (MWeatherHomeCtx *ctx, GtkWindow *parent)
{
  fprintf (stderr, "mweather settings\n");
  return NULL;
}

void
hildon_home_applet_lib_deinitialize (MWeatherHomeCtx *ctx)
{
   osso_deinitialize(ctx->ctxt);
   weather_info_free(ctx->info);
   gweather_gconf_free(ctx->gconf);

   if (ctx->updateTimer) {
      g_source_remove(ctx->updateTimer);
   }

   g_free(ctx);

   fprintf(stderr, "mweather-home deinitialize\n");
}

