/* 
 * Copyright (c) 2006, Jakub Pavelek <jpavelek@welho.com>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *        documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "pidinfo.h"

#define PDIR "/proc/"
#define BUFFSIZE 128

static PidFields fields[] = {	{ PI_PID, 	"Pid:" },
								{ PI_PPID,	"PPid" },
								{ PI_NAME, 	"Name:" },
								{ PI_VMRSS, "VmRSS:" },
								{ PI_VMDATA,"VmData:" },
								{ PI_VMLIB,	"VmLib:" },
								{ 0, NULL } };
								
static char str[BUFFSIZE];

static GDir*
open_pdir (void)
{
	GError *error = NULL;
	GDir *dir = NULL;

	dir = g_dir_open (PDIR, 0, &error);
	if (error) {
		fprintf (stderr, "ERR: Unable to open directory: %s\n", error->message);
		g_error_free (error);
		return NULL;
	}

	return dir;
}


static gchar*
read_cmd_line (guint pid)
{
	gchar *full_path;
	gchar *contents = NULL;
	gsize lenght;
	gchar *tmpstr;
	gchar *lastslash;
	GError *error = NULL;
	
	tmpstr = g_strdup_printf ("%d", pid);
	full_path = g_strconcat (PDIR, tmpstr, "/cmdline", NULL);
	
	if (!g_file_get_contents (full_path, &contents, &lenght, &error)) {
		fprintf (stderr, "ERR: can't read file %s: %s\n", full_path, error->message); 
		g_error_free (error);
		return NULL;
	}
	
	g_free (tmpstr);
	g_free (full_path);
	
	tmpstr = contents + strlen(contents);
	lastslash = NULL;
	while (tmpstr != contents) {
		if (tmpstr[0] == '/') {
			lastslash = tmpstr;
			break;
		}
		tmpstr = tmpstr - 1;
	}
	
	if (lastslash) {
		tmpstr = g_strdup (lastslash + 1);
		g_free (contents);
		return tmpstr;
	} else
		return contents;
}

guint
getticksforpid (gint pid)
{
	gchar *statpath, *contents;
	gsize lenght;
	GError *error = NULL;
	gchar **splits;
	gint ticks;
	
	/* Now read also the load info */
	statpath = g_strdup_printf ("/proc/%d/stat", pid);
	if (!g_file_get_contents (statpath, &contents, &lenght, &error)) {
		fprintf (stderr, "ERR: can't read file %s: %s\n", statpath, error->message);
		g_error_free (error);
		return -1;
	}
	
	splits = g_strsplit_set (contents, " ",  -1);
    sscanf (splits[14], "%d", &ticks);
    
	g_strfreev (splits);
	g_free (contents);
	g_free (statpath);
	
	return ticks;
}



static PidInfo*
read_process_info (const gchar* path)
{
	PidInfo *pi = NULL;
	gchar *contents = NULL;
	gchar *found;
	gsize lenght;
	GError *error = NULL;
	gint index;

	if (!g_file_get_contents (path, &contents, &lenght, &error)) {
		fprintf (stderr, "ERR: can't read file %s: %s\n", path, error->message);
		g_error_free (error);
		return NULL;
	}

	pi = g_new0 (PidInfo, 1);

	index = 0;
	while (fields[index].needle != NULL) {
		found = g_strstr_len (contents, -1, fields[index].needle);
		if (found) {
			switch (fields[index].index) {
			case PI_PID:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->pid);
				break;
			case PI_PPID:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->ppid);
				break;
			case PI_UID:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->uid);
				break;
			case PI_NAME:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%s\n", str); /* TODO: limit the read size! */
				if ((g_strstr_len (str, BUFFSIZE, "maemo-launcher") != 0) ||
					(strlen (str) == 15) ) 
				{ 
					pi->name = read_cmd_line (pi->pid);
				} else {
					pi->name = g_strdup (str);
				}
				break;
			case PI_VMRSS:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->vmrss);
				break;
			case PI_VMDATA:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->vmdata);
				break;
			case PI_VMLIB:
				sscanf (found + 1 + g_utf8_strlen(fields[index].needle, -1), "%d", &pi->vmlib);
				break;
			default:
				fprintf(stderr, "ERR: Search field not found\n");
				break;
			}
		}
		index++;
	}
	
	g_free (contents);
	
	pi->lastload = getticksforpid (pi->pid);
	pi->load = NULL; /* This gets calculated later */
	
	return pi;
}


/*
 * API functions
 */

GSList*
pidinfo_get_processes (int cut_from)
{
	GSList *list = NULL;
	GDir *pdir = NULL;
	const char *path;
	gchar *full_path;
	
	pdir = open_pdir ();
	if (!pdir)
		return NULL;

	while ((path = g_dir_read_name (pdir)) != NULL) {
		if ( g_ascii_isdigit(path[0]) == FALSE)
			continue;	/* Skip non-process files in the directory */
		full_path = g_strconcat (PDIR, path, "/status", NULL);
		list = g_slist_append (list, (gpointer) read_process_info (full_path));
		g_free (full_path);
	}

	g_dir_close (pdir);
	
	return list;
}




gint
pidinfo_sort_by_size (gconstpointer a, gconstpointer b)
{
	PidInfo *pa = (PidInfo*)a;
	PidInfo *pb = (PidInfo*)b;

	if (pa->vmrss > pb->vmrss)
		return -1;
	else if (pa->vmrss < pb->vmrss)
		return 1;
	else
		return 0;
}




const PidInfo*
pidinfo_get_by_pid (const GSList *list, gint pid)
{
	const GSList *iter = list;
	
	while (iter) {
		if (((PidInfo*)iter->data)->pid == pid) {
			return (PidInfo*)iter->data;
		}
		iter = iter->next;
	}
	
	return NULL;
}














/* Temporary code for process tree printouts */


void
pidinfo_print_children (GSList *list, guint pid, guint level)
{
	GSList *iter = list;
	PidInfo *cpi;
	int i;
	
	while (iter) {
		cpi = (PidInfo*)iter->data;
		if (!cpi)
			return;
		if (cpi->ppid == pid) {
			for (i=0;i<level;i++) fprintf(stderr, "\t");
			fprintf (stderr, "|- %s (%d)\n", cpi->name, cpi->pid);
			pidinfo_print_children (list, cpi->pid, level + 1);
		}
		iter = iter->next;
	}		
}

void
pidinfo_print_tree (gint pid, GSList *list)
{
	const PidInfo *pi;
	
	pi = pidinfo_get_by_pid ( list, pid);
	
	if (!pi)
		return;
	
	pidinfo_print_children (list, pid, 1);
}
