/* 
 * Copyright (c) 2006, Jakub Pavelek <jpavelek@welho.com>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *        documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <signal.h>
#include <glib.h>
#include "pid_ui.h"

#define LOAD_SHOW_THRESHOLD 1 /* Do not bother about processes smaller than 10kB */

enum {
    LOAD_COLUMN_PID,
	LOAD_COLUMN_PNAME,
	LOAD_COLUMN_UID,
	LOAD_COLUMN_SIZE,
	LOAD_COLUMN_LOAD,
	LOAD_COLUMN_LASTLOAD,
	LOAD_COLUMN_NCOLUMNS
};

static gint selected_pid;
static GtkTreeIter iter;
static GSList *list;
extern gint deltaU;

extern int kill (__pid_t __pid, int __sig); /* Dunno how to hack around the 'broken' include file */


static gchar*
load_int_to_size (gint mem)
{
	gchar* size;
	
	if (mem >= 1024)
	{
		size = g_strdup_printf ("%d MB", mem/1024);
	} else {
		size = g_strdup_printf ("%d kB", mem);
	}
	return size;
}



static void
load_populate_iter (gpointer data, gpointer user_data)
{
	GtkListStore *store = (GtkListStore*)user_data;
	PidInfo *d = (PidInfo*)data;
	GtkTreeIter iter;

	if (d->vmrss > LOAD_SHOW_THRESHOLD) {
		gtk_list_store_append (store, &iter);

		gtk_list_store_set (store, &iter, 
			LOAD_COLUMN_PNAME, d->name, 
			LOAD_COLUMN_PID, d->pid,
			LOAD_COLUMN_UID, d->uid,
			LOAD_COLUMN_SIZE, load_int_to_size (d->vmrss), 
			LOAD_COLUMN_LOAD, d->load, -1);
	}
}


static void
load_free_pidinfos (gpointer data, gpointer user_data)
{
	PidInfo *d = (PidInfo*)data;
	
	g_free (d->name);
}




GtkListStore*
populate_store (void)
{
	GtkListStore *store;
	
	if (list) {
		g_slist_foreach (list, load_free_pidinfos, NULL);
		g_slist_free (list);
		list = NULL;
	}

	list = pidinfo_get_processes (-1);
	if (!list)
		return NULL;

	list = g_slist_sort (list, pidinfo_sort_by_size);
	store = gtk_list_store_new (LOAD_COLUMN_NCOLUMNS, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);
	g_slist_foreach (list, load_populate_iter, store);

	return store;
}





static void
pid_tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel*)data;

	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, LOAD_COLUMN_PID, &selected_pid, -1);
/* FIXME: remove this later */
pidinfo_print_tree (selected_pid, list);
	}
}


static gboolean
add_process_watch (GSList *list)
{
	gboolean retval = TRUE;
/* FIXME: remove */
fprintf (stderr, "\tAdding a watch for process %d\n", selected_pid);
	return retval;
}

static gboolean
kill_process (GSList *list)
{
	/* Should do some filtering too - do not kill processes that can't or that will cause reboot */
	GtkWidget *dialog;
	GtkWidget *button_yes;
	GtkWidget *button_no;
	GtkWidget *vbox;
	GtkWidget *cb;
	GtkWidget *label;
	gint result;
	gboolean retval;
	const PidInfo *pi;
	gchar *tstr;
	guint signal;
	
	dialog = gtk_dialog_new_with_buttons ("Confirmation",
                                           NULL,
                                           GTK_DIALOG_MODAL,
                                           NULL);
	
	button_yes = gtk_dialog_add_button (GTK_DIALOG(dialog), "Yes", LOAD_DIALOG_YES);
	button_no = gtk_dialog_add_button (GTK_DIALOG(dialog), "No", LOAD_DIALOG_NO);
	
	pi = pidinfo_get_by_pid (list, selected_pid);
	
	vbox = gtk_vbox_new (FALSE, 5);
		
	tstr = g_strdup_printf("Terminate process %s\n", pi->name);
	label = gtk_label_new (tstr);
	g_free (tstr);
	gtk_box_pack_start (GTK_BOX(vbox), label, TRUE, TRUE, 0);
	
	cb = gtk_check_button_new_with_label ("Force quit?");
	gtk_box_pack_start (GTK_BOX(vbox), cb, TRUE, TRUE, 0);
	
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox),
                            vbox,
                            TRUE, 
                            TRUE,
                            0);
							
	gtk_widget_show_all(dialog);
	
	result = gtk_dialog_run (GTK_DIALOG (dialog));
	
	switch (result) {
	case LOAD_DIALOG_YES:
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(cb)) == TRUE) {
			signal = SIGKILL;
		} else {
			signal = SIGTERM;
		}
		kill (pi->pid, signal);
		retval = TRUE;
		break;
	case LOAD_DIALOG_NO:
	default:
		retval = FALSE;
		break;
	}
	gtk_widget_destroy (dialog);
	
	return retval;
}

gboolean
pidui_update_load_detail (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gint pid;
	guint oldticks, newticks;
	gchar* str;
	GtkListStore *store = (GtkListStore*) data;
	
	gtk_tree_model_get (model, iter, LOAD_COLUMN_PID, &pid, -1);
	gtk_tree_model_get (model, iter, LOAD_COLUMN_LASTLOAD, &oldticks, -1);
	newticks = getticksforpid (pid);
	if (deltaU == 0) {
		str = g_strdup ("0 %");
	} else {
		if ((100*(newticks-oldticks)/deltaU) < 100) {
			str = g_strdup_printf ("%d %%", 100*(newticks-oldticks)/deltaU);
		} else {
			return FALSE;
		}
	}
	gtk_list_store_set (GTK_LIST_STORE(store), iter, LOAD_COLUMN_LOAD, str, -1);
	gtk_list_store_set (GTK_LIST_STORE(store), iter, LOAD_COLUMN_LASTLOAD, newticks, -1);
	g_free (str);
	
	return FALSE;
}


static gboolean
pidui_update_load (gpointer data)
{
	GtkListStore *store = (GtkListStore*) data;
	
	gtk_tree_model_foreach (GTK_TREE_MODEL(store), pidui_update_load_detail, store);
	 	
	return TRUE;	
}



void
activate_list_processes_item (GtkMenuItem *item, gpointer data)
{
    GtkWidget *dialog;
    GtkWidget *button_close;
    GtkWidget *button_kill;
    GtkWidget *button_watch;
    gint result;
	GtkListStore *store;
	GtkWidget *tree;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkWidget *sw;
	GtkTreeSelection *select;
	guint timeout_id;
	
    dialog = gtk_dialog_new_with_buttons ("Processes",
                                           NULL,
                                           GTK_DIALOG_MODAL,
                                           NULL);

	button_watch = gtk_dialog_add_button(GTK_DIALOG(dialog), "Watch", LOAD_DIALOG_WATCH);
	button_kill = gtk_dialog_add_button(GTK_DIALOG(dialog), "Kill", LOAD_DIALOG_KILL);
    button_close = gtk_dialog_add_button(GTK_DIALOG(dialog), "Close", LOAD_DIALOG_CLOSE);

	list = NULL;
	store = populate_store ();
	tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree), FALSE);
	gtk_widget_set_size_request (GTK_WIDGET(tree), 600, 200);

	sw = gtk_scrolled_window_new ( NULL, NULL );
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(sw),
									GTK_POLICY_NEVER,
									GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(sw), GTK_WIDGET(tree));

	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), sw, TRUE, TRUE, 0);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("PID",
                                                   renderer,
                                                   "text", LOAD_COLUMN_PID,
                                                   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Process name",
                                                   renderer,
                                                   "text", LOAD_COLUMN_PNAME,
                                                   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Size",
                                                   renderer,
                                                   "text", LOAD_COLUMN_SIZE,
                                                   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);
	
	
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Load",
                                                   renderer,
                                                   "text", LOAD_COLUMN_LOAD,
                                                   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);
	
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree), TRUE);

	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	selected_pid = -1;
	
	g_signal_connect (G_OBJECT (select), "changed",
                  G_CALLBACK (pid_tree_selection_changed_cb),
                  NULL);
	timeout_id = g_timeout_add (2000, pidui_update_load, store);
	
				  
    gtk_widget_show_all(dialog);

run:
    result = gtk_dialog_run (GTK_DIALOG (dialog));
	
    switch (result) {
    case LOAD_DIALOG_CLOSE:
		gtk_widget_destroy (dialog);
        break;
	case LOAD_DIALOG_KILL:
		if (kill_process (list)) { /* Killing succeeded - refresh the list */
			gtk_tree_view_set_model (GTK_TREE_VIEW (tree), NULL);
			gtk_list_store_clear (store);
			g_slist_free (list);
			store = NULL;
			list = NULL;
			/*FIXME: this is non-deterministic! Make sure we wait long enough but not too long */
			g_thread_yield  (); /* Killing done in another thread, lets wait a bit? */
			store = populate_store ();
			gtk_tree_view_set_model (GTK_TREE_VIEW (tree), GTK_TREE_MODEL (store));
		}
		goto run;
	case LOAD_DIALOG_WATCH:
		if (add_process_watch (list)) {
			/* TODO: Implement the watch addition code */
		}
		goto run;
		break;
    default:
		gtk_widget_destroy (dialog);
		break;
	}
	
	g_source_remove (timeout_id);
	
	return;
}
