// Author: Zelea
// Date: 31 July 2007
// License: http://www.gnu.org/licenses/gpl.txt

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>
#include <X11/Xft/Xft.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include "rivers.h"

#define WSIZE		800
#define HSIZE		480

#define BBORD 		5
#define SELECTBORD 	4
#define BBARW 		80

#define TILEW           48
#define TILEH           64

#define SIZEX ( TILEX * TILEW )
#define SIZEY ( TILEY * TILEH )
#define OFFX ( ( WSIZE - SIZEX ) / 2 )
#define OFFY ( ( HSIZE - SIZEY ) / 2 )

Atom          atoms_WINDOW_STATE;
Atom          atoms_WINDOW_STATE_FULLSCREEN;

Display      *dpy;
int           mScreen;
XImage       *mImage;
XftFont      *fontn, *fontb, *fontt, *fontp;
XftDraw      *dxft;
XftColor      cxft, bxft;
u08          *wraster;
u08          *tiles, *hitiles;
int           lastx, lasty;
int           seconds, pairs, left;
int           pair_ent;
int           hix, hiy;
int           game_mode;
int           signal_safe;
int           mWidth, mHeight, mDepth;
unsigned long mBlackPixel;
unsigned long mWhitePixel;
unsigned long color_menu;
unsigned long color_xor_select;
unsigned long color_bkg;
Window        mWin;
GC            mGC;
int           board[TILEX + 2][TILEY + 2];
struct sPair  SP[TILEX * TILEY / 2];
struct sPair  last_pair;
int           hall[FAME + 1];
int           finish;
static XGlyphInfo ext;

enum game_mode_type
{
  MNORMAL, MPAUSE, MHALL, MABOUT
};

enum button_fill
{
  NEXT, FILL
};

enum button_type
{
  BNEW, BABOUT, BHINT, BBACK, BSHUFFLE, BRESET, BAUTO, BPAUSE, BQUIT
};
struct sButton button[] = {
  {"New", NEXT, 0, 0, 0, 0, 0, 0, 0},
  {"About", FILL, 0, 0, 0, 0, 0, 0, 0},
  {"Hint", FILL, 0, 0, 0, 0, 0, 0, 0},
  {"Back", FILL, 0, 0, 0, 0, 0, 0, 0},
  {"Shuffle", NEXT, 0, 0, 0, 0, 0, 0, 0},
  {"Reset", NEXT, 0, 0, 0, 0, 0, 0, 0},
  {"Auto", NEXT, 0, 0, 0, 0, 0, 0, 0},
  {"Pause", FILL, 0, 0, 0, 0, 0, 0, 0},
  {"Quit", FILL, 0, 0, 0, 0, 0, 0, 0},
};
int           nrbuttons = sizeof( button ) / sizeof( struct sButton );

enum stat_format
{
  DECIMAL, CHECKBOX, TIME
};
struct sStatus values[] = {
  {"Time", NEXT, TIME, &seconds, 0},
  {"Best", NEXT, TIME, &best_seconds, 0},
  {"Worst", NEXT, TIME, &worst_seconds, 0},
  {"Pairs", FILL, DECIMAL, &pairs, 0},
  {"Tiles", FILL, DECIMAL, &left, 0},
  {"Sound", FILL, CHECKBOX, &sound, 0},
  {"Cheat", NEXT, CHECKBOX, &cheat, 0},
};
int           nrvalues = sizeof( values ) / sizeof( struct sStatus );

static void   copy_tile( int hi, int nr, int x, int y );

// alarm function called every second to count time
static void
wakeup( int signum )
{
  char          val[20];
  int           lv, xoff;

  signal( SIGALRM, SIG_IGN );
  lv = 0;
  if ( signal_safe )
    lv = sprintf( val, "%d:%02d", seconds / 60, seconds % 60 );
  if ( game_mode == MNORMAL )
    seconds++;
  if ( signal_safe )
  {
    XSetBackground( dpy, mGC, color_menu );
    XSetForeground( dpy, mGC, mBlackPixel );
    XftTextExtents8( dpy, fontn, ( FcChar8 * ) val, lv, &ext );
    xoff = ( BBARW - ext.width ) / 2;
    XftDrawRect( dxft, &bxft, 0, values[0].hval - ext.height, BBARW,
	ext.height + 1 );
    XftDrawString8( dxft, &cxft, fontn, xoff, values[0].hval,
	( FcChar8 * ) val, lv );
    XFlush( dpy );
  }
  alarm( 1 );
  signal( SIGALRM, wakeup );
}

static int
create_GC( Window mNewWindow, GC * mNewGC )
{
  XGCValues     mGCValues;

  mGCValues.fill_style = FillSolid;
  *mNewGC = XCreateGC( dpy, mNewWindow, GCFillStyle, &mGCValues );
  if ( *mNewGC == 0 )
    return 0;
  else
  {
    XSetForeground( dpy, *mNewGC, mWhitePixel );
    XSetBackground( dpy, *mNewGC, mBlackPixel );
    return 1;
  }
}

static        Window
OpenWindow( int x, int y, int width, int height, int flag, GC * mNewGC )
{
  XSetWindowAttributes mWinAttributes;
  unsigned long mWinMask;
  XSizeHints    mSH;
  XWMHints      mWMH;
  Window        mNewWindow;

  // Setting attributes
  mWinAttributes.border_pixel = WhitePixel( dpy, mScreen );
  mWinAttributes.background_pixel = BlackPixel( dpy, mScreen );
  mWinAttributes.override_redirect = False;

  mWinMask = CWBackPixel | CWBorderPixel | CWOverrideRedirect;

  mNewWindow = XCreateWindow( dpy,
      RootWindow( dpy, mScreen ),
      x, y, width, height, 0, mDepth,
      InputOutput, CopyFromParent, mWinMask, &mWinAttributes );

#if( FULL_SCREEN )
  // Set the wmhints needed for fullscreen 
  XChangeProperty( dpy, mNewWindow, atoms_WINDOW_STATE,
      XA_ATOM, 32, PropModeReplace,
      ( unsigned char * ) &atoms_WINDOW_STATE_FULLSCREEN, 1 );
#endif

  mWMH.initial_state = NormalState;
  mWMH.flags = StateHint;

  XSetWMHints( dpy, mNewWindow, &mWMH );

  // fixed unresizebale window
  mSH.flags = PSize | PMinSize | PMaxSize;	// | PPosition
//  mSH.x = x;
//  mSH.y = y;
  mSH.width = mSH.min_width = mSH.max_width = width;
  mSH.height = mSH.min_height = mSH.max_height = height;

  XSetNormalHints( dpy, mNewWindow, &mSH );

  if ( create_GC( mNewWindow, mNewGC ) == 0 )
  {
    XDestroyWindow( dpy, mNewWindow );
    return ( ( Window ) 0 );
  }

  XMapWindow( dpy, mNewWindow );
  XFlush( dpy );

  return mNewWindow;
}

static void
display_button( int b )
{
  int           h;

  // button background
  XSetBackground( dpy, mGC, color_menu );
  XSetForeground( dpy, mGC, color_menu );
  XFillRectangle( dpy, mWin, mGC,
      button[b].x1, button[b].y1,
      button[b].x2 - button[b].x1, button[b].y2 - button[b].y1 );
  // button text
  h = fontb->ascent + fontb->descent;
  XftDrawString8( dxft, &cxft, fontb, button[b].txtx,
      button[b].y1 + BBORD / 2 + h, ( FcChar8 * ) button[b].name,
      strlen( button[b].name ) );
  // button outilnes
  XSetForeground( dpy, mGC, mBlackPixel );
  XDrawRectangle( dpy, mWin, mGC,
      button[b].x1, button[b].y1,
      button[b].x2 - button[b].x1, button[b].y2 - button[b].y1 );
  XSetForeground( dpy, mGC, mWhitePixel );
  if ( button[b].pressed )
  {
    XDrawLine( dpy, mWin, mGC,
	button[b].x1, button[b].y2, button[b].x2, button[b].y2 );
    XDrawLine( dpy, mWin, mGC,
	button[b].x2, button[b].y1, button[b].x2, button[b].y2 );
  }
  else
  {
    XDrawLine( dpy, mWin, mGC,
	button[b].x1, button[b].y1, button[b].x2, button[b].y1 );
    XDrawLine( dpy, mWin, mGC,
	button[b].x1, button[b].y1, button[b].x1, button[b].y2 );
  }
}

static void
refresh_status( int x, int y )
{
  char          val[20];
  int           fills, fillspace, b, l, lv;
  int           h, m, s, xoff, yoff;

  for ( fills = b = 0; b < nrvalues; b++ )
    if ( values[b].fill == FILL )
      fills++;
  h = fontb->ascent + fontb->descent;
  values[0].hval = y + 3 * BBORD + 2 * h;
  fillspace =
      ( HSIZE - y - 3 * BBORD - nrvalues * 2 * ( 2 * BBORD + h ) ) / fills;
  for ( b = 1; b < nrvalues; b++ )
    values[b].hval = values[b - 1].hval + 2 * ( 2 * BBORD + h )
	+ values[b].fill * fillspace;
  for ( b = 0; b < nrvalues; b++ )
  {
    l = strlen( values[b].field );
    XftTextExtents8( dpy, fontb, ( FcChar8 * ) values[b].field, l, &ext );
    xoff = x + ( BBARW - ext.width ) / 2;
    XftDrawRect( dxft, &bxft, 0, values[b].hval - h - ext.height, BBARW,
	ext.height + 1 );
    XftDrawString8( dxft, &cxft, fontb, xoff, values[b].hval - h,
	( FcChar8 * ) values[b].field, l );
  }
  h = fontn->ascent + fontn->descent;
  for ( b = 0; b < nrvalues; b++ )
  {
    if ( values[b].format == CHECKBOX )
    {
      xoff = x + ( BBARW - 3 * BBORD ) / 2;
      yoff = y + values[b].hval;
      XSetForeground( dpy, mGC, color_menu );
      XFillRectangle( dpy, mWin, mGC, xoff, yoff - 3 * BBORD,
	  3 * BBORD, 3 * BBORD );
      XSetForeground( dpy, mGC, mBlackPixel );
      XDrawRectangle( dpy, mWin, mGC, xoff, yoff - 3 * BBORD,
	  3 * BBORD, 3 * BBORD );
      if ( *values[b].val )
      {
	XSetLineAttributes( dpy, mGC, 3, FillSolid, CapRound, JoinMiter );
	XDrawLine( dpy, mWin, mGC, xoff + BBORD, yoff - 2 * BBORD,
	    xoff + 2 * BBORD, yoff - BBORD );
	XDrawLine( dpy, mWin, mGC, xoff + BBORD, yoff - BBORD,
	    xoff + 2 * BBORD, yoff - 2 * BBORD );
	XSetLineAttributes( dpy, mGC, 1, FillSolid, CapButt, JoinMiter );
      }
      continue;
    }
    if ( values[b].format == TIME )
    {
      m = *( values[b].val ) / 60;
      s = *( values[b].val ) % 60;
      lv = sprintf( val, "%d:%02d", m, s );
    }
    else
      lv = sprintf( val, "%d", *( values[b].val ) );
    XftTextExtents8( dpy, fontn, ( FcChar8 * ) val, lv, &ext );
    xoff = x + ( BBARW - ext.width ) / 2;
    XftDrawRect( dxft, &bxft, 0, values[b].hval - ext.height, BBARW,
	ext.height + 1 );
    XftDrawString8( dxft, &cxft, fontn, xoff, values[b].hval,
	( FcChar8 * ) val, lv );
  }
}

// calculate button sizes and layout 
// set the active area of each button in the button structure
// so that a mouse click event can compare with this coordinates
// and determine if a button is pressed or released
static void
init_buttons( int x, int y )
{
  int           fills, fillspace, b;
  int           w, h, maxw;

  for ( fills = maxw = b = 0; b < nrbuttons; b++ )
  {
    XftTextExtents8( dpy, fontb, ( FcChar8 * ) button[b].name,
	strlen( button[b].name ), &ext );
    w = ext.width;
    if ( w > maxw )
      maxw = w;
    button[b].txtx = WSIZE - ( BBARW + w ) / 2;
    if ( button[b].fill == FILL )
      fills++;
  }
  h = fontb->ascent + fontb->descent;
  fillspace = ( HSIZE - y - nrbuttons * ( h + 3 * BBORD ) - BBORD ) / fills;
  button[0].y1 = y + BBORD;
  for ( b = 1; b < nrbuttons; b++ )
    button[b].y1 =
	button[b - 1].y1 + h + 3 * BBORD + button[b].fill * fillspace;
  for ( b = 0; b < nrbuttons; b++ )
  {
    button[b].y2 = button[b].y1 + h + 2 * BBORD;
    button[b].x1 = button[b].x2 = x + ( BBARW - maxw ) / 2 - BBORD;
    button[b].x2 += maxw + 2 * BBORD;
    if ( button[b].on )
      display_button( b );
  }
}

static void
status_bar( int x, int y, int w, int h, unsigned long bbck )
{
  XSetBackground( dpy, mGC, bbck );
  XSetForeground( dpy, mGC, bbck );
  XFillRectangle( dpy, mWin, mGC, x, y, w, h );
  refresh_status( x, y );
}

static void
button_bar( int x, int y, int w, int h, unsigned long bbck )
{
  XSetBackground( dpy, mGC, bbck );
  XSetForeground( dpy, mGC, bbck );
  XFillRectangle( dpy, mWin, mGC, x, y, w, h );
  init_buttons( x, y );
}

static void
refresh_buttons( void )
{
  if ( pairs )
    button[BHINT].on = 1;
  if ( last_pair.tile >= 0 )
    button[BBACK].on = 1;
  button[BRESET].on = ( game_mode == MHALL );
  button[BAUTO].on = cheat;
  button[BABOUT].on = 1;
  button[BPAUSE].on = 1;
  button[BSHUFFLE].on = 1;
  button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
}

static void
update_window( int mode )
{
  int           i, j;

  if ( mode )
  {
    XSetForeground( dpy, mGC, color_bkg );
    XFillRectangle( dpy, mWin, mGC, BBARW, 0, WSIZE - 2 * BBARW, HSIZE );
    button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
    status_bar( 0, 0, BBARW, HSIZE, color_menu );
  }
  for ( j = 1; j <= TILEY; j++ )
    for ( i = 1; i <= TILEX; i++ )
      copy_tile( 0, board[i][j], i, j );
  XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
  XFlush( dpy );
}

static void
clear_border( void )
{
  XSetForeground( dpy, mGC, color_bkg );
  XFillRectangle( dpy, mWin, mGC, BBARW, 0, WSIZE - 2 * BBARW, OFFY );
  XFillRectangle( dpy, mWin, mGC, BBARW, HSIZE - OFFY, WSIZE - 2 * BBARW,
      OFFY );
  XFillRectangle( dpy, mWin, mGC, BBARW, OFFY, OFFX - BBARW,
      HSIZE - 2 * OFFY );
  XFillRectangle( dpy, mWin, mGC, WSIZE - OFFX, OFFY, OFFX - BBARW,
      HSIZE - 2 * OFFY );
}

// draw a segment part of the hint path and
// chop it if it's exiting the screen
static void
draw_line( int chop, int x1, int y1, int x2, int y2 )
{
  x1 = OFFX + ( x1 - 1 ) * TILEW + TILEW / 2;
  y1 = OFFY + ( y1 - 1 ) * TILEH + TILEH / 2;
  x2 = OFFX + ( x2 - 1 ) * TILEW + TILEW / 2;
  y2 = OFFY + ( y2 - 1 ) * TILEH + TILEH / 2;
  if ( y1 < 0 )
    y1 = BBORD;
  if ( y2 < 0 )
    y2 = BBORD;
  if ( y1 > HSIZE )
    y1 = HSIZE - BBORD;
  if ( y2 > HSIZE )
    y2 = HSIZE - BBORD;
  if ( chop )
  {
    if ( x1 == x2 )
    {
      if ( y1 < y2 )
	y1 += TILEH / 2;
      if ( y1 > y2 )
	y1 -= TILEH / 2;
    }
    if ( y1 == y2 )
    {
      if ( x1 < x2 )
	x1 += TILEW / 2;
      if ( x1 > x2 )
	x1 -= TILEW / 2;
    }
  }
  if ( chop == 2 )
  {
    if ( x1 == x2 )
    {
      if ( y1 < y2 )
	y2 -= TILEH / 2;
      if ( y1 > y2 )
	y2 += TILEH / 2;
    }
    if ( y1 == y2 )
    {
      if ( x1 < x2 )
	x2 -= TILEW / 2;
      if ( x1 > x2 )
	x2 += TILEW / 2;
    }
  }
  XDrawLine( dpy, mWin, mGC, x1, y1, x2, y2 );
}

// show the hint path between two tiles
static void
show_path( struct sPair *P )
{
  int           p1, p2;

  XSetForeground( dpy, mGC, mBlackPixel );
  // if the path is a straight line draw it and return
  if ( P->xx == 0 && P->yx == 0 )
  {
    draw_line( 2, P->x1, P->y1, P->x2, P->y2 );
    return;
  }
  p1 = p2 = 0;
  // draw segment between the corner and tile
  // which shares one coordinate with the corner
  if ( P->x1 == P->xx || P->y1 == P->yx )
  {
    draw_line( 1, P->x1, P->y1, P->xx, P->yx );
    p1 = 1;
  }
  if ( P->x2 == P->xx || P->y2 == P->yx )
  {
    draw_line( 1, P->x2, P->y2, P->xx, P->yx );
    p2 = 1;
  }
  // the path has only one corner, return
  if ( p1 && p2 )
    return;
  // draw the remaining two segments
  // between the corner and second tile
  if ( p1 )
  {
    if ( P->x1 == P->xx )
    {
      draw_line( 1, P->x2, P->y2, P->x2, P->yx );
      draw_line( 0, P->xx, P->yx, P->x2, P->yx );
    }
    if ( P->y1 == P->yx )
    {
      draw_line( 1, P->x2, P->y2, P->xx, P->y2 );
      draw_line( 0, P->xx, P->yx, P->xx, P->y2 );
    }
  }
  if ( p2 )
  {
    if ( P->x2 == P->xx )
    {
      draw_line( 1, P->x1, P->y1, P->x1, P->yx );
      draw_line( 0, P->xx, P->yx, P->x1, P->yx );
    }
    if ( P->y2 == P->yx )
    {
      draw_line( 1, P->x1, P->y1, P->xx, P->y1 );
      draw_line( 0, P->xx, P->yx, P->xx, P->y1 );
    }
  }
}

static void
show_hint( void )
{
  struct sPair *P;
  char         *hip = "30 seconds penalty for using hint";
  int           h;

  if ( pair_ent == pairs )
  {
    XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
    return;
  }
  P = &SP[pair_ent++];
  copy_tile( 1, board[P->x1][P->y1], P->x1, P->y1 );
  copy_tile( 1, board[P->x2][P->y2], P->x2, P->y2 );
  XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
  show_path( P );
  if ( !cheat )
  {
    seconds += 30;		// 30 seconds penalty
    XftTextExtents8( dpy, fontn, ( FcChar8 * ) hip, strlen( hip ), &ext );
    if ( P->xx == 0 || P->yx == 0 )
      h = HSIZE + ext.y - ext.height;
    else
      h = ext.y;
    XftDrawString8( dxft, &cxft, fontn, ( WSIZE - ext.width ) / 2, h,
	( FcChar8 * ) hip, strlen( hip ) );
  }
  refresh_status( 0, 0 );
  XFlush( dpy );
}

static void
restore_last( void )
{
  copy_tile( 0, last_pair.tile, last_pair.x1, last_pair.y1 );
  copy_tile( 0, last_pair.tile, last_pair.x2, last_pair.y2 );
  XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
  board[last_pair.x1][last_pair.y1] = board[last_pair.x2][last_pair.y2] =
      last_pair.tile;
  last_pair.tile = -1;
  left += 2;
  find_pairs(  );
  refresh_status( 0, 0 );
  button[BBACK].on = 0;
  button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
  XFlush( dpy );
}

static void
hall_of_fame( void )
{
  char         *title = " Hall of Fame ";
  char          msg[20];
  int           i, w, h, xr, yr, sp;

  XSetForeground( dpy, mGC, color_bkg );
  XFillRectangle( dpy, mWin, mGC, OFFX, OFFY, SIZEX, SIZEY );
  xr = OFFX + BBORD / 2;
  yr = OFFY + BBORD / 2;
  w = SIZEX - BBORD;
  h = SIZEY - BBORD;
  XSetForeground( dpy, mGC, color_menu );
  XSetLineAttributes( dpy, mGC, BBORD, FillSolid, CapButt, JoinMiter );
  XDrawRectangle( dpy, mWin, mGC, xr, yr, w, h );
  XSetLineAttributes( dpy, mGC, 1, FillSolid, CapButt, JoinMiter );
  h = fontt->ascent + fontt->descent;
  sp = SIZEY - h;
  yr = OFFY + h;
  XftTextExtents8( dpy, fontt, ( FcChar8 * ) title, strlen( title ), &ext );
  XftDrawString8( dxft, &cxft, fontt, ( WSIZE - ext.width ) / 2, yr,
      ( FcChar8 * ) title, strlen( title ) );
  h = fontb->ascent + fontb->descent;
  yr += 3 * h;
  sp -= 3 * h;
  sp /= FAME / 2;
  for ( i = 0; i < FAME; i++ )
  {
    if ( hall[i] == MAXTIME )
      continue;
    sprintf( msg, " %2d    %d:%02d ", i + 1, hall[i] / 60, hall[i] % 60 );
    XftTextExtents8( dpy, fontb, ( FcChar8 * ) msg, strlen( msg ), &ext );
    xr = OFFX + SIZEX / 4 - ext.width / 2;
    if ( i >= FAME / 2 )
      xr += SIZEX / 2;
    XftDrawString8( dxft, &cxft, fontb, xr, yr + ( i % ( FAME / 2 ) ) * sp,
	( FcChar8 * ) msg, strlen( msg ) );
  }
  button[BRESET].on = 1;
  button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
  XFlush( dpy );
  game_mode = MHALL;
}

static void
show_about( void )
{
  char         *title = "How to Play the Game";
  int           w, h, xr, yr;

  if ( game_mode == MABOUT )
  {
    refresh_buttons(  );
    update_window( 1 );
    game_mode = MNORMAL;
  }
  else
  {
    button[BHINT].on = 0;
    button[BBACK].on = 0;
    button[BAUTO].on = 0;
    button[BPAUSE].on = 0;
    button[BSHUFFLE].on = 0;
    button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
    XSetForeground( dpy, mGC, color_bkg );
    XFillRectangle( dpy, mWin, mGC, OFFX, OFFY, SIZEX, SIZEY );
    xr = OFFX + BBORD / 2;
    yr = OFFY + BBORD / 2;
    w = SIZEX - BBORD;
    h = SIZEY - BBORD;
    XSetForeground( dpy, mGC, color_menu );
    XSetLineAttributes( dpy, mGC, BBORD, FillSolid, CapButt, JoinMiter );
    XDrawRectangle( dpy, mWin, mGC, xr, yr, w, h );
    XSetLineAttributes( dpy, mGC, 1, FillSolid, CapButt, JoinMiter );
    XftTextExtents8( dpy, fontt, ( FcChar8 * ) title, strlen( title ), &ext );
    w = ( WSIZE - ext.width ) / 2;
    h = OFFY + 2 * BBORD;
    XftDrawString8( dxft, &cxft, fontt, w, h + ext.y,
	( FcChar8 * ) title, strlen( title ) );
    h += ext.height;
    XftTextExtents8( dpy, fontb, ( FcChar8 * ) GAME_VERSION,
	strlen( GAME_VERSION ), &ext );
    XftDrawString8( dxft, &cxft, fontb,
	( WSIZE + SIZEX ) / 2 - BBORD - ext.width, HSIZE - OFFY - BBORD,
	( FcChar8 * ) GAME_VERSION, strlen( GAME_VERSION ) );
    justify_about( OFFX + 2 * BBORD, h + BBORD,
	OFFX + SIZEX - 2 * BBORD, HSIZE - OFFY - BBORD - ext.height );
    XFlush( dpy );
    game_mode = MABOUT;
  }
}

static int
sort_ascend( const void *a, const void *b )
{
  return ( *( int * ) a - *( int * ) b );
}

// sort the hall of fame times
void
set_hall_times( void )
{
  int           i;

  qsort( hall, FAME + 1, sizeof( int ), sort_ascend );
  best_seconds = hall[0];
  for ( i = FAME; i >= 0; i-- )
    if ( hall[i] != MAXTIME )
      break;
  if ( i < 0 )
    i = 0;
  worst_seconds = hall[i];
}

static void
game_finished( void )
{
  char         *msg[] = { " Congratulations ",
    " you have finished ", " the board!!! "
  };
  int           i, j, w, maxw, yoff, h, m, xr, yr;

  for ( i = 0; i < nrbuttons; i++ )
    button[i].on = 0;
  button[BNEW].on = 1;
  button[BQUIT].on = 1;
  button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
  game_mode = MPAUSE;
  hall[FAME] = seconds;
  set_hall_times(  );
  refresh_status( 0, 0 );
  XSetBackground( dpy, mGC, color_bkg );
  XSetForeground( dpy, mGC, mBlackPixel );
  h = fontt->ascent + fontt->descent;
  m = sizeof( msg ) / sizeof( char * );
  yoff = ( HSIZE - h * m ) / 2 + fontt->ascent;
  for ( maxw = i = 0; i < m; i++ )
  {
    XftTextExtents8( dpy, fontt, ( FcChar8 * ) msg[i], strlen( msg[i] ),
	&ext );
    w = ext.width;
    if ( w > maxw )
      maxw = w;
    XftDrawString8( dxft, &cxft, fontt, ( WSIZE - w ) / 2, yoff + h * i,
	( FcChar8 * ) msg[i], strlen( msg[i] ) );
  }
  w = maxw, h *= m;
  xr = ( WSIZE - w ) / 2;
  yr = ( HSIZE - h ) / 2;
  for ( j = 0; j < 7; j++ )
  {
    XSetForeground( dpy, mGC, color_menu );
    for ( i = 0; i < BBORD; i++ )
    {
      XDrawRectangle( dpy, mWin, mGC, xr--, yr--, w++, h++ );
      w++;
      h++;
    }
    XSetForeground( dpy, mGC, mBlackPixel );
    for ( i = 0; i < BBORD; i++ )
    {
      XDrawRectangle( dpy, mWin, mGC, xr--, yr--, w++, h++ );
      w++;
      h++;
    }
  }
}

static void
select_tile( int x, int y )
{
  if ( x == lastx && y == lasty )
    return;
  if ( !board[x][y] )
    return;
  hix = hiy = 0;
  if ( pair_ent )
  {
    pair_ent = 0;
    lastx = lasty = 0;
    update_window( 0 );
  }
  if ( check_remove( x, y ) )
  {
    copy_tile( 1, board[x][y], x, y );
    XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
    play_sound( 0 );

    last_pair.tile = board[x][y];
    last_pair.x1 = lastx;
    last_pair.y1 = lasty;
    last_pair.x2 = x;
    last_pair.y2 = y;
    board[lastx][lasty] = 0;
    board[x][y] = 0;
    left -= 2;
    find_pairs(  );
    refresh_status( 0, 0 );
    copy_tile( 0, 0, lastx, lasty );
    copy_tile( 0, 0, x, y );
    lastx = lasty = 0;
    if ( left )
    {
      if ( !button[BBACK].on )
      {
	button[BBACK].on = 1;
	button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
      }
      if ( pairs == left / 2 )
      {
	button[BAUTO].on = 1;
	button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
      }
    }
  }
  else
  {
    if ( lastx && lasty )
    {
      play_sound( 2 );
      copy_tile( 0, board[lastx][lasty], lastx, lasty );
      lastx = lasty = 0;
    }
    else
    {
      play_sound( 1 );
      lastx = x;
      lasty = y;
    }
    if ( lastx && lasty )
      copy_tile( 1, board[x][y], x, y );
  }
  XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
  if ( !left )
    game_finished(  );
  refresh_status( 0, 0 );
  XFlush( dpy );
}

static void
copy_tile( int hi, int nr, int x, int y )
{
  int           i, j, r, c;
  u08          *s, *raster;
  u16          *ss, *ds;
  u32          *dl;
  u32           v;

  nr--;
  x--;
  y--;
  if ( nr >= 0 )
  {
    nr = tiles_select[nr];
    r = nr / TilesCols;
    c = nr % TilesCols;
    if ( hi )
      raster = hitiles;
    else
      raster = tiles;
    if ( mDepth > 16 )
      for ( j = 0; j < TILEH; j++ )
      {
	s = raster + ( ( r * TILEH + j ) * TilesCols + c ) * TILEW * 3;
	dl = ( void * ) wraster + ( ( y * TILEH + j ) * TILEX +
	    x ) * TILEW * 4;
	for ( i = 0; i < TILEW; i++ )
	{
	  v = 0;
	  v |= *s++;
	  v <<= 8;
	  v |= *s++;
	  v <<= 8;
	  v |= *s++;
	  *dl++ = v;
	}
      }
    else
      for ( j = 0; j < TILEH; j++ )
      {
	ss = ( void * ) raster + ( ( r * TILEH + j ) * TilesCols + c ) *
	    TILEW * 2;
	ds = ( void * ) wraster + ( ( y * TILEH + j ) * TILEX + x ) *
	    TILEW * 2;
	for ( i = 0; i < TILEW; i++ )
	  *ds++ = *ss++;
      }
  }
  else
  {
    if ( mDepth > 16 )
      for ( j = 0; j < TILEH; j++ )
      {
	dl = ( void * ) wraster +
	    ( ( y * TILEH + j ) * TILEX + x ) * TILEW * 4;
	for ( i = 0; i < TILEW; i++ )
	  *dl++ = color_bkg;
      }
    else
      for ( j = 0; j < TILEH; j++ )
      {
	ds = ( void * ) wraster +
	    ( ( y * TILEH + j ) * TILEX + x ) * TILEW * 2;
	for ( i = 0; i < TILEW; i++ )
	  *ds++ = color_bkg;
      }
    board[x + 1][y + 1] = 0;
  }
}

static void
motion_rectangle( int x, int y )
{
  XGCValues     V;
  int           i, xr, yr, w, h;

  if ( !board[x][y] )
    return;
  XSetForeground( dpy, mGC, color_xor_select );
  V.function = GXxor;
  XChangeGC( dpy, mGC, GCFunction, &V );
  xr = OFFX + ( x - 1 ) * TILEW;
  yr = OFFY + ( y - 1 ) * TILEH;
  w = TILEW - 1;
  h = TILEH - 1;
  for ( i = 0; i < SELECTBORD; i++ )
  {
    XDrawRectangle( dpy, mWin, mGC, xr++, yr++, w--, h-- );
    w--;
    h--;
  }
  V.function = GXcopy;
  XChangeGC( dpy, mGC, GCFunction, &V );
}

static void
circle_tile( int x, int y )
{
  if ( x == hix && y == hiy )
    return;
  if ( hix || hiy )
    motion_rectangle( hix, hiy );
  if ( !x && !y )
  {
    hix = hiy = 0;
    return;
  }
  motion_rectangle( x, y );
  hix = x;
  hiy = y;
}

static void
enter_pause( void )
{
  char         *msg = " Pause ";
  int           w, h;

  update_window( 1 );
  if ( game_mode == MPAUSE )
  {
    refresh_buttons(  );
    game_mode = MNORMAL;
  }
  else
  {
    button[BHINT].on = 0;
    button[BBACK].on = 0;
    button[BAUTO].on = 0;
    button[BRESET].on = 0;
    button[BSHUFFLE].on = 0;
    button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
    XftTextExtents8( dpy, fontp, ( FcChar8 * ) msg, strlen( msg ), &ext );
    w = ext.width;
    h = fontp->ascent;
    XftDrawString8( dxft, &cxft, fontp, ( WSIZE - w ) / 2, ( HSIZE + h ) / 2,
	( FcChar8 * ) msg, strlen( msg ) );
    game_mode = MPAUSE;
  }
}

static void
enter_automode( void )
{
  struct sPair *P;

  button[BHINT].on = 0;
  button[BABOUT].on = 0;
  button[BBACK].on = 0;
  button[BPAUSE].on = 0;
  button[BSHUFFLE].on = 0;
  button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
  update_window( 0 );
  for ( pair_ent = 0; pair_ent < pairs; pair_ent++ )
  {
    P = &SP[pair_ent];
    copy_tile( 1, board[P->x1][P->y1], P->x1, P->y1 );
    copy_tile( 1, board[P->x2][P->y2], P->x2, P->y2 );
    XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
    left -= 2;
    refresh_status( 0, 0 );
    XFlush( dpy );
    play_sound( 0 );
    copy_tile( 0, 0, P->x1, P->y1 );
    copy_tile( 0, 0, P->x2, P->y2 );
    XPutImage( dpy, mWin, mGC, mImage, 0, 0, OFFX, OFFY, SIZEX, SIZEY );
    XFlush( dpy );
  }
  pair_ent = 0;
  hix = hiy = 0;
  lastx = lasty = 0;
  find_pairs(  );
  refresh_status( 0, 0 );
  refresh_buttons(  );
  if ( !left )
    game_finished(  );
  XFlush( dpy );
}

static int
new_game( void )
{
  int           bon[] = { 1, 1, 1, 0, 1, 0, 0, 1, 1 };
  int           b;

  game_mode = MNORMAL;
  hix = hiy = lastx = lasty = 0;
  pairs = pair_ent = 0;
  last_pair.tile = -1;
  seconds = 0;
  for ( b = 0; b < nrbuttons; b++ )
    button[b].on = bon[b];
  button[BAUTO].on = cheat;
  return new_board(  );
}

static void
shuffle_game( void )
{
  char         *shp = "Board shuffled, 40 seconds penalty";

  shuffle_board(  );
  pair_ent = 0;
  button[BBACK].on = 0;
  update_window( 1 );
  if ( !cheat )
  {
    seconds += 40;		// 40 seconds penalty
    XftTextExtents8( dpy, fontn, ( FcChar8 * ) shp, strlen( shp ), &ext );
    XftDrawString8( dxft, &cxft, fontn, ( WSIZE - ext.width ) / 2, ext.y,
	( FcChar8 * ) shp, strlen( shp ) );
  }
}

static void
reset_scores( void )
{
  int           i;

  for ( i = 0; i < FAME; i++ )
    hall[i] = MAXTIME;
  hall_of_fame(  );
}

// main event loop
static void
doEvents(  )
{
  XEvent        pe;
  XButtonPressedEvent *be = ( XButtonPressedEvent * ) & pe;
  XMotionEvent *me = ( XMotionEvent * ) & pe;
  XKeyEvent    *ke = ( XKeyEvent * ) & pe;
  char          kbuf[10];	// store max 10 events in a queue
  KeySym        keysym;
  XComposeStatus compose;
  int           i, nr, row, col;

  while ( 1 )
  {
    // Wait for X Event
    signal_safe = 1;
    XNextEvent( dpy, &pe );	// this should get first exposure event
    signal_safe = 0;
    switch ( pe.type )
    {
      case Expose:
	update_window( 1 );
	break;

      case KeyPress:
	clear_border(  );
	nr = XLookupString( ke, kbuf, sizeof( kbuf ), &keysym, &compose );
	switch ( keysym )
	{
	  case XK_q:
	    finish = 1;
	    return;
	  case XK_n:
	    if ( button[BNEW].on )
	      new_game(  );
	    return;
	  case XK_p:
	    if ( button[BPAUSE].on )
	      enter_pause(  );
	    break;
	  case XK_r:
	    if ( button[BRESET].on )
	      reset_scores(  );
	    break;
	  case XK_a:
	    if ( button[BAUTO].on )
	      enter_automode(  );
	    break;
	  case XK_s:
	    if ( button[BSHUFFLE].on )
	      shuffle_game(  );
	    break;
	  case XK_b:
	    if ( button[BBACK].on )
	      restore_last(  );
	    break;
	  case XK_h:
	    if ( button[BHINT].on )
	      show_hint(  );
	    break;
	}
	break;

      case ButtonPress:
	switch ( be->button )
	{
	  case Button1:	// left button
	    if ( be->y >= OFFY && be->y < OFFY + SIZEY &&
		be->x >= OFFX && be->x < OFFX + SIZEX )
	    {
	      if ( game_mode == MPAUSE )
	      {
		hall_of_fame(  );
		break;
	      }
	      if ( game_mode == MHALL )
	      {
		if ( left )
		{
		  enter_pause(  );
		  break;
		}
		new_game(  );
		update_window( 1 );
		break;
	      }
	      game_mode = MNORMAL;
	      clear_border(  );
	      refresh_buttons(  );
	      col = ( be->x - OFFX ) / TILEW;
	      row = ( be->y - OFFY ) / TILEH;
	      select_tile( col + 1, row + 1 );
	      break;
	    }
	    for ( i = 0; i < nrbuttons; i++ )
	    {
	      if ( !button[i].on )
		continue;
	      if ( be->x <= button[i].x2 && be->x >= button[i].x1 &&
		  be->y <= button[i].y2 && be->y >= button[i].y1 )
	      {
		button[i].pressed = 1;
		display_button( i );
		XFlush( dpy );
		break;
	      }
	    }
	    for ( i = 0; i < nrvalues; i++ )
	    {
	      if ( values[i].format != CHECKBOX )
		continue;
	      if ( be->x < ( BBARW - 3 * BBORD ) / 2
		  || be->x > ( BBARW + 3 * BBORD ) / 2 )
		continue;
	      if ( be->y < values[i].hval - 3 * BBORD
		  || be->y > values[i].hval )
		continue;
	      // toggle checkbox
	      *values[i].val ^= 1;
	      refresh_status( 0, 0 );
	      button[BAUTO].on = cheat;
	      button_bar( WSIZE - BBARW, 0, BBARW, HSIZE, color_menu );
	      break;
	    }
	}
	break;

      case ButtonRelease:
	clear_border(  );
	switch ( be->button )
	{
	  case Button1:	// left button
	    for ( i = 0; i < nrbuttons; i++ )
	    {
	      if ( !button[i].on )
		continue;
	      if ( be->x <= button[i].x2 && be->x >= button[i].x1 &&
		  be->y <= button[i].y2 && be->y >= button[i].y1 )
	      {
		button[i].pressed = 0;
		display_button( i );
		XFlush( dpy );
		switch ( i )
		{
		  case BNEW:
		    new_game(  );
		    return;
		  case BABOUT:
		    show_about(  );
		    break;
		  case BSHUFFLE:
		    shuffle_game(  );
		    break;
		  case BHINT:
		    show_hint(  );
		    break;
		  case BPAUSE:
		    enter_pause(  );
		    break;
		  case BRESET:
		    reset_scores(  );
		    break;
		  case BBACK:
		    restore_last(  );
		    break;
		  case BAUTO:
		    enter_automode(  );
		    break;
		  case BQUIT:	// Quit button
		    finish = 1;
		    return;
		  default:
		    break;
		}
	      }
	    }
	    break;
	}
	break;

      case MotionNotify:
	// move the square cursor as the mouse moves along
	// so that the tile under the cursor is marked
	if ( game_mode == MNORMAL )
	{
	  if ( me->y >= OFFY && me->y < OFFY + SIZEY &&
	      me->x >= OFFX && me->x < OFFX + SIZEX )
	  {
	    col = ( me->x - OFFX ) / TILEW;
	    row = ( me->y - OFFY ) / TILEH;
	    circle_tile( col + 1, row + 1 );
	  }
	  else
	    circle_tile( 0, 0 );
	}
	XFlush( dpy );
	break;
      default:
	break;
    }
  }
}

// create a X11 (full screen in Nokia mode) window
static int
create_display(  )
{
  XWindowAttributes wa;

  dpy = XOpenDisplay( NULL );

  mScreen = DefaultScreen( dpy );
  mDepth = DefaultDepth( dpy, mScreen );
  if ( mDepth < 16 )
  {
    fprintf( stderr,
	"Sorry you need at least 16bpp to run the game (%dbpp)\n", mDepth );
    return 0;
  }
  XGetWindowAttributes( dpy, RootWindow( dpy, mScreen ), &wa );
  mWidth = wa.width;
  mHeight = wa.height;
  atoms_WINDOW_STATE = XInternAtom( dpy, "_NET_WM_STATE", False );
  atoms_WINDOW_STATE_FULLSCREEN
      = XInternAtom( dpy, "_NET_WM_STATE_FULLSCREEN", False );
  return 1;
}

// allocate fonts, colors and pixmap
static int
init_window( int x, int y, int w, int h )
{
  Colormap      cmap;
  XColor        color, colorrgb;

  mBlackPixel = BlackPixel( dpy, mScreen );
  mWhitePixel = WhitePixel( dpy, mScreen );
  cmap = DefaultColormap( dpy, mScreen );

  mWin = OpenWindow( x, y, w, h, 0, &mGC );

  // select fonts
  fontn = XftFontOpenName( dpy, mScreen, FontNormal );
  fontb = XftFontOpenName( dpy, mScreen, FontBold );
  fontt = XftFontOpenName( dpy, mScreen, FontTitle );
  fontp = XftFontOpenName( dpy, mScreen, FontPause );
  if ( !fontn || !fontb || !fontt || !fontp )
  {
    printf( "Not all fonts matched\n" );
    return 0;
  }
  dxft = XftDrawCreate( dpy, mWin, DefaultVisual( dpy, mScreen ), cmap );

  // select a few colors
  XftColorAllocName( dpy, DefaultVisual( dpy, mScreen ), cmap, ColorFont,
      &cxft );
  XftColorAllocName( dpy, DefaultVisual( dpy, mScreen ), cmap, ColorMenu,
      &bxft );
  color_menu = bxft.pixel;
  if ( XAllocNamedColor( dpy, cmap, ColorSelect, &color, &colorrgb ) )
    color_xor_select = color.pixel;
  if ( XAllocNamedColor( dpy, cmap, ColorBkg, &color, &colorrgb ) )
    color_bkg = color.pixel;

  // link to keyboard and mouse events
  XSelectInput( dpy, mWin, KeyPressMask | ButtonPressMask |
      ButtonReleaseMask | PointerMotionMask | ExposureMask );

  mImage = XCreateImage( dpy, CopyFromParent, mDepth,
      ZPixmap, 0, ( char * ) wraster, SIZEX, SIZEY, mDepth > 16 ? 32 : 16,
      0 );
  XInitImage( mImage );
  return 1;
}

// increase tile brigthness with 30%
// to multiply with 1.3 use only integer multiplication (fast)
static void
precompute_highlight_raster( void )
{
  int           i, l;
  u08          *s, *d;
  u16           r, g, b, rh, gh, bh;
  u16          *ds, *dhs;
  u16           v;

  if ( mDepth > 16 )
  {
    l = TilesCols * TILEW * TilesRows * TILEH * 3;
    s = tiles;
    d = hitiles;
    for ( i = 0; i < l; i++ )
    {
      v = ( *s++ * 85197 ) >> 16;	// * 1.3
      v = v > 0xff ? 0xff : v;
      *d++ = v;
    }
  }
  else
  {
    l = TilesCols * TILEW * TilesRows * TILEH;
    s = tiles;
    ds = ( void * ) tiles;
    dhs = ( void * ) hitiles;
    for ( i = 0; i < l; i++ )
    {
      r = *s++;
      g = *s++;
      b = *s++;
      rh = ( r * 85197 ) >> 16;	// * 1.3
      rh = rh > 0xff ? 0xff : rh;
      gh = ( g * 85197 ) >> 16;
      gh = gh > 0xff ? 0xff : gh;
      bh = ( b * 85197 ) >> 16;
      bh = bh > 0xff ? 0xff : bh;
      v = 0;
      v = ( r & 0xf8 ) << 8 | ( g & 0xf8 ) << 3 | ( b & 0xf8 ) >> 3;
      *ds++ = v;
      v = 0;
      v = ( rh & 0xf8 ) << 8 | ( gh & 0xf8 ) << 3 | ( bh & 0xf8 ) >> 3;
      *dhs++ = v;
    }
  }
}

static void
destroy_window( void )
{
  if ( mImage )
    XDestroyImage( mImage );	// destroy the XImage

  XFreeGC( dpy, mGC );		// free the Graphics Context object
  XDestroyWindow( dpy, mWin );
  XCloseDisplay( dpy );
}

int
main( int argc, char *argv[] )
{
  if ( !create_display(  ) )
    return 0;
  if ( ( wraster = calloc( SIZEX * SIZEY, mDepth > 16 ? 4 : 2 ) ) == NULL )
    return 0;

  read_ini( RIVERS_INI );

  if ( !read_JPEG( &tiles, TilesFile ) )
    return 0;
  if ( ( hitiles = malloc( TilesCols * TILEW * TilesRows * TILEH *
      ( mDepth > 16 ? 3 : 2 ) ) ) == NULL )
    return 0;
  precompute_highlight_raster(  );

  if ( !init_window( 0, 0, WSIZE, HSIZE ) )
    return 0;
  XStoreName( dpy, mWin, GAME_VERSION );

  wakeup( 0 );
  if ( new_game(  ) )
  {
    finish = 0;
    do
    {
      update_window( 1 );
      doEvents(  );
    }
    while ( !finish );
  }
  destroy_window(  );

  write_ini( RIVERS_INI );

//free( wraster ); /* already done by window destruction */
  free( hitiles );
  free( tiles );
  exit( 0 );
}
