// Author: Zelea
// Date: 07 August 2007
// License: http://www.gnu.org/licenses/gpl.txt

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include "rivers.h"

static int    connect_board[TILEX + 2][TILEY + 2];
static int    hint_board[TILEX + 2][TILEY + 2];
static int    xx, yx;

#define BOARDSIZE ( ( TILEY + 2 ) * ( TILEX + 2 ) * sizeof( int ) )

enum directions
{
  LEFT, RIGHT, UP, DOWN
};

// starting from a tile fill the connect board in one direction
// until another tile is present or end of board is reached
static int
fill_dir( int x, int y, int dir, int val )
{
  while ( 1 )
  {
    switch ( dir )
    {
      case LEFT:
	x--;
	if ( x < 0 )
	  return 0;
	break;
      case RIGHT:
	x++;
	if ( x > TILEX + 1 )
	  return 0;
	break;
      case UP:
	y--;
	if ( y < 0 )
	  return 0;
	break;
      case DOWN:
	y++;
	if ( y > TILEY + 1 )
	  return 0;
	break;
    }
    if ( board[x][y] )
      return 0;
    // check if we intersect another path and return if so
    if ( connect_board[x][y] )
    {
      xx = x;
      yx = y;
      return 1;
    }
    connect_board[x][y] = val;
  }
}

static int
horizontal_connect( int x1, int x2 )
{
  int           x, y, t;

  if ( x1 == x2 )
    return 0;
  if ( x2 < x1 )
  {
    t = x1;
    x1 = x2;
    x2 = t;
  }
  for ( y = 0; y <= TILEY + 1; y++ )
  {
    // the path must exist at both ends
    if ( !connect_board[x1][y] || !connect_board[x2][y] )
      continue;
    // the 2 points must belong to separate paths (1 and 2)
    if ( connect_board[x1][y] == connect_board[x2][y] )
      continue;
    for ( x = x1 + 1; x <= x2 - 1; x++ )
      if ( board[x][y] )
	break;
    // there must be no tile obstructing the horizontal line
    if ( board[x][y] )
      continue;
    // remember the intersection point
    xx = x;
    yx = y;
    return 1;
  }
  return 0;
}

static int
vertical_connect( int y1, int y2 )
{
  int           x, y, t;

  if ( y1 == y2 )
    return 0;
  if ( y2 < y1 )
  {
    t = y1;
    y1 = y2;
    y2 = t;
  }
  for ( x = 0; x <= TILEX + 1; x++ )
  {
    // the path must exist at both ends
    if ( !connect_board[x][y1] || !connect_board[x][y2] )
      continue;
    // the 2 points must belong to separate paths (1 and 2)
    if ( connect_board[x][y1] == connect_board[x][y2] )
      continue;
    for ( y = y1 + 1; y <= y2 - 1; y++ )
      if ( board[x][y] )
	break;
    // there must be no tile obstructing the vertical line
    if ( board[x][y] )
      continue;
    // remember the intersection point
    xx = x;
    yx = y;
    return 1;
  }
  return 0;
}

// check if 2 tiles form a pair
static int
connect_pair( int x1, int y1, int x2, int y2 )
{
  // reset intersection
  xx = yx = 0;
  // first the trivial cases of adjacent tiles
  if ( x1 == x2 && abs( y1 - y2 ) == 1 )
    return 1;
  if ( y1 == y2 && abs( x1 - x2 ) == 1 )
    return 1;
  memset( connect_board, 0, BOARDSIZE );
  // fill the connect board in all 4 directions 
  // starting from the first tile
  connect_board[x1][y1] = 1;
  fill_dir( x1, y1, LEFT, 1 );
  fill_dir( x1, y1, RIGHT, 1 );
  fill_dir( x1, y1, UP, 1 );
  fill_dir( x1, y1, DOWN, 1 );
  // return if the second tile is in the path
  if ( connect_board[x2][y2] )
    return 1;
  // fill the connect board in all 4 directions 
  // starting from the second tile
  // and return if we intersect any of the paths
  // created by the first tile
  connect_board[x2][y2] = 2;
  if ( fill_dir( x2, y2, LEFT, 2 ) )
    return 1;
  if ( fill_dir( x2, y2, RIGHT, 2 ) )
    return 1;
  if ( fill_dir( x2, y2, UP, 2 ) )
    return 1;
  if ( fill_dir( x2, y2, DOWN, 2 ) )
    return 1;
  // if there is no intersection check if
  // any of the paths can be connected by
  // either a horizontal or a vertical line
  // (this will form the most complex path
  // with 2 corners)
  if ( horizontal_connect( x1, x2 ) )
    return 1;
  if ( vertical_connect( y1, y2 ) )
    return 1;
  return 0;
}

void
find_pairs( void )
{
  struct sPair *P = NULL;
  int           i, j, x, y;

  // save the board to the temporary hint board
  memcpy( hint_board, board, BOARDSIZE );
  pairs = 0;
  // try all tiles to find pairs
  for ( y = 1; y <= TILEY; y++ )
    for ( x = 1; x <= TILEX; x++ )
      if ( board[x][y] && board[x][y] != -1 )
	for ( j = 1; j <= TILEY; j++ )
	  for ( i = 1; i <= TILEX; i++ )
	    if ( board[i][j] && board[i][j] != -1 )
	      if ( ( board[i][j] == board[x][y] ) && ( i != x || j != y ) )
		if ( connect_pair( x, y, i, j ) )
		{
		  P = &SP[pairs++];
		  P->tile = board[x][y];
		  P->x1 = x;
		  P->y1 = y;
		  P->x2 = i;
		  P->y2 = j;
		  // xx,yx is one corner of the path (if any)
		  P->xx = xx;
		  P->yx = yx;
		  // remove the pair from the board
		  board[x][y] = -1;
		  board[i][j] = -1;
		}
  // restore the board
  memcpy( board, hint_board, BOARDSIZE );
}

void
shuffle_board( void )
{
  int           fd, i, j, x, y, count;
  u08           v;

  // use the pseudo-random device for randomness
  // Warning: this will reduce the code portability
  if ( ( fd = open( "/dev/urandom", O_RDONLY ) ) == -1 )
    return;
  // mark the places where tiles are still present
  // on the connect board
  memset( connect_board, 0, BOARDSIZE );
  for ( y = 1; y <= TILEY; y++ )
    for ( x = 1; x <= TILEX; x++ )
      if ( board[x][y] )
	connect_board[x][y] = 1;
  memcpy( hint_board, board, BOARDSIZE );
  // reset the board
  memset( board, 0, BOARDSIZE );
  count = left;
  for ( y = 1; y <= TILEY; y++ )
    for ( x = 1; x <= TILEX; x++ )
      // if there was a tile at this position
      if ( connect_board[x][y] )
      {
	// get a random tile from the remaining saved tiles
	read( fd, &v, 1 );
	v %= count;
	for ( j = 1; j <= TILEY; j++ )
	  for ( i = 1; i <= TILEX; i++ )
	    if ( hint_board[i][j] )
	      if ( v-- == 0 )
	      {
		// place it on the new board while removing it
		// from the saved (hint) board
		board[x][y] = hint_board[i][j];
		hint_board[i][j] = 0;
		goto next_tile;
	      }
      next_tile:
	count--;
      }
  close( fd );
  find_pairs(  );
}

int
new_board( void )
{
  int           fd, i, j, x, y;
  u08           v;

  if ( ( fd = open( "/dev/urandom", O_RDONLY ) ) == -1 )
    return 0;
  memset( board, 0, BOARDSIZE );
  left = TILEX * TILEY;
  for ( j = 0; j < 4; j++ )
    for ( i = 1; i <= TILEX * TILEY / 4; i++ )
    {
      read( fd, &v, 1 );
      v %= TILEX * TILEY;
      y = v / TILEX + 1;
      x = v % TILEX + 1;
      while ( board[x][y] )
      {
	x++;
	if ( x > TILEX )
	{
	  x = 1;
	  y++;
	  if ( y > TILEY )
	    y = 1;
	}
      }
      board[x][y] = i;
    }
  close( fd );
  find_pairs(  );
  return 1;
}

// check if the previous highlighted tile 
// and the current one form a pair
int
check_remove( int x, int y )
{
  // return 'no pair' if there is no previous tile
  if ( !lastx || !lasty )
    return 0;
  // return 'no pair' if the previous and current tile do not match
  if ( board[x][y] != board[lastx][lasty] )
    return 0;
  return connect_pair( x, y, lastx, lasty );
}
