/*********************************************************************
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <string.h>
#include <sys/stat.h>
#include <openobex/obex.h>


#include "obex_client.h"
#include "obex_app.h"
#include "btxfer.h"

#define TRUE  1
#define FALSE 0



//
// Wait for an obex command to finish.
//
void syncwait(obex_t *handle)
{
	obex_app_context *gt;
	int ret;
	
	gt = OBEX_GetUserData(handle);

	while(!gt->clientdone) {
		//printf("syncwait()\n");
		ret = OBEX_HandleInput(handle, 10);
		if(ret < 0) {
			printf("Error while doing OBEX_HandleInput()\n");
			break;
		}
		if(ret == 0) {
			/* If running cable. We get no link-errors, so cancel on timeout */
			printf("Timeout waiting for data. Aborting\n");
			OBEX_CancelRequest(handle, FALSE);
			break;
		}
	}

	gt->clientdone = FALSE;
}

//
//
//
void client_done(obex_t *handle, obex_object_t *object, int obex_cmd, int obex_rsp)
{
    obex_app_context *gt;
	gt = OBEX_GetUserData(handle);

	switch(obex_cmd)	{
	case OBEX_CMD_CONNECT:
		connect_client_done(handle, object, obex_rsp);
		break;
	case OBEX_CMD_DISCONNECT:
		disconnect_client_done(handle, object, obex_rsp);
		break;
	case OBEX_CMD_PUT:
		put_client_done(handle, object, obex_rsp);
		break;
	case OBEX_CMD_GET:
		/*@todo : we need to send abort here */
        break;
	case OBEX_CMD_SETPATH:
		setpath_client_done(handle, object, obex_rsp);
		break;
	}
	gt->clientdone = TRUE;
}

//
//
//
void connect_client(obex_t *handle)
{
	obex_object_t *object;
	obex_headerdata_t hd;

	if(! (object = OBEX_ObjectNew(handle, OBEX_CMD_CONNECT)))	{
		printf("Error\n");
		return;
	}

	hd.bs = "Linux";
	if(OBEX_ObjectAddHeader(handle, object, OBEX_HDR_WHO, hd, 6,
				OBEX_FL_FIT_ONE_PACKET) < 0)	{
		printf("Error adding header\n");
		OBEX_ObjectDelete(handle, object);
		return;
	}
	OBEX_Request(handle, object);
	syncwait(handle);
}

//
//
//
void connect_client_done(obex_t *handle, obex_object_t *object, int obex_rsp)
{
	uint8_t *nonhdrdata;

	if(obex_rsp == OBEX_RSP_SUCCESS) {
		printf("Connect OK!\n");
		if(OBEX_ObjectGetNonHdrData(object, &nonhdrdata) == 4) {
			printf("Version: 0x%02x. Flags: 0x%02x\n", nonhdrdata[0], nonhdrdata[1]);
		}
	}	
	else {
		printf("Connect failed 0x%02x!\n", obex_rsp);
	}
}


//
//
//
void disconnect_client(obex_t *handle)
{
	obex_object_t *object;

	if(! (object = OBEX_ObjectNew(handle, OBEX_CMD_DISCONNECT)))	{
		printf("Error\n");
		return;
	}

	OBEX_Request(handle, object);
	syncwait(handle);
}

//
//
//
void disconnect_client_done(obex_t *handle, obex_object_t *object, int obex_rsp)
{
	printf("Disconnect done!\n");
	OBEX_TransportDisconnect(handle);
}


int fillstream(obex_t *handle, obex_object_t *object)
{
	int                    actual;
	obex_headerdata_t       hv;
    obex_app_context *gt = OBEX_GetUserData(handle);
    btxfer_state *state = (btxfer_state*)gt->user_data;
    bt_device* dev = state->current_selected_bt_device;
    
	printf("Filling stream!\n");
    if(dev->buffer == NULL){
        dev->buffer = malloc(OBEX_STREAM_CHUNK);
    }
    memset(dev->buffer, 0, OBEX_STREAM_CHUNK);
    actual = read(dev->fd, dev->buffer, OBEX_STREAM_CHUNK);
	if(actual > 0) {
		/* Read was ok! */
		hv.bs = dev->buffer;
		OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
				hv, actual, OBEX_FL_STREAM_DATA);
	}
	else if(actual == 0) {
		/* EOF */
		hv.bs = dev->buffer;
		close(dev->fd); dev->fd = -1;
		OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
				hv, 0, OBEX_FL_STREAM_DATAEND);
	}
	else {
		/* Error */
		hv.bs = NULL;
		close(dev->fd); dev->fd = -1;
		OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
				hv, 0, OBEX_FL_STREAM_DATA);
	}
    
    dev->current_xfer_transfer_size += actual;
	return actual;
}

void put_client_buffered(obex_t *handle)
{
    obex_object_t *object;
    obex_app_context *gt = OBEX_GetUserData(handle);
    btxfer_state *state = (btxfer_state*)gt->user_data;
    bt_device* dev = state->current_selected_bt_device;
    
    obex_headerdata_t hd;
    char rname[256]; /* remote name */
    char* lname = dev->current_xfer_file; /* local name */
    char* name;
    unsigned int rname_size;
    uint8_t *buf;
    int file_size;
    struct stat stats;
    /* get the size of the file */
    stat(lname, &stats);
    file_size = stats.st_size;
    dev->current_xfer_file_size = file_size;
    dev->current_xfer_transfer_size = 0;
    /* open file */
    dev->fd = open(lname, O_RDONLY, 0);
    if (dev->fd < 0) {
        return;
    }
    
    /* Build object */
    object = OBEX_ObjectNew(handle, OBEX_CMD_PUT);
    
    /* set the name of the file to header in unicode */
    name = basename(lname);
    rname_size = OBEX_CharToUnicode(rname, name, sizeof(rname));
    /*
    free(name);
    */
    hd.bs = rname;
    OBEX_ObjectAddHeader(handle, object, 
                         OBEX_HDR_NAME, hd, rname_size, 0);
    /* set the size of the file */
    hd.bq4 = file_size;
    OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, hd, sizeof(uint32_t), 0);
    
    /* set the body as stream */
    hd.bs = NULL;
    OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, hd, 0, OBEX_FL_STREAM_START);

    /* notify the UI that we are now sending the file */
    state->obex_ui_event(CLIENT_EVENT_PUT_START, 
                         state, name, file_size);

    /* issue obex request */
    OBEX_Request(handle, object);
    /* go into transaction till the request is over */
    syncwait(handle);
}

//
//
//
void put_client_done(obex_t *handle, obex_object_t *object, int obex_rsp)
{
	if(obex_rsp == OBEX_RSP_SUCCESS) {
		printf("PUT successful!\n");
        obex_app_context *gt = OBEX_GetUserData(handle);
        /* lets get the conditional mutex and signal it */
        
    
	}	
	else {
		printf("PUT failed 0x%02x!\n", obex_rsp);
	}
}

//
//
//
void setpath_client(obex_t *handle)
{
	char setpath_data[2] = {0,0};
	obex_object_t *object;
	char path[200];
	int path_size;
	obex_headerdata_t hd;

	printf("SETPATH> ");
	scanf("%s", path);

	if(! (object = OBEX_ObjectNew(handle, OBEX_CMD_SETPATH)))	{
		printf("Error\n");
		return;
	}

	path_size = OBEX_CharToUnicode(path, path, sizeof(path));

	hd.bs = path;
	OBEX_ObjectAddHeader(handle, object, OBEX_HDR_NAME, hd,
				path_size, OBEX_FL_FIT_ONE_PACKET);

	OBEX_ObjectSetNonHdrData(object, setpath_data, 2);
	OBEX_Request(handle, object);
	syncwait(handle);
}

//
//
//
void setpath_client_done(obex_t *handle, obex_object_t *object, int obex_rsp)
{
	if(obex_rsp == OBEX_RSP_SUCCESS) {
		printf("SETPATH successful!\n");
	}	
	else {
		printf("SETPATH failed 0x%02x!\n", obex_rsp);
	}
}
	
