/*********************************************************************
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>

#ifdef HAVE_BLUETOOTH
#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#endif

#include <openobex/obex.h>

#include "btxfer.h"
#include "obex_app.h"
#include "obex_client.h"
#include "obex_server.h"


void cleanup_xfer_transfer(btxfer_state* state, int mode){
    bt_device* dev = &state->bt_local_device;
    if(mode == OBEX_SERVER){
        dev = &state->bt_local_device;
        if(dev->current_xfer_file_friendly_name)
            free(dev->current_xfer_file_friendly_name);
    }else{
        dev = state->current_selected_bt_device;
        if(dev->buffer){
            free(dev->buffer);
            dev->buffer = NULL;
        }
    }
    dev->current_xfer_transfer_size = 0;
    dev->current_xfer_file_size = 0;
    dev->current_xfer_file_friendly_name = NULL;
    /* update UI */
    state->obex_ui_event(CLIENT_EVENT_REQ_DONE, state, mode);
}
        
//
// Called by the obex-layer when some event occurs.
//
void obex_event(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
    /* retreive crucial state info */
    obex_app_context *gt;
    gt = OBEX_GetUserData(handle);
    btxfer_state *state = (btxfer_state*)gt->user_data;
            
	switch (event)	{
	case OBEX_EV_PROGRESS:
        if(mode != OBEX_CLIENT) {
            /* all incomplete requests come as progress
               so even a PUT (0x02) would have to be first
               handled in server_request as we need to
               extract important info like name and file
               size */
            server_request(handle, object, event, obex_cmd);
        }
        /* tell the UI something is happening here */
		state->obex_ui_event(CLIENT_EVENT_PUT_CONT, state, mode);
        break;

    case OBEX_EV_ABORT:
        /* @todo : link to the cancel from other side, so depending
        on the state we need to clean up here */
        /* infor UI to clean up */
        /*
        cleanup_xfer_transfer(state, mode);
		*/
        break;

	case OBEX_EV_REQDONE:
        /* this is called when a final OBEX command is done */
		if(mode == OBEX_CLIENT) {
			client_done(handle, object, obex_cmd, obex_rsp);
        }else{
        	server_done(handle, object, obex_cmd, obex_rsp);
		}
        cleanup_xfer_transfer(state, mode);
		break;
	case OBEX_EV_REQHINT:
        /* here we might be able to check what req is comming */
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE,
                          OBEX_RSP_SUCCESS);
		/* here we need to get into the read stream mode */
        if(mode != OBEX_CLIENT) {
            /*@todo : we need to put up a dialog box here to
               user if he wants to accept BUT we dont yet have 
               the file name and the size */
            if(obex_cmd == OBEX_CMD_PUT){
                OBEX_ObjectReadStream(handle, object, NULL);
            }
        }
        break;

	case OBEX_EV_REQ:
		server_request(handle, object, event, obex_cmd);
		break;

	case OBEX_EV_LINKERR:
        /* @todo : this is when the link gets broken : so if the transfer is
           active we need to show to the user a dialog box about
           error condition */
		OBEX_TransportDisconnect(handle);
        cleanup_xfer_transfer(state, mode);
        gt->serverdone = TRUE;
        
        {
        bt_device* dev = &state->bt_local_device;
        close(dev->fd);
        dev->fd = 0;
        dev->buffer = NULL;
        }
        break;
    case OBEX_EV_STREAMAVAIL:
        /* now we have data available to be read */
        if(mode != OBEX_CLIENT) {
            bt_device* dev = &state->bt_local_device;
            int len = 0;
            int actual = 0;
            len = OBEX_ObjectReadStream(handle, object, &dev->buffer);
            dev->current_xfer_transfer_size += len;
            if(dev->fd == 0){
                char filename[512] = {0,};
                strcat(filename, state->xfer_receive_file_dir);
                strcat(filename, dev->current_xfer_file_friendly_name);
                dev->fd = open(filename, O_RDWR | O_CREAT | O_APPEND | O_TRUNC);
                if(dev->fd < 0){
                    /* @todo : error to save file, send ABORT*/
                }    
            }
            /* write to file */
            while(len != 0){
                actual = write(dev->fd, dev->buffer + actual, len);
                len -= actual;
            }
        }
        break;
	case OBEX_EV_STREAMEMPTY:
		fillstream(handle, object);
		break;

	default:
		printf("Unknown event %02x!\n", event);
		break;
	}
}

int obex_xfer_init (int bt_channel, 
                    char *bt_address,
                    obex_t **obex_handle, 
                    void* user_data)
{
	bdaddr_t bdaddr;
    uint8_t channel;
    obex_t *handle = NULL;
    obex_app_context* context = (obex_app_context*)malloc(sizeof(obex_app_context));
    memset(context, 0, sizeof(obex_app_context));
    
    /* set channel and convert bt_address to internal
       consumable bdaddr_t */
       
    channel = (bt_channel == -1) ? BT_CHANNEL : bt_channel;
	str2ba(bt_address, &bdaddr);
    
    /* lets save context */
    context->channel = channel;
    context->bdaddr = bdaddr;
    context->user_data = user_data;
    
    /* initialize the OBEX engine */
	if(! (handle = OBEX_Init(OBEX_TRANS_BLUETOOTH,
                             obex_event, 0)))
    {
        return -1;
	}
	OBEX_SetUserData(handle, context);
    *obex_handle = handle;
	return 0;
}

/*
@todo : convert all return values to ENUM
*/
int obex_xfer_transport_connect (obex_t *obex_handle)
{

    obex_app_context *context = (obex_app_context*)OBEX_GetUserData(obex_handle);
    bdaddr_t bdaddr = context->bdaddr;
    uint8_t channel = context->channel;
    
    if (bacmp(&context->bdaddr, BDADDR_ANY) == 0) {
        return -1;
    }
    
    if(BtOBEX_TransportConnect(obex_handle, BDADDR_ANY, &bdaddr, channel) <0) {
        return -2;
    }
    return 0;
}

int obex_xfer_transport_disconnect (obex_t *obex_handle)
{
    OBEX_TransportDisconnect(obex_handle);
}

int obex_xfer_connect (obex_t *obex_handle)
{
    connect_client(obex_handle);
    return 0;
}

int obex_xfer_send (obex_t *obex_handle)
{
    /*
    put_client(obex_handle);
    */
    put_client_buffered(obex_handle);
}

int obex_xfer_disconnect (obex_t *obex_handle)
{
    disconnect_client(obex_handle);
}

#if 0
    while (!end) {
        printf("> ");
        scanf("%s", cmd);
        switch (cmd[0] | 0x20)  {
            case 'q':
                end=1;
            break;
            case 'g':
                get_client(handle, &global_context);
            break;
            case 't':
                setpath_client(handle);
            break;
            case 'x':
                push_client(handle);
            break;
            case 's':
                if(btobex) {
                    if(BtOBEX_ServerRegister(handle, BDADDR_ANY, channel) < 0) {
                        printf("Server register error! (Bluetooth)\n");
                        break;
                    }
                }
                /* No process server events */
                server_do(handle);
            break;
            default:
                printf("Unknown command %s\n", cmd);
        }
    }
#endif