/*
  WifiInfo - Show current Wifi status
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cmath>

#include <Lum/Audio/PlaySound.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/FuelGauge.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#if defined(CAN_USE_STATUS_HILDON)
  #include "StatusHildon.h"
#elif defined(CAN_USE_STATUS_LINUX)
  #include "StatusLinux.h"
#else
  #include "StatusNone.h"
#endif

#include "Configuration.h"
#include "ConfigDialog.h"

#define NETWORK_REFRESH    10
#define POLL_ERROR_TIMEOUT 60

#include <iostream>
static Lum::Def::AppInfo info;

typedef Lum::Model::StdTable<Status::Network> NetworkModel;
typedef Lum::Base::Reference<NetworkModel>    NetworkModelRef;

class NetworkObject : public Lum::Object
{
private:
  Lum::OS::FontRef  font;
  Lum::OS::ColorRef red;
  Lum::OS::ColorRef green;
  Lum::OS::ColorRef yellow;
  Lum::OS::ColorRef grey;
  Status::Network   network;

public:
  NetworkObject()
  : font(Lum::OS::display->GetFont()),
    red(1,0,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
    green(0,1,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
    yellow(1,0.5,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
    grey(0.7,0.7,0.7,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor))
  {
    // no code
  }

  void SetNetwork(const Status::Network& network)
  {
    this->network=network;
  }

  void CalcSize()
  {
    minWidth=font->height+
             Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder)+
             3*font->StringWidth(L"m");
    minHeight=Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
              font->height+
              Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
              font->height+
              Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);

    width=minWidth;
    height=minHeight;

    Object::CalcSize();
  }

  void Draw(int x, int y, size_t w, size_t h)
  {
    Object::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    Lum::OS::DrawInfo *draw=GetDrawInfo();

    //
    // Network security type
    //

    if (network.encryption!=0) {
      if (network.encryption & Status::cryptNone) {
        draw->PushForeground(green);
      }
      else if (network.encryption & Status::cryptWEP) {
        draw->PushForeground(yellow);
      }
      else if (network.encryption & Status::cryptWPA_PSK) {
        draw->PushForeground(red);
      }
      else if (network.encryption & Status::cryptWPA_EAP) {
        draw->PushForeground(red);
      }
      else if (network.encryption & Status::cryptWPA2) {
        draw->PushForeground(red);
      }
    }
    else {
      draw->PushForeground(grey);
    }

    draw->FillRectangle(this->x,
                        this->y+Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder),
                        font->height,font->height);
    draw->PopForeground();

    //
    // Name of network
    //

    draw->PushFont(font);
    draw->PushForeground(parent->GetTextColor(draw));
    draw->DrawString(this->x+font->height+Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder),
                     this->y+font->ascent,
                     network.essid);
    draw->PopForeground();
    draw->PopFont();

    //
    // Channel number
    //

    if (network.channel>=0) {
      std::wstring channel=L"Channel: "+Lum::Base::NumberToWString(network.channel);

      draw->PushFont(font);
      draw->PushForeground(parent->GetTextColor(draw));
      draw->DrawString(this->x+width-font->StringWidth(channel)+1,
                       this->y+font->ascent,
                       channel);
      draw->PopForeground();
      draw->PopFont();
    }

    //
    // Signal strength
    //

    double percent=network.quality/100.0;

    if (percent<=0.2) {
      draw->PushForeground(red);
    }
    else {
      draw->PushForeground(green);
    }

    draw->FillRectangle(this->x,
                        this->y+
                        Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                        font->height+
                        Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder),
                        (size_t)(width*percent),height-2);


    draw->PopForeground();

    Lum::OS::FontRef    font=Lum::OS::display->GetFont();
    std::wstring        label=Lum::Base::NumberToWString(lround(percent*100))+L"%";
    Lum::OS::FontExtent extent;

    font->StringExtent(label,extent);

    draw->PushForeground(Lum::OS::Display::textColor);
    draw->PushFont(font,0);
    draw->DrawString(this->x+(width-extent.width+extent.left+extent.right)/2,
                     this->y+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font->height+
                     Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder)+
                     font->ascent,
                     label);
    draw->PopFont();
    draw->PopForeground();
  }

  static NetworkObject* Create(bool horizFlex, bool vertFlex)
  {
    NetworkObject *object=new NetworkObject();

    object->SetFlex(horizFlex,vertFlex);

    return object;
  }
};

class NetworkDataProvider : public NetworkModel::DataProvider
{
private:
  NetworkObject *object;

public:
  NetworkDataProvider()
  : object(NetworkObject::Create(true,false))
  {
    // no code
  }

  virtual ~NetworkDataProvider()
  {
    delete object;
  }

  /*
  std::wstring GetString(const NetworkModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return iter->essid;
    case 2:
        if (iter->channel>=0) {
          return Lum::Base::NumberToWString(iter->channel);
        }
        else {
          return L"?";
        }
    default:
      return L"";
    }
  } */

  Lum::Object* GetObject(const NetworkModel::Iterator& iter, size_t column) const
  {
    object->SetNetwork(*iter);
    return object;
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::StringRef    type;
  Lum::Model::StringRef    essid;
  Lum::Model::StringRef    accesspoint;
  Lum::Model::StringRef    client;
  Lum::Model::DoubleRef    quality;
  Lum::Model::StringRef    signalNoise;
  Lum::Model::StringRef    channel;
  Lum::Model::BooleanRef   powerSaving;

  Lum::Model::IntRef       refreshTime;

  NetworkModelRef          tableModel;

  Lum::Model::BooleanRef   scanForNetworks;
  Lum::Model::BooleanRef   wardrive;
  Lum::Model::ActionRef    powerSavingOnAction;
  Lum::Model::ActionRef    powerSavingOffAction;

  Lum::Model::ActionRef    statusTimer;
  Lum::Model::ActionRef    statusErrorTimer;
  Lum::Model::ActionRef    networksTimer;
  Lum::Model::ActionRef    networksErrorTimer;

  Lum::Model::ActionRef    configAction;
  Lum::Model::ActionRef    aboutAction;

#if defined(CAN_USE_STATUS_HILDON)
    StatusHildon           status;
#elif defined(CAN_USE_STATUS_LINUX)
    StatusLinux            status;
#else
    StatusNone             status;
#endif

  Lum::Audio::PlaySound    *scanNetworkSound;
  Lum::Audio::PlaySound    *openNetworkSound;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textView;

    textView=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textView->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textView;
  }

  void ParseArguments()
  {
    size_t i=0;

    while (i<Lum::OS::display->GetArgCount()) {
      if (Lum::OS::display->GetArg(i)==L"-d" &&
          i+1<Lum::OS::display->GetArgCount()) {
        i++;
        status.SetDefaultInterface(Lum::Base::WStringToString(Lum::OS::display->GetArg(i)));
      }

      i++;
    }
  }

  void CancelStatus()
  {
    Lum::OS::display->RemoveTimer(statusTimer);
    Lum::OS::display->RemoveTimer(statusErrorTimer);
  }

  void RequestStatus()
  {
    CancelStatus();

    if (status.UpdateStatus()) {
      Lum::OS::display->AddTimer(refreshTime->Get()+POLL_ERROR_TIMEOUT,0,statusErrorTimer);
    }
    else {
      status.type=Status::typeNone;
      UpdateStatus();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,statusTimer);
    }
  }

  void CancelNetworks()
  {
    Lum::OS::display->RemoveTimer(networksTimer);
    Lum::OS::display->RemoveTimer(networksErrorTimer);
  }

  void RequestNetworks()
  {
    CancelNetworks();

    if (status.SupportsNetworkRetrieval()) {
      if (scanNetworkSound!=NULL) {
        std::cout << "*** Playing scan sound..." << std::endl;
        scanNetworkSound->Play();
      }
      if (status.UpdateNetworks()) {
        Lum::OS::display->AddTimer(NETWORK_REFRESH+POLL_ERROR_TIMEOUT,0,networksErrorTimer);
      }
      else {
        tableModel->Clear();
        Lum::OS::display->AddTimer(NETWORK_REFRESH,0,networksTimer);
      }
    }
  }

  void UpdateStatus()
  {
    if (status.type==Status::typeNone || status.bitrate==0) {
      type->Set(L"-");
      essid->Set(L"-");
      channel->Set(L"-");
      accesspoint->Set(L"-");
      client->Set(L"-");
      quality->SetNull();
      signalNoise->Set(L"-");
      powerSaving->SetNull();
      return;
    }

    std::wstring tmr;

    essid->Set(status.essid);

    tmr=status.typeName;

    tmr.append(L" ");

    switch (status.type) {
    case Status::typeAuto:
      tmr.append(L"Auto");
      break;
    case Status::typeAdHoc:
      tmr.append(L"Ad-Hoc");
      break;
    case Status::typeInfrastructure:
      tmr.append(L"Infrastructure");
      break;
    case Status::typeMaster:
      tmr.append(L"Master");
      break;
    case Status::typeRepeater:
    tmr.append(L"Repeater");
      break;
    case Status::typeSecond:
      tmr.append(L"Secondary master");
      break;
    case Status::typeMonitor:
      tmr.append(L"Passive monitor");
      break;
    case Status::typeMesh:
      tmr.append(L"Mesh");
      break;
    default:
      tmr.append(L"-");
    }

    tmr.append(L" ");

    if (status.bitrate>1000000 && status.bitrate==(status.bitrate/1000000)*1000000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000000));
      tmr.append(L"Mb/s");
    }
    else if (status.bitrate>1000 && status.bitrate==(status.bitrate/1000)*1000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000));
      tmr.append(L"Kb/s");
    }
    else {
    tmr.append(Lum::Base::NumberToWString(status.bitrate));
      tmr.append(L"b/s");
    }

    type->Set(tmr);

    if (status.channel>=0) {
      channel->Set(Lum::Base::NumberToWString(status.channel));
    }
    else {
      channel->Set(L"-");
    }

    accesspoint->Set(status.accesspoint);

    client->Set(std::wstring(status.ip)+L" ("+status.mac+L")");

    if (status.quality>=0 && status.quality<=100) {
      quality->Set(status.quality);
    }
    else {
      quality->SetNull();
    }

    signalNoise->Set(Lum::Base::NumberToWString(status.signal-0x100)+L" dBm / "+Lum::Base::NumberToWString(status.noise-0x100)+L" dBm");

    switch (status.powerSaving) {
    case Status::powerSavingUnknown:
      powerSaving->SetNull();
      break;
    case Status::powerSavingOn:
      powerSaving->Set(true);
      break;
    case Status::powerSavingOff:
      powerSaving->Set(false);
      break;
    }
  }

  void UpdateNetworks()
  {
    std::cout << "Updating networks..." << std::endl;

    // First try to update existing entries in the current table
    for (std::list<Status::Network>::const_iterator iter=status.networks.begin();
         iter!=status.networks.end();
         ++iter) {
      size_t i;

      // Find the right entry
      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->essid==tableModel->GetEntry(i).essid) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) { // Found an entry => Update
        tableModel->GetEntry(i).quality=iter->quality;
      }
      else { // No entry => add
        tableModel->Append(*iter);
      }
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;

    while(i<=tableModel->GetRows()) {
      std::wstring                               essid=tableModel->GetEntry(i).essid;
      std::list<Status::Network>::const_iterator iter;

      // Find essid from table entry in network list
      iter=status.networks.begin();
      while (iter!=status.networks.end()) {
        if (iter->essid==essid) {
          break;
        }

        ++iter;
      }

      if (iter==status.networks.end()) { // table entry is not in list
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
    std::cout << "Updating networks done." << std::endl;

    if (wardrive->IsEnabled() && wardrive->Get()) {
      bool playsound=false;

      std::cout << "Wardriving..." << std::endl;

      for (std::list<Status::Network>::const_iterator iter=status.networks.begin();
           iter!=status.networks.end();
           ++iter) {
        if (iter->quality>=25 && (iter->encryption & Status::cryptNone)) {
          playsound=true;
        }
      }

      if (playsound) {
        std::cout << "At least one open network with quality>=25% found!" << std::endl;
        if (openNetworkSound!=NULL) {
          openNetworkSound->Play();
        }
        else {
          Lum::OS::display->Beep();
        }
      }
      std::cout << "Wardriving done." << std::endl;
    }
  }

public:
  MyWindow()
  : type(new Lum::Model::String()),
    essid(new Lum::Model::String()),
    accesspoint(new Lum::Model::String()),
    client(new Lum::Model::String()),
    quality(new Lum::Model::Double()),
    signalNoise(new Lum::Model::String()),
    channel(new Lum::Model::String()),
    powerSaving(new Lum::Model::Boolean()),
    refreshTime(new Lum::Model::Int(5)),
    tableModel(new NetworkModel(new NetworkDataProvider())),
    scanForNetworks(new Lum::Model::Boolean()),
    wardrive(new Lum::Model::Boolean(false)),
    powerSavingOnAction(new Lum::Model::Action()),
    powerSavingOffAction(new Lum::Model::Action()),
    statusTimer(new Lum::Model::Action()),
    statusErrorTimer(new Lum::Model::Action()),
    networksTimer(new Lum::Model::Action()),
    networksErrorTimer(new Lum::Model::Action()),
    configAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    scanNetworkSound(NULL),
    openNetworkSound(NULL)
  {
    quality->SetRange(0,100);
    refreshTime->SetRange(0,10);

    powerSaving->Disable();
    wardrive->Disable();

    Observe(scanForNetworks);
    Observe(wardrive);
    Observe(powerSavingOnAction);
    Observe(powerSavingOffAction);

    Observe(refreshTime);
    Observe(statusTimer);
    Observe(statusErrorTimer);
    Observe(networksTimer);
    Observe(networksErrorTimer);
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
    Observe(configAction);
    Observe(aboutAction);

    Observe(status.statusChangedAction);
    Observe(status.networksChangedAction);

    scanForNetworks->Set(false);

    Lum::Base::Path openNetworkSoundPath;

    openNetworkSoundPath.SetNativeDir(Lum::Base::Path::GetCWD());
    openNetworkSoundPath.AppendDir(L"data");
    openNetworkSoundPath.AppendDir(L"sounds");
    openNetworkSoundPath.SetBaseName(L"opennetwork.wav");

#if defined(APP_DATADIR)
    if (!openNetworkSoundPath.Exists()) {
      openNetworkSoundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
      openNetworkSoundPath.AppendDir(L"sounds");
      openNetworkSoundPath.SetBaseName(L"opennetwork.wav");
    }
#endif

    if (openNetworkSoundPath.Exists()) {
      openNetworkSound=Lum::Audio::GetPlaySound();
      if (openNetworkSound!=NULL) {
        openNetworkSound->SetFilename(openNetworkSoundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
      }
    }

    Lum::Base::Path scanNetworkSoundPath;

    scanNetworkSoundPath.SetNativeDir(Lum::Base::Path::GetCWD());
    scanNetworkSoundPath.AppendDir(L"data");
    scanNetworkSoundPath.AppendDir(L"sounds");
    scanNetworkSoundPath.SetBaseName(L"scannetwork.wav");

#if defined(APP_DATADIR)
    if (!scanNetworkSoundPath.Exists()) {
      scanNetworkSoundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
      scanNetworkSoundPath.AppendDir(L"sounds");
      scanNetworkSoundPath.SetBaseName(L"scannetwork.wav");
    }
#endif


    if (scanNetworkSoundPath.Exists()) {
      scanNetworkSound=Lum::Audio::GetPlaySound();
      if (scanNetworkSound!=NULL) {
        scanNetworkSound->SetFilename(scanNetworkSoundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
      }
    }

    // For testing purposes

    /*
    status.networks.clear();

    Status::Network network;

    network.essid=L"Test 1";
    network.quality=90;
    network.channel=6;
    network.type=Status::typeInfrastructure;
    network.encryption=0;
    status.networks.push_back(network);

    network.essid=L"Test 2";
    network.quality=100;
    network.channel=6;
    network.type=Status::typeInfrastructure;
    network.encryption=Status::cryptWPA2;
    status.networks.push_back(network);

    network.essid=L"Test 3";
    network.quality=10;
    network.channel=6;
    network.type=Status::typeInfrastructure;
    network.encryption=Status::cryptWPA2;
    status.networks.push_back(network);*/
  }

  ~MyWindow()
  {
    delete scanNetworkSound;
    delete openNetworkSound;
  }

  virtual void PreInit()
  {
    Lum::Def::MultiView   multiView(Lum::Def::Desc(L"WifiInfo"));

    Lum::Label            *label;
    Lum::Panel            *hPanel;

    std::wstring          tmp;

    label=Lum::Label::Create(true,true);
    label->AddLabel(L"ESSID:",CreateTextValue(essid,28));
    label->AddLabel(L"Channel:",CreateTextValue(channel,2));
    label->AddLabel(L"Type:",CreateTextValue(type,28));
    label->AddLabel(L"AP:",CreateTextValue(accesspoint,17));
    label->AddLabel(L"Client:",CreateTextValue(client,28));
    label->AddLabel(L"Quality:",Lum::FuelGauge::Create(quality,true,false));
    label->AddLabel(L"SNR:",CreateTextValue(signalNoise,18));

    hPanel=Lum::HPanel::Create();
    hPanel->Add(new Lum::LED(powerSaving));

    if (status.CanChangePowerSaving()) {
      hPanel->AddSpace();
      hPanel->Add(Lum::Button::Create(L"_On",powerSavingOnAction));
      hPanel->AddSpace();
      hPanel->Add(Lum::Button::Create(L"O_ff",powerSavingOffAction));
    }

    label->AddLabel(L"Power saving:",hPanel);

    multiView.AddView(Lum::Def::MultiView::View(0,Lum::Def::Desc(L"Current"),label));

    if (status.SupportsNetworkRetrieval()) {
      Lum::Panel                         *panel;
      Lum::Table                         *table;
      Lum::Model::HeaderRef              headerModel;
      Lum::Model::SingleLineSelectionRef selection;
      Lum::Def::Boolean                  scanNetworksDef(Lum::Def::Desc(L"Scan for networks"),
                                                         scanForNetworks);
      Lum::Def::Boolean                  wardriveDef(Lum::Def::Desc(L"Signal open networks"),
                                                     wardrive);

      panel=Lum::VPanel::Create(true,true);
      panel->Add(Lum::HPanel::Create(true,false)
                 ->Add(Lum::Label::Create(true,false)
                        ->AddLabel(L"Scan for networks:",
                                   Lum::OS::display->GetBehaviour()->GetBooleanControl(scanNetworksDef,true,false)))
                        ->AddSpace()
                        ->Add(Lum::Label::Create(true,false)
                              ->AddLabel(L"Wardrive:",
                                         Lum::OS::display->GetBehaviour()->GetBooleanControl(wardriveDef,true,false))));
      panel->AddSpace();

      selection=new Lum::Model::SingleLineSelection;

      headerModel=new Lum::Model::HeaderImpl();
      headerModel->AddColumn(L"Network",Lum::Base::Size::stdCharWidth,15,true);
      /**
      headerModel->AddColumn(L"C",Lum::Base::Size::stdCharWidth,2);
      headerModel->AddColumn(L"Encrypt",Lum::Base::Size::stdCharWidth,7);
      headerModel->AddColumn(L"Quality",Lum::Base::Size::stdCharWidth,10);*/

      table=new Lum::Table();
      table->SetFlex(true,true);
      table->SetHeaderModel(headerModel);
      table->SetModel(tableModel);
      //table->SetSelection(selection);
      //table->SetSelectionAction(showDetail);
      //table->SetDoubleClickAction(showDetail);
      table->SetShowHeader(false);
      table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,4);
      table->GetTableView()->SetAutoFitColumns(true);
      table->GetTableView()->SetAutoHSize(true);

      panel->Add(table);

      multiView.AddView(Lum::Def::MultiView::View(1,Lum::Def::Desc(L"All"),panel));
    }

    Lum::Def::Menu *preMenu=Lum::Def::Menu::Create();

    preMenu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(configAction)
      ->End();

    Lum::Def::Menu *postMenu=Lum::Def::Menu::Create();

    postMenu
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    //SetMenu(menu);

    multiView.SetMenu(preMenu,postMenu);

    Lum::OS::display->GetBehaviour()->ApplyMultiViewDlg(this,multiView);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      ParseArguments();
      UpdateStatus();
      RequestStatus();

      if (!LoadConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Power management warning!",
                              L"This application periodically polls the WLAN stack of\n"
                              L"the OS while running (even if in background!).\n"
                              L"\n"
                              L"If this applications runs on an device in battery mode\n"
                              L"this will result in more than usual battery draining,\n"
                              L"potentially significantly reducing the time the device\n"
                              L"can run on battery!");
      }
    }
    if (model==GetClosedAction() &&
        GetClosedAction()->IsFinished()) {
      SaveConfig();
    }
    else if (model==scanForNetworks) {
      if (!scanForNetworks->IsNull() &&
          *scanForNetworks) {
        tableModel->SetEmptyText(L"(No networks found)");
        RequestNetworks();
        wardrive->Enable();
      }
      else {
        tableModel->SetEmptyText(L"(Scanning not enabled)");
        CancelNetworks();
        UpdateNetworks();
        wardrive->Disable();
      }
    }
    else if (model==powerSavingOnAction &&
             powerSavingOnAction->IsFinished()) {
      status.SetPowerSaving(true);
    }
    else if (model==powerSavingOffAction &&
             powerSavingOffAction->IsFinished()) {
      status.SetPowerSaving(false);
    }
    else if ((model==statusTimer && statusTimer->IsFinished()) ||
             (model==statusErrorTimer && statusErrorTimer->IsFinished())) {
      RequestStatus();
    }
    else if ((model==networksTimer && networksTimer->IsFinished()) ||
             (model==networksErrorTimer && networksErrorTimer->IsFinished())) {
      RequestNetworks();
    }
    else if (model==refreshTime && IsOpen()) {
      std::cout << "Refreshing: " << refreshTime->Get() << std::endl;
      RequestStatus();
    }
    else if (model==status.statusChangedAction &&
             status.statusChangedAction->IsFinished()) {
      Lum::OS::display->RemoveTimer(statusErrorTimer);
      UpdateStatus();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,statusTimer);
    }
    else if (model==status.networksChangedAction &&
             status.networksChangedAction->IsFinished()) {
      Lum::OS::display->RemoveTimer(networksErrorTimer);
      UpdateNetworks();
      Lum::OS::display->AddTimer(NETWORK_REFRESH,0,networksTimer);
    }
    else if (model==configAction && configAction->IsFinished()) {
      ConfigDialog *dialog;

      CancelStatus();
      dialog=new ConfigDialog(refreshTime);
      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }
      RequestStatus();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MyWindow>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"How is your Wifi?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

