/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "accountsmodel.hpp"
#include "providerinfo.hpp"

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

struct AccountsItem
{
  QString alias;
  QIcon icon;
  QString providerName;
};

//////////////////////////////////////////////////////////////////////////

class AccountsModelPrivate : public QObject
{
  Q_OBJECT

public:
  AccountsModel* self;

  // Properties
  AccountsPtr accounts;

  QList<AccountsItem> data;

  // Functions
public slots:
  void onModelReset();
  void onAccountAdded(const AccountConfig& config);
  void onAccountRemoved(const AccountConfig& config);
};

//////////////////////////////////////////////////////////////////////////

void AccountsModelPrivate::onModelReset()
{
  // Clear the data cache
  data.clear();

  if ( !accounts )
    return;

  // Initially fill the data
  foreach(const AccountConfig& config, accounts->accounts())
  {
    // Get the provider info
    ProviderInfo providerInfo = accounts->providerInfoForAlias(config.alias());

    // Make a new accounts item
    AccountsItem item;
    item.alias        = config.alias();
    item.icon         = providerInfo.icon();
    item.providerName = providerInfo.visibleName();

    data << item;
  }
}

//////////////////////////////////////////////////////////////////////////

void AccountsModelPrivate::onAccountAdded(const AccountConfig& config)
{
  // Get the provider info
  ProviderInfo providerInfo = accounts->providerInfoForAlias(config.alias());

  // Make a new accounts item
  AccountsItem item;
  item.alias        = config.alias();
  item.icon         = providerInfo.icon();
  item.providerName = providerInfo.visibleName();

  self->beginInsertRows(QModelIndex(), data.size(), data.size());
  data << item;
  self->endInsertRows();
}

//////////////////////////////////////////////////////////////////////////

void AccountsModelPrivate::onAccountRemoved(const AccountConfig& config)
{
  for(int i=0; i < data.size(); ++i)
  {
    const AccountsItem& item = data.at(i);

    if ( item.alias != config.alias() )
      continue;

    self->beginRemoveRows(QModelIndex(), i, i);
    data.removeAt(i);
    self->endRemoveRows();

    break;
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

AccountsModel::AccountsModel(QObject* parent /* = 0 */)
: QAbstractItemModel(parent)
, d(new AccountsModelPrivate)
{
  d->self = this;

  // Connect the reset event
  connect(this, SIGNAL(modelReset()), d, SLOT(onModelReset()));

  QHash<int, QByteArray> roleNames;
  roleNames.insert(AccountAliasRole, "alias");
  roleNames.insert(AccountProviderNameRole, "providerName");
  roleNames.insert(AccountProviderIconRole, "providerIcon");

  setRoleNames(roleNames);
}

//////////////////////////////////////////////////////////////////////////

AccountsModel::~AccountsModel()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void AccountsModel::setAccounts(const AccountsPtr& accounts)
{
  // Did something change?
  if ( d->accounts == accounts )
    return;

  // Disconnect the old accounts
  if ( d->accounts )
  {
    disconnect(d->accounts.data(), SIGNAL(accountAdded(AccountConfig)), d, SLOT(onAccountAdded(AccountConfig)));
    disconnect(d->accounts.data(), SIGNAL(accountRemoved(AccountConfig)), d, SLOT(onAccountRemoved(AccountConfig)));
  }

  // Take the accounts
  d->accounts = accounts;

  // Connect the new accounts
  if ( d->accounts )
  {
    connect(d->accounts.data(), SIGNAL(accountAdded(AccountConfig)), d, SLOT(onAccountAdded(AccountConfig)));
    connect(d->accounts.data(), SIGNAL(accountRemoved(AccountConfig)), d, SLOT(onAccountRemoved(AccountConfig)));
  }

  // Reload the model data
  reset();
}

//////////////////////////////////////////////////////////////////////////

AccountsPtr AccountsModel::accounts() const
{
  return d->accounts;
}

//////////////////////////////////////////////////////////////////////////

int AccountsModel::columnCount(const QModelIndex& parent /* = QModelIndex */) const
{
  Q_UNUSED(parent);
  return 1;
}

//////////////////////////////////////////////////////////////////////////

int AccountsModel::rowCount(const QModelIndex& parent /* = QModelIndex */) const
{
  return parent.isValid() ? 0 : d->data.size();
}

//////////////////////////////////////////////////////////////////////////

QModelIndex AccountsModel::parent(const QModelIndex& index) const
{
  Q_UNUSED(index);
  return QModelIndex();
}

//////////////////////////////////////////////////////////////////////////

QModelIndex AccountsModel::index(int row, int column, const QModelIndex& parent /* = QModelIndex */) const
{
  // Sanity check
  if ( row < 0 || row >= d->data.size() || column != 0 || parent.isValid() )
    return QModelIndex();

  return createIndex(row, column);
}

//////////////////////////////////////////////////////////////////////////

QVariant AccountsModel::headerData(int section, Qt::Orientation orientation, int role /* = Qt::DisplayRole */) const
{
  if ( section != 0 || orientation != Qt::Horizontal || role != Qt::DisplayRole )
    return QVariant();

  return tr("Accounts");
}

//////////////////////////////////////////////////////////////////////////

QVariant AccountsModel::data(const QModelIndex& index, int role /* = Qt::DisplayRole */) const
{
  int row = index.row();

  if ( !index.isValid() || index.column() != 0 || row < 0 || row >= d->data.size() )
    return QVariant();

  const AccountsItem& item = d->data.at(row);

  switch(role)
  {
  case Qt::DisplayRole:
    return QString("%1 (%2)").arg(item.alias).arg(item.providerName);

  case Qt::DecorationRole:
    return item.icon;

  case AccountAliasRole:
    return item.alias;

  case AccountProviderNameRole:
    return item.providerName;

  case AccountProviderIconRole:
    return item.icon;

  default:
    break;
  }

  return QVariant();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "accountsmodel.moc"
