/* Tracker-update: add/remove mimetypes in tracker
 * Copyright (C) 2009, Ivan Frade (ivan.frade@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 */
#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>
#include <sqlite3.h>

#define EXIT_FAILURE -1
#define EXIT_SUCCESS  0

#define USER "user"

static gchar *category = NULL;
static gchar **add_mimes = NULL;
static gchar **rm_mimes = NULL;

static GOptionEntry entries[] = {
	{ "category", 'c', 0, G_OPTION_ARG_STRING, &category,
	  "Category to update",
	/* Translators: this will appear like --option=STRING running --help */
	  NULL
	},
	{ "add", 'a', 0, G_OPTION_ARG_STRING_ARRAY, &add_mimes,
	  "mimetypes to add to the category",
	/* Translators: this will appear like --option=NUMBER running --help */
	  NULL
	},
	{ "del", 'd', 0, G_OPTION_ARG_STRING_ARRAY, &rm_mimes,
          "mimetypes to remove from the category",
	/* Translators: this will appear like --option=NUMBER running --help */
	  NULL
	},
	{ NULL }
};

/*
 * Adapted from libtracker-common/tracker-type-utils.c
 */
static GSList *
string_list_to_gslist (gchar **strv,
                       gsize   size)
{
	GSList *list;
	gsize	i;
	gsize	size_used;

	g_return_val_if_fail (strv != NULL, NULL);

	if (size < 1) {
		size_used = g_strv_length (strv);
	} else {
		size_used = size;
	}

	list = NULL;

	for (i = 0; i < size; i++) {
		if (strv[i]) {
			list = g_slist_prepend (list, g_strdup (strv[i]));
		} else {
			break;
		}
	}

	return g_slist_reverse (list);
}

gchar **
gslist_to_string_list (GSList *list)
{
	GSList	*l;
	gchar  **strv;
	gint	 i;

	strv = g_new0 (gchar*, g_slist_length (list) + 1);

	for (l = list, i = 0; l; l = l->next) {
		if (!l->data) {
			continue;
		}

		strv[i++] = g_strdup (l->data);
	}

	strv[i] = NULL;

	return strv;
}

/*
 * Return TRUE if there is some modification to be saved
 */
static gboolean
update_services_file (GKeyFile *services, 
                      const gchar *category,
                      gchar **add_mimes,
                      gchar **rm_mimes)
{
        gchar  **mimes;
        GSList  *mime_list = NULL, *item = NULL;
        gint     mimes_counter;
        GError  *error = NULL;
        gint     it;
        gboolean modified = FALSE;

        mimes = g_key_file_get_string_list (services, category, "Mimes", 
                                            &mimes_counter, &error);
        if (!mimes) {
                if (add_mimes) {
                        g_key_file_set_string_list (services, 
                                                    category, 
                                                    "Mimes",
                                                    (const gchar * const *)add_mimes, 
                                                    g_strv_length (add_mimes));
                        modified = TRUE;
                } 

                if (rm_mimes) {
                        g_message ("'%s' doesn't have mimes assigned. Unable to remove anything\n",
                                   category);
                }
                return modified;
        }

        mime_list = string_list_to_gslist (mimes, g_strv_length (mimes));

        if (add_mimes) {
                for (it = 0; add_mimes[it] != NULL; it++) {
                        if (g_slist_find_custom (mime_list, 
                                                 add_mimes[it], 
                                                 (GCompareFunc)g_strcmp0) == NULL) {
                                mime_list = g_slist_prepend (mime_list, add_mimes [it]);
                                modified = TRUE;
                        } else {
                                g_debug ("'%s' already in the mimes list", add_mimes [it]);
                        }
                }
        }

        if (rm_mimes) {
                for (it = 0; rm_mimes[it] != NULL; it++) {
                        item = g_slist_find_custom (mime_list, 
                                                    rm_mimes[it], 
                                                    (GCompareFunc)g_strcmp0);
                        if (item) {
                                mime_list = g_slist_delete_link (mime_list, item);
                                modified = TRUE;
                        } else {
                                g_debug ("'%s' is not in '%s' categories list", rm_mimes [it], category);
                        }
                }
        }

        if (!modified) {
                return FALSE;
        }

        if (g_slist_length (mime_list) == 0) {
                g_key_file_remove_key  (services, category, 
                                        "Mimes", NULL);
        } else {
                gchar **new_mimes = gslist_to_string_list (mime_list);
                g_key_file_set_string_list (services, category, 
                                            "Mimes", 
                                            (const gchar * const *)new_mimes, 
                                            g_strv_length (new_mimes));
        }

        g_strfreev (mimes);

        return TRUE;
}

gint
get_category_id (sqlite3 *common, const gchar *category)
{
        gint rc;
        sqlite3_stmt *stmt;
        gint typeId;

        rc = sqlite3_prepare_v2 (common,
                                 "SELECT TypeID FROM ServiceTypes WHERE TypeName = ?",
                                 -1,
                                 &stmt,
                                 NULL);
        g_return_val_if_fail (rc == SQLITE_OK, -1);
                            
        rc = sqlite3_bind_text (stmt, 1, category, -1, SQLITE_STATIC);
        g_return_val_if_fail (rc == SQLITE_OK, -1);
        
        rc = sqlite3_step (stmt);
        if (rc == SQLITE_ROW) {
                typeId = sqlite3_column_int (stmt, 0);
        } else {
                typeId = -1;
        }

        sqlite3_finalize (stmt);
        return typeId;
}

void
update_mimes_table_add (sqlite3 *common, gint categoryid, gchar **add_mimes)
{
        sqlite3_stmt *stmt;
        gint rc, it;
        const gchar *query = "INSERT OR REPLACE INTO FileMimes (Mime, ServiceTypeId) VALUES (?, ?)";
        rc = sqlite3_prepare_v2 (common,
                                 query,
                                 -1,
                                 &stmt,
                                 NULL);
        g_return_if_fail (rc == SQLITE_OK);

        for (it = 0; add_mimes[it] != NULL; it++) {
                rc = sqlite3_bind_text (stmt, 1, add_mimes[it], -1, SQLITE_STATIC);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_bind_int (stmt, 2, categoryid);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_step (stmt);
                if (rc != SQLITE_DONE) {
                        g_warning ("Unable to add mime '%s' (error %d)\n", add_mimes[it], rc);
                }
                sqlite3_reset (stmt);
        }

        sqlite3_finalize (stmt);
}

void
update_mimes_table_rm (sqlite3* common, gint categoryid, gchar **rm_mimes)
{
        sqlite3_stmt *stmt;
        gint rc, it;
        const gchar *query = "DELETE FROM FileMimes WHERE Mime = ? AND ServiceTypeID = ?";
        rc = sqlite3_prepare_v2 (common,
                                 query,
                                 -1,
                                 &stmt,
                                 NULL);
        g_return_if_fail (rc == SQLITE_OK);

        for (it = 0; rm_mimes[it] != NULL; it++) {
                rc = sqlite3_bind_text (stmt, 1, rm_mimes[it], -1, SQLITE_STATIC);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_bind_int (stmt, 2, categoryid);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_step (stmt);
                if (rc != SQLITE_DONE) {
                        g_warning ("Unable to rm mime '%s' (error %d)\n", rm_mimes[it], rc);
                }
                sqlite3_reset (stmt);
        }

        sqlite3_finalize (stmt);
}

void 
update_files_table (sqlite3 *filemeta, gint new_category_id, gchar **mimes)
{
        sqlite3_stmt *stmt;
        gint rc, it;
        const gchar *query = "UPDATE Services SET ServiceTypeID = ? WHERE Mime = ?";
        rc = sqlite3_prepare_v2 (filemeta,
                                 query,
                                 -1,
                                 &stmt,
                                 NULL);
        g_return_if_fail (rc == SQLITE_OK);

        for (it = 0; mimes[it] != NULL; it++) {
                rc = sqlite3_bind_int (stmt, 1, new_category_id);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_bind_text (stmt, 2, mimes[it], -1, SQLITE_STATIC);
                g_return_if_fail (rc == SQLITE_OK);

                rc = sqlite3_step (stmt);
                if (rc != SQLITE_DONE) {
                        g_warning ("Unable to update mime '%s' to category '%d' (error %d)\n", 
                                   rm_mimes[it], new_category_id, rc);
                }
                sqlite3_reset (stmt);
        }

        sqlite3_finalize (stmt);
}

void
update_tracker_dbs (const gchar *category, gchar **add_mimes, gchar **rm_mimes)
{
        sqlite3      *common, *file_meta;
        gchar        *path_common, *path_file_meta;
        gint          categoryId;
        gint          othersId;

        path_common = g_build_filename (G_DIR_SEPARATOR_S, "home", USER, 
                                        ".local", "share", "tracker", "data",
                                        "common.db", NULL);

        sqlite3_open_v2 (path_common, &common, SQLITE_OPEN_READWRITE, NULL);

        /* Get category ID */
        categoryId = get_category_id (common, category);
        if (categoryId == -1) {
                g_critical ("Category '%s' doesn't exist in the DB", category);
                return;
        }

        /* Update common.db :  */
        if (add_mimes) {
                update_mimes_table_add (common, categoryId, add_mimes);
        }

        if (rm_mimes) {
                update_mimes_table_rm (common, categoryId, rm_mimes);
        }

        othersId = get_category_id (common, "Other");
        g_return_if_fail (othersId != -1);

        sqlite3_close (common);
        g_free (path_common);

        /* Update category for all files in file-meta.db */

        path_file_meta = g_build_filename (G_DIR_SEPARATOR_S, 
                                           "home", USER, 
                                           ".cache", "tracker", 
                                           "file-meta.db", NULL);
        sqlite3_open_v2 (path_file_meta, &file_meta, SQLITE_OPEN_READWRITE, NULL);
        if (add_mimes) {
                update_files_table (file_meta, categoryId, add_mimes);
        }

        if (rm_mimes) {
                update_files_table (file_meta, othersId, rm_mimes);
        }
        sqlite3_close (file_meta);
        g_free (path_file_meta);
}


gint
main (gint argc, gchar **argv)
{
        GOptionContext *context;
        GKeyFile *services = NULL;
        gchar    *services_filename = NULL;
        gchar    *contents = NULL;
        GError   *error = NULL;
        gint      size;

        g_type_init ();

        context = g_option_context_new ("- Add or remove mimetypes from a category");

	g_option_context_add_main_entries (context, entries, NULL);
	g_option_context_parse (context, &argc, &argv, NULL);

        if (!category) {
		gchar *help;

		g_printerr ("Category is a mandatory parameter");
		g_printerr ("\n\n");

		help = g_option_context_get_help (context, TRUE, NULL);
		g_option_context_free (context);
		g_printerr ("%s", help);
		g_free (help);

		return EXIT_FAILURE;
        }

        if (!add_mimes && !rm_mimes) {
		gchar *help;

		g_printerr ("Missing mimetypes to add/remove");
		g_printerr ("\n\n");

		help = g_option_context_get_help (context, TRUE, NULL);
		g_option_context_free (context);
		g_printerr ("%s", help);
		g_free (help);

		return EXIT_FAILURE;
        }

        g_option_context_free (context);

        services_filename = g_build_filename (SHAREDIR, 
                                              "tracker", 
                                              "services", 
                                              "default.service", NULL);
        services = g_key_file_new ();

        if (!g_key_file_load_from_file (services, services_filename, G_KEY_FILE_NONE, &error)) {

                if (error != NULL) {
                        g_printerr ("%s\n", error->message);
                        g_error_free (error);
                } else {
                        g_printerr ("Error loading '%s'\n", services_filename);
                }
        
                g_key_file_free (services);
                g_free (services_filename);
                return EXIT_FAILURE;
        }

        if (!g_key_file_has_group (services, category)) {
                g_printerr ("Category '%s' not available in the configuration file\n", category);
                g_key_file_free (services);
                g_free (services_filename);
                return EXIT_FAILURE;
        }

        if (update_services_file (services, category, add_mimes, rm_mimes)) {
                contents = g_key_file_to_data (services, &size, &error);
                g_file_set_contents (services_filename, contents, size, &error);

                g_free (contents);
                if (error) {
                        g_key_file_free (services);
                        g_free (services_filename);
                        g_printerr ("Unable to save file (%s)\n", error->message);
                        return EXIT_FAILURE;
                }
        }

        g_key_file_free (services);
        g_free (services_filename);

        update_tracker_dbs (category, add_mimes, rm_mimes);

        return EXIT_SUCCESS;
}
