#include <pugixml/pugixml.hpp>


#include <ting/debug.hpp>


#include "Preferences.hpp"
#include "FSFile.hpp"
#include "utils.hpp"


const char* DThereminTag = "Theremin";
const char* DNumOctavesAttribute = "numOctaves";
const char* DTimbreAttribute = "timbre";
const char* DLockCurrentOctaveAttribute = "lockCurrentOctave";
const char* DBasePitchAttribute = "basePitch";

const char* DNetworkOptionsTag = "NetworkOptions";
const char* DUsernameAttribute = "username";
const char* DPasswordAttribute = "password";
const char* DAutologinAttribute = "autologin";
const char* DBroadcastAttribute = "broadcast";

const char* DKeyboardTag = "Keyboard";
const char* DScrPosAttribute = "scrPos";

const char* DAnglephoneTag = "Anglephone";


void Preferences::ClampBasePitch(){
	if(this->basePitch + 12 * this->numOctaves > (DHighestOctave() + 1) * 12){
		this->basePitch = (DHighestOctave() - (this->numOctaves)) * 12;
	}
}



void Preferences::LoadFromFile(){
//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): enter" << std::endl)

	file::FSFile f(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);

//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): reading file" << std::endl)

	pugi::xml_document doc;
	{
		ting::Array<ting::byte> rawFile;
		try{
			rawFile = f.LoadWholeFileIntoMemory();
		}catch(file::File::Exc &e){
			LOG(
					<< "Preferences::LoadFromFile(): could not load preferences file." << std::endl
					<< "\t" << " e = " << e.What() << std::endl
				)
			return;
		}

		doc.parse(reinterpret_cast<char*>(rawFile.Buf()));
	}

	//Load theremin mode settings
	{
		pugi::xml_node opts = doc.child(DThereminTag);
		if(!opts.empty()){
			pugi::xml_attribute numOctavesAttr = opts.attribute(DNumOctavesAttribute);
			if(!numOctavesAttr.empty()){
				ting::uint num = numOctavesAttr.as_uint();
				if(num != 0 && num <= 3)//TODO: check for that inside of this->SetNumOctaves()
					this->SetNumOctaves(num);
			}

			pugi::xml_attribute timbreAttr = opts.attribute(DTimbreAttribute);
			if(!timbreAttr.empty()){
				this->SetTimbre(TimbreNameToType(timbreAttr.value()));
			}

			pugi::xml_attribute lockCurOctaveAttr = opts.attribute(DLockCurrentOctaveAttribute);
			if(!timbreAttr.empty()){
				this->SetLockCurrentOctave(lockCurOctaveAttr.as_bool());
			}

			pugi::xml_attribute basePitchAttr = opts.attribute(DBasePitchAttribute);
			if(!basePitchAttr.empty()){
				this->SetBasePitch(basePitchAttr.as_uint());
			}
		}
	}

	//Load keyboard mode settings
	{
		pugi::xml_node kb = doc.child(DKeyboardTag);
		if(!kb.empty()){
			pugi::xml_attribute scrPosAttr = kb.attribute(DScrPosAttribute);
			if(!scrPosAttr.empty()){
				this->SetKbScrPos(scrPosAttr.as_uint());
			}

			pugi::xml_attribute timbreAttr = kb.attribute(DTimbreAttribute);
			if(!timbreAttr.empty()){
				this->SetKbTimbre(TimbreNameToType(timbreAttr.value()));
			}
		}
	}

	//Load anglephone mode settings
	{
		pugi::xml_node ang = doc.child(DAnglephoneTag);
		if(!ang.empty()){
			pugi::xml_attribute timbreAttr = ang.attribute(DTimbreAttribute);
			if(!timbreAttr.empty()){
				this->SetAngTimbre(TimbreNameToType(timbreAttr.value()));
			}
		}
	}

	//Read network settings
	{
		pugi::xml_node opts = doc.child(DNetworkOptionsTag);
		if(!opts.empty()){
			pugi::xml_attribute usernameAttr = opts.attribute(DUsernameAttribute);
			if(!usernameAttr.empty()){
				this->SetUsername(usernameAttr.value());
			}

			pugi::xml_attribute passwordAttr = opts.attribute(DPasswordAttribute);
			if(!passwordAttr.empty()){
				this->SetPassword(passwordAttr.value());
			}

			pugi::xml_attribute autologinAttr = opts.attribute(DAutologinAttribute);
			if(!autologinAttr.empty()){
				this->SetAutologin(autologinAttr.as_bool());
			}

			pugi::xml_attribute broadcastAttr = opts.attribute(DBroadcastAttribute);
			if(!broadcastAttr.empty()){
				this->SetBroadcast(broadcastAttr.as_bool());
			}
		}
	}//~read network settings
}



void Preferences::SaveToFile(){
	pugi::xml_document doc;

	{
		pugi::xml_node n = doc.append_child(pugi::node_declaration);
		n.set_name("xml");
		n.append_attribute("version").set_value("1.0");
	}

	//Save theremin mode settings
	{
		pugi::xml_node opts = doc.append_child();
		opts.set_name(DThereminTag);
		opts.append_attribute(DNumOctavesAttribute) = this->GetNumOctaves();
		opts.append_attribute(DTimbreAttribute).set_value(
				TimbreTypeToName(this->GetTimbre()).c_str()
			);
		opts.append_attribute(DLockCurrentOctaveAttribute) = this->GetLockCurrentOctave();
		opts.append_attribute(DBasePitchAttribute) = this->GetBasePitch();
	}

	//Save keyboard mode settings
	{
		pugi::xml_node kb = doc.append_child();
		kb.set_name(DKeyboardTag);
		kb.append_attribute(DScrPosAttribute) = this->GetKbScrPos();
		kb.append_attribute(DTimbreAttribute).set_value(
				TimbreTypeToName(this->GetKbTimbre()).c_str()
			);
	}

	//Save anglemode mode settings
	{
		pugi::xml_node ang = doc.append_child();
		ang.set_name(DAnglephoneTag);
		ang.append_attribute(DTimbreAttribute).set_value(
				TimbreTypeToName(this->GetAngTimbre()).c_str()
			);
	}

	//Save network settings
	{
		pugi::xml_node ns = doc.append_child();//create NetworkSettings node
		ns.set_name(DNetworkOptionsTag);
		
		ns.append_attribute(DUsernameAttribute) = ASS(this->GetUsername().c_str());
		ns.append_attribute(DPasswordAttribute) = ASS(this->GetPassword().c_str());
		ns.append_attribute(DAutologinAttribute) = this->GetAutologin();
		ns.append_attribute(DBroadcastAttribute) = this->GetBroadcast();
	}



	class PrefsWriter : public pugi::xml_writer{
		file::FSFile f;
	public:
		PrefsWriter(const std::string& fileName) :
				f(fileName)
		{
			f.Open(file::File::CREATE);
		}

		//override
		void write(const void *data, size_t size){
			ting::Array<ting::byte> buf(size);
			memcpy(buf.Buf(), data, size);
			f.Write(buf);
		}
	} writer(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);
	
    doc.print(writer);
}
