// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//	SQLite wrapper

#include "SQLite.hpp"



#include <ting/debug.hpp>
#include <ting/Exc.hpp>



using namespace sqlite;



Database::Database(const std::string& dbFilename) :
		numCursors(0)
{
	if(sqlite3_open(dbFilename.c_str(), &this->db) != SQLITE_OK){
		//NOTE: if this->db is 0 it is ok to pass it to sqlite3_close()
		sqlite3_close(this->db);
		throw ting::Exc("sqlite::Database::Database(): opening/creating db file failed");
	}
}



Database::~Database(){
	//check if all statements were finalized
	ASSERT_ALWAYS(this->numCursors == 0)

	//close database connection
	sqlite3_close(this->db);
}



Privates::Statement::Statement(sqlite3* database, const std::string& sqlStatement) :
		db(database)
{
	ASSERT(this->db)

	int res = sqlite3_prepare_v2(
			this->db,
			sqlStatement.c_str(),
			-1,
			&this->stmt,
			0
		);

	if(res == SQLITE_OK && this->stmt != 0){
		return;
	}

	if(res == SQLITE_ERROR)
		throw SQLExc(sqlite3_errmsg(this->db));

	throw ting::Exc("Database::Statement::Statement(): preparing failed");
}



Privates::Statement::~Statement(){
	this->Destroy();
}



void Privates::Statement::Destroy(){
	if(this->stmt){
		ASSERT(this->db)
		sqlite3_finalize(this->stmt);
	}else{
		ASSERT(!this->db)
	}
}



bool Privates::Statement::Step(){
	ASSERT(this->stmt)
	
	switch(sqlite3_step(this->stmt)){
		case SQLITE_ROW:
			return false;
			break;
		case SQLITE_DONE:
			return true;
			break;
		case SQLITE_ERROR:
			ASSERT(this->db)
			throw SQLExc(sqlite3_errmsg(this->db));
			break;
		default:
			throw ting::Exc("Database::Statement::Step(): sqlite3_step() failed");
			break;
	}
}



void Database::CreateTable(const std::string& name, const std::string& columns){
	ASSERT(this->db)

	Privates::Statement stmt(
			this->db,
			std::string("CREATE TABLE ") + name + "(" + columns + ")"
		);

	//normally, CREATE TABLE should be done in one step, so Step() should return true.
	if(!stmt.Step()){
		ASSERT(false)
	}
}



void Database::Insert(
		const std::string& table,
		const std::string& columns,
		const std::string& values
	)
{
	std::string s("INSERT INTO ");
	s += table;

	if(columns.size() > 0){
		s += "(" + columns + ")";
	}

	s += " VALUES (" + values + ")";

	Privates::Statement stmt(
			this->db,
			s
		);

	//normally, INSERT should be done in one step, so Step() should return true.
	if(!stmt.Step()){
		ASSERT(false)
	}
}



void Database::Update(
		const std::string& table,
		const std::string& what,
		const std::string& inRowsWhere
	)
{
	std::string s("UPDATE ");
	s += table;
	s += " SET " + what;
	s += " WHERE " + inRowsWhere;

	Privates::Statement stmt(
			this->db,
			s
		);

	//normally, UPDATE should be done in one step, so Step() should return true.
	if(!stmt.Step()){
		ASSERT(false)
	}
}



void Database::Delete(
		const std::string& fromTable,
		const std::string& rowsWhere
	)
{
	std::string s("DELETE FROM ");
	s += fromTable;
	s += " WHERE " + rowsWhere;

	Privates::Statement stmt(
		this->db,
		s
	);

	//normally, DELETE should be done in one step, so Step() should return true.
	if(!stmt.Step()){
		ASSERT(false)
	}
}



Cursor Database::Query(
		const std::string& table,
		const std::string& columns,
		const std::string& where
	)
{
	std::string s("SELECT ");
	s += columns;
	s += " FROM " + table;

	if(where.size() > 0){
		s += " WHERE " + where;
	}
	
	return Cursor(this, s);
}



Cursor::Cursor(Database* database, const std::string& sqlQuery) :
		db(database),
		statement(ASS(database->db), sqlQuery)
{
	ASSERT(this->db)

	{
		int cols = sqlite3_column_count(this->statement.stmt);
		ASSERT(cols >= 0)
		this->numColumns = ting::uint(cols);
	}

	++this->db->numCursors;
}



Cursor::~Cursor(){
	this->Destroy();
}



void Cursor::Destroy(){
	if(this->db){
		--this->db->numCursors;
	}
}



/**
 * @brief Move to next row.
 * @return true - if moved to next row,
 *         false - if moved after last row (end of rows list reached).
 *                 If this method is called again after it returned false it
 *                 will throw an exception.
 */
bool Cursor::MoveToNext(){
	if(!this->statement.stmt)
		throw ting::Exc("sqlite::Statement::Step(): statement object is invalid");

	return !this->statement.Step();
}



ting::uint Cursor::FindColumnByName(const std::string& column){
	ASSERT(this->statement.stmt)//assert this is valid cursor

	for(ting::uint i = 0; i < this->numColumns; ++i){
		const char* name = sqlite3_column_origin_name(
				this->statement.stmt,
				int(i)
			);

		//sqlite3_column_origin_name() returns 0 if it cannot allocate memory
		if(name == 0)
			throw std::bad_alloc();

		if(column.compare(name) == 0){
			return i;
		}
	}

	throw SQLExc("Cursor::FindColumnByName(): column not found");
}



std::string Cursor::GetText(ting::uint column){
	ASSERT(this->statement.stmt)
	ASSERT(column < this->numColumns)
	
	return std::string(
			reinterpret_cast<const char*>(
					sqlite3_column_text(
							this->statement.stmt,
							int(column)
						)
				)
		);
}



int Cursor::GetInt(ting::uint column){
	ASSERT(this->statement.stmt)
	ASSERT(column < this->numColumns)

	return sqlite3_column_int(this->statement.stmt, int(column));
}



ting::s64 Cursor::GetInt64(ting::uint column){
	ASSERT(this->statement.stmt)
	ASSERT(column < this->numColumns)

	return sqlite3_column_int64(this->statement.stmt, int(column));
}



double Cursor::GetDouble(ting::uint column){
	ASSERT(this->statement.stmt)
	ASSERT(column < this->numColumns)

	return sqlite3_column_double(this->statement.stmt, int(column));
}



ting::Array<ting::byte> Cursor::GetBlob(ting::uint column){
	ASSERT(this->statement.db)
	ASSERT(this->statement.stmt)
	ASSERT(column < this->numColumns)

	//NOTE: Call sqlite3_column_blob() and then sqlite3_column_bytes() exactly
	//in that order as sqlite3 documentation suggests.
	const void* blob = sqlite3_column_blob(this->statement.stmt, int(column));
	int numBytes = sqlite3_column_bytes(this->statement.stmt, int(column));

	if(numBytes == 0 || blob == 0){
		if(sqlite3_errcode(this->statement.db) == SQLITE_NOMEM){
			throw std::bad_alloc();
		}

		return ting::Array<ting::byte>();//return empty array
	}

	ASSERT(blob)
	ASSERT(numBytes > 0)

	//alloc array
	ting::Array<ting::byte> ret( (ting::uint(numBytes)) );//double parentheses to prevent compiler to treat this line as function declaration

	//copy data to array
	memcpy(ret.Buf(), blob, ret.SizeInBytes());

	return ret;
}
