/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

#include <gtkmm/container.h>
#include <gtkmm/box.h>
#include <glibmm-2.4/glibmm/dispatcher.h>

#include <ting/Singleton.hpp>
#include <ting/debug.hpp>

#include "stdafx.hpp"
#include "ListenWindow.hpp"
#include "MainWindow.hpp"
#include "MainThread.hpp"
#include "utils.hpp"



using namespace ting;



ListenWindow::ListenWindow() :
		statusLabel(DStrOffline(), Gtk::ALIGN_LEFT),
		playerNameLabel("", Gtk::ALIGN_LEFT),
		nextButton("next")
{
	//set CAN_FOCUS flag to be able to receive keyboard events
	this->set_flags(Gtk::CAN_FOCUS);

	this->signal_delete_event().connect(
			sigc::mem_fun(*this, &ListenWindow::OnDeleteEvent)
		);

	Gtk::VBox *vbox = new Gtk::VBox();
	this->add(*Gtk::manage(vbox));

	{
		Gtk::HBox *h = new Gtk::HBox();
		vbox->add(*Gtk::manage(h));

		h->set_homogeneous(false);

		h->add(*Gtk::manage(new Gtk::Label("Network status: ", Gtk::ALIGN_RIGHT)));

		h->add(this->statusLabel);
	}

	{
		Gtk::HBox *h = new Gtk::HBox();
		vbox->add(*Gtk::manage(h));

		h->set_homogeneous(false);

		h->add(*Gtk::manage(new Gtk::Label("Player name: ", Gtk::ALIGN_RIGHT)));

		h->add(this->playerNameLabel);
	}

	{
		Gtk::HBox *hbox = Gtk::manage(new Gtk::HBox());
		vbox->add(*hbox);

		this->nextButton.signal_clicked().connect(
				sigc::mem_fun(*this, &ListenWindow::OnNextButtonClicked)
			);
		hbox->add(this->nextButton);
	}

	//connect dispatcher
	{
		Ptr<Signal0> sig(new Signal0());
		sig->Connect(this, &ListenWindow::NetworkStatusDispatcherHandler);
		this->networkStatusDispatcher = GlibDispatcher::Inst().Connect(sig);
	}
	//subscribe to network state notifications
	MainThread::Inst().AddNetworkStateListener_ts(this);


	//connect dispatcher
	{
		Ptr<Signal0> sig(new Signal0());
		sig->Connect(this, &ListenWindow::CurPlayerDispatcherHandler);
		this->curPlayerDispatcher = GlibDispatcher::Inst().Connect(sig);
	}
	//subscribe to current player notifications
	MainThread::Inst().AddCurrentPlayerListener_ts(this);

	this->show_all_children();
}



ListenWindow::~ListenWindow(){
//		TRACE(<< "~ListenWindow(): invoked" << std::endl)
	//After removing the listener we can be sure
	//that no notification will occur.
	MainThread::Inst().RemoveNetworkStateListener_ts(this);
	//disconnect dispatcher
	GlibDispatcher::Inst().Disconnect(this->networkStatusDispatcher);
	
	
	MainThread::Inst().RemoveCurrentPlayerListener_ts(this);
	//disconnect dispatcher
	GlibDispatcher::Inst().Disconnect(this->curPlayerDispatcher);
}



bool ListenWindow::OnDeleteEvent(GdkEventAny* event){
	//Stop listening on window close.
	MainThread::Inst().StopListening_ts();

	//Return false to call default on_delete_event handler.
	return false;
}



void ListenWindow::OnNextButtonClicked(){
	MainThread::Inst().ListenNext_ts();
}



void ListenWindow::NetworkStatusDispatcherHandler(){
	MainThread::ENetworkState st;
	{
		ting::Mutex::Guard mutexGuard(this->dispatcherArgMutex);
		ASSERT(this->dispatcherArg.size() > 0)
		st = this->dispatcherArg.front();
		this->dispatcherArg.pop_front();
	}
	switch(st){
		default:
		case MainThread::OFFLINE_LOGIN_FAILED:
		case MainThread::OFFLINE:
			this->statusLabel.set_text(DStrOffline());
			break;
		case MainThread::CONNECTING:
			this->statusLabel.set_text(DStrConnecting());
			break;
		case MainThread::CONNECTED:
			this->statusLabel.set_text(DStrConnected());
			break;
		case MainThread::LOGGING_IN:
			this->statusLabel.set_text(DStrLoggingIn());
			break;
		case MainThread::LOGGED_IN:
			this->statusLabel.set_text(DStrLoggedIn());
			break;
	}

}



//override
void ListenWindow::OnStateChange(MainThread::ENetworkState newState){
	{
		ting::Mutex::Guard mutexGuard(this->dispatcherArgMutex);
		this->dispatcherArg.push_back(newState);
	}
	GlibDispatcher::Inst().Emit(this->networkStatusDispatcher);
}



void ListenWindow::CurPlayerDispatcherHandler(){
	std::string plr;
	{
		ting::Mutex::Guard mutexGuard(this->curPlayerDispatcherArgMutex);
		ASSERT(this->curPlayerDispatcherArg.size() > 0)
		plr = this->curPlayerDispatcherArg.front();
		this->curPlayerDispatcherArg.pop_front();
	}

	if(plr.size() > 0){
		this->playerNameLabel.set_text(plr);
	}else{
		this->playerNameLabel.set_text("No one is playing currently");
	}
}



//override
void ListenWindow::CurrentPlayer(const std::string& playerUsername){
	{
		ting::Mutex::Guard mutexGuard(this->curPlayerDispatcherArgMutex);
		this->curPlayerDispatcherArg.push_back(
				std::string(playerUsername.c_str()) //copy string
			);
	}
	GlibDispatcher::Inst().Emit(this->curPlayerDispatcher);
}



//ListenWindow::ListenWindow(){
//	this->add(this->listenView);
//	this->show_all_children();//Show all window child widgets
//}
