#include "dialogs.hpp"
#include <QVBoxLayout>
#include <hildon-extras-1/hildon-extras/qt-he-wrapper.h>

static QPointer<Dialogs> _instance = NULL;
static QPointer<StickyNotesWidget> _currentNote;

static QPointer<QDialog> _quickOptionsDialog;
static QPointer<QPushButton> _backgroundColorButton;
static QPointer<QPushButton> _foregroundColorButton;
static QPointer<QPushButton> _fontButton;
static QPointer<QSlider> _alphaSlider;

Dialogs::Dialogs(QObject *parent)
    : QObject(parent)
{
}

Dialogs *Dialogs::instance()
{
    if (_instance == NULL)
        _instance = new Dialogs(QCoreApplication::instance());

    return _instance;
}

void Dialogs::showQuickOptionsDialog()
{
    static bool isShown = false;

    if (isShown)
        return;

    if (_quickOptionsDialog == NULL)
    {
        _quickOptionsDialog = new QDialog;
        _quickOptionsDialog->setWindowTitle("Quick options");
        _backgroundColorButton = new QPushButton("Change background colour");
        _foregroundColorButton = new QPushButton("Change foreground colour");
        _fontButton = new QPushButton("Change font");
        _alphaSlider = new QSlider(Qt::Horizontal);
        _alphaSlider->setMinimum(0);
        _alphaSlider->setMaximum(255);
        QVBoxLayout *layout = new QVBoxLayout;
        layout->addWidget(_backgroundColorButton);
        layout->addWidget(_foregroundColorButton);
        layout->addWidget(_fontButton);
        QHBoxLayout *hLayout = new QHBoxLayout;
        hLayout->addWidget(new QLabel("Opacity"));
        hLayout->addWidget(_alphaSlider);
        layout->addLayout(hLayout);
        _quickOptionsDialog->setLayout(layout);

        connect(_backgroundColorButton, SIGNAL(clicked()), this, SLOT(showBackgroundColorSelection()));
        connect(_foregroundColorButton, SIGNAL(clicked()), this, SLOT(showForegroundColorSelection()));
        connect(_fontButton, SIGNAL(clicked()), this, SLOT(showFontSelection()));
    }

    if (QObject::sender()->inherits("StickyNotesWidget"))
    {
        isShown = true;
        _currentNote = (StickyNotesWidget*)QObject::sender();

        if (_currentNote->isReadOnly())
        {
            _foregroundColorButton->hide();
            _fontButton->hide();
        }
        else
        {
            _foregroundColorButton->show();
            _fontButton->show();
        }

        int alpha = _currentNote->backgroundColor().alpha();

        if (_alphaSlider->value() != alpha)
            _alphaSlider->setValue(alpha);

        _quickOptionsDialog->exec();

        if (_alphaSlider->value() != alpha)
            _currentNote->setAlpha(_alphaSlider->value());

        isShown = false;
    }
}

void Dialogs::showBackgroundColorSelection()
{
    static bool isShown = false;

    if (!isShown)
    {
        isShown = true;
        QColor newColor = QtHeWrapper::showHeSimpleColorDialog(NULL, _currentNote->backgroundColor(), true, "Pick a background color for this note");
        newColor.setAlpha(_currentNote->backgroundColor().alpha());
        _currentNote->setBackgroundColor(newColor);
        isShown = false;
    }
}

void Dialogs::showForegroundColorSelection()
{
    static bool isShown = false;

    if (!isShown)
    {
        isShown = true;
        QColor newColor = QtHeWrapper::showHeSimpleColorDialog(NULL, _currentNote->foregroundColor(), true, "Pick a foreground color for this note");
        _currentNote->setForegroundColor(newColor);
        isShown = false;
    }
}

void Dialogs::showFontSelection()
{
    static bool isShown = false;

    if (!isShown)
    {
        isShown = true;
        QFont newFont = QtHeWrapper::showHeFontDialog(NULL, _currentNote->font(), "Pick a font for this note");
        _currentNote->setFont(newFont);
        isShown = false;
    }
}

void Dialogs::showAboutDialog()
{
    QtHeWrapper::showHeAboutDialog(NULL,
                                   "The easiest-to-use note widget for\nyour mobile computer.",
                                   "Timur Kristf (C) 2010, Licensed under EUPL v1.1",
                                   "http://sticky-notes.garage.maemo.org/",
                                   "http://talk.maemo.org/showthread.php?t=57049",
                                   "https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=venemo%40msn%2ecom&lc=HU&item_name=Sticky%20Notes%20development%20for%20Maemo&currency_code=EUR&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted",
                                   "sticky-notes");
}
