
#include "squeezster.h"
#include "ui_squeezster.h"
#include "ui_connectionDialog.h"
#include "ui_optionsdialog.h"
#include "ui_libraryviewdialog.h"
#include "squeezster_ui_definitions.h"

//----------------------------------------------
//----------------------------------------------
// squeezster(QWidget *parent)
// Main window
//----------------------------------------------
//----------------------------------------------
squeezster::squeezster(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::squeezster),
    networkSession(0)
{
    ui->setupUi(this);


#if defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif

    progressTimer.setSingleShot(false);
    progressTimer.setInterval(PROGRESSINTERVAL);
    itemSelectTimer.setSingleShot(true);
    itemSelectTimer.setInterval(ITEMSELECTTIME);

    tcpSocket = new QTcpSocket(this);
    inTS = new QTextStream();
    outTS = new QTextStream();
    myResponse = new squeezeResponse(tcpSocket);

    mySqueezeEnv.initDone=false;
    pixmapBusy = false;
    connectTries = 0;

    connect(tcpSocket, SIGNAL(error(QAbstractSocket::SocketError)),
            this, SLOT(displayError(QAbstractSocket::SocketError)));
    connect(&manager,SIGNAL(onlineStateChanged(bool)),this,SLOT(startApp(bool)));
    connect(myResponse, SIGNAL(responseReceived(QList<QByteArray>)),this, SLOT(responseListHandler(QList<QByteArray>)));
    connect(tcpSocket, SIGNAL(readyRead()), myResponse, SLOT(readResponse()));

    connect(this, SIGNAL(playerStatusUpdated(squeezePlayer*)),this, SLOT(updateUI(squeezePlayer*)));
    //connect(this, SIGNAL(activePlayerUpdated(squeezePlayer*)),this, SLOT(updateUI(squeezePlayer*)));
    connect(&progressTimer,SIGNAL(timeout()),this,SLOT(updateProgress()));
    connect(&itemSelectTimer,SIGNAL(timeout()),this,SLOT(itemUnselect()));

    // Find servers. Result currently not used.
    serverDiscovery();

    // Load connection settings and user credentials
    readSettings(settings);
    qDebug() << "readSettings - screensaverInhibit" << settings.screensaverIsInhibited();
    // Inhibit screensaver
    if(settings.screensaverIsInhibited()) {
        screenSaverInhibiter = new QSystemScreenSaver();
        screenSaverInhibiter->setScreenSaverInhibit();
    }

    initUI();

    if (manager.capabilities() & QNetworkConfigurationManager::NetworkSessionRequired) {
    }
    // Initialize networkmanager and simple cache
    networkManager = new QNetworkAccessManager(this);
    //connect(networkManager,SIGNAL(networkSessionConnected()),this,SLOT(startApp()));
    connect(networkManager,SIGNAL(finished(QNetworkReply*)), this, SLOT(pixmapDownloaded(QNetworkReply*)));
    connect(networkManager,SIGNAL(authenticationRequired(QNetworkReply*,QAuthenticator*)),this,SLOT(httpAuthentSlot(QNetworkReply*, QAuthenticator*)));

    //diskCache = new QNetworkDiskCache(this);
    //diskCache->setCacheDirectory("cacheDir");
    //diskCache->setMaximumCacheSize(DISKCACHESIZE);
    //networkManager->setCache(diskCache);


    startApp(manager.isOnline());
    //    }
    //    else
    //        qDebug() << "networkmanager NetworkSessionRequired false";

}


//----------------------------------------------
// squeezster::startApp()
// Connect to server and init environment
//----------------------------------------------
void squeezster::startApp(bool isOnline) {
    qDebug() << "startApp() is onLine:" << isOnline;
    if (isOnline) {
        //&& (networkManager->networkAccessible()==QNetworkAccessManager::Accessible)) {
        requestConnect();
        if (!mySqueezeEnv.initDone)
            initEnvironment();
    } else {
        QMessageBox::information(this, tr("squeezster"),
                                 tr("The device is not online. Please connect the device and start Squeezster again."));
        exit(0);
    }
}


//----------------------------------------------
// squeezster::requestConnect()
// Update connection if settings have changed.
//----------------------------------------------
void squeezster::httpAuthentSlot(QNetworkReply* networkReply, QAuthenticator* authenticator)
{
    qDebug() << "httpAuthentSlot(QNetworkReply* networkReply, QAuthenticator* authenticator)";
    authenticator->setUser(settings.username());
    authenticator->setPassword(settings.password());
}


//----------------------------------------------
// squeezster::requestConnect()
// Update connection if settings have changed.
//----------------------------------------------
void squeezster::requestConnect(const UserSettings &newSettings)
{
    qDebug() << "squeezster::requestConnect(const UserSettings &newSettings)";
    //bool inhibited = settings.screensaverIsInhibited();
    //newSettings.setScreensaverIsInhibited(inhibited);
    if(settings == newSettings)
        qDebug() << "UserSettings: no change";
    else {
        settings = newSettings;
        writeSettings(settings);
    }
    startApp();
}


//----------------------------------------------
// squeezster::requestConnect()
// Connect to server
//----------------------------------------------

bool squeezster::requestConnect()
{
    bool success=false;
    qDebug() << "requestConnect()";
    tcpSocket->blockSignals(true);
    if(tcpSocket->isOpen()) {
        tcpSocket->abort();
        qDebug() << "Closing and reopening tcpSocket";
        tcpSocket->waitForDisconnected(5000);
    }
    do {
        squeezster::tcpSocket->connectToHost(settings.ipAddress(),settings.cliPort(),QIODevice::ReadWrite);
        if (squeezster::tcpSocket->waitForConnected(5000)) {
            qDebug() << "requestConnect() tcpSocket open";
            squeezster::tcpSocket->setTextModeEnabled(true);
            inTS->setDevice(squeezster::tcpSocket);
            inTS->setCodec("UTF-8");
            outTS->setDevice(squeezster::tcpSocket);
            outTS->setCodec("UTF-8");
            myResponse->setTcpSocket(tcpSocket);
            myResponse->setTextStream(inTS);
            qDebug() << "requestConnect() OK" << settings.ipAddress() <<
                        QString::number(settings.cliPort()) << QString::number(settings.httpPort());
            success = true;
            inTS->readAll(); // Empty buffer
            // Send login command
            QStringList loginCommand;
            loginCommand << "login" << settings.username() << settings.password();
            send(loginCommand,true);
            tcpSocket->waitForReadyRead(5000);
            QString resp=inTS->readAll();
            if (resp.contains("wrongpassword")) {
                qDebug() << "Wrong password";
                squeezster::constructConnDialog();
                success = false;}
            qDebug() << "RequestConnect() successful. resp:" << resp;
            tcpSocket->blockSignals(false);
            return(true);
        } else {
            qDebug() << "requestConnect() FAIL";
            qDebug() << "Error could not connect to " << settings.ipAddress() << ":" << settings.cliPort();
            if(connectTries++<MAXCONNECTTRIES)
                squeezster::constructConnDialog();
            else {
                qDebug() << "Could not connect.";
                exit(99);
            }
            tcpSocket->blockSignals(false);
            return(false);
        }
    } while (!success);
}


//----------------------------------------------
// squeezster::initEnvironment()
// Init server connection,
// find players & request basic data.
//----------------------------------------------

void squeezster::initEnvironment()
{
    qDebug() << "initEnvironment()";
    qint8 i,c;
    QString localResponse;
    QStringList stringList;
    squeezePlayer *newPlayer;


    tcpSocket->blockSignals(true);

    // Collect commands not to be parsed as "xmlbrowser"
    responseWhitelist["status"] = true;
    responseWhitelist["favorites"] = true;
    responseWhitelist["favorites items"] = true;
    responseWhitelist["albums"] = true;
    responseWhitelist["artists"] = true;
    responseWhitelist["titles"] = true;
    responseWhitelist["artists"] = true;
    responseWhitelist["radios"] = true;
    responseWhitelist["apps"] = true;
    responseWhitelist["musicfolder"] = true;

    responseBlacklist["power"] = true;

    // Empty read buffer
    inTS->readAll();

    // Tell the server to send updates to us
    stringList << "listen" << "1";
    send(stringList,true);
    tcpSocket->waitForReadyRead(5000); localResponse=inTS->readAll();
    qDebug() << "Check listen:" << localResponse;

    // Ask the server about the number of players
    stringList.clear();
    stringList << "players" << "0";
    send(stringList,true);
    tcpSocket->waitForReadyRead(5000); localResponse=inTS->readLine();
    qDebug() << "Check count:" << localResponse;
    localResponse=QString(getInitValue(localResponse,"count"));
    if (!localResponse.isEmpty())
        mySqueezeEnv.numberOfPlayers = localResponse.toInt();
    else // Unexpected answer from the server
    { qDebug() << "Couldn't initialize Squeeze environment. Please check connection settings.";
        QMessageBox::warning(this, tr("Squeezster"),
                             tr("Couldn't initialize Squeeze environment. Please check connection settings."));

        return;
    }

    // Ask the server for the player details
    for (i=0; i<mySqueezeEnv.numberOfPlayers; i++)
    {
        newPlayer = new squeezePlayer();
        c = i+48;
        stringList.clear();
        stringList << "players" << QStringList(QString(char(c)))<<"1";
        send(stringList,true);
        tcpSocket->waitForReadyRead(5000); localResponse=inTS->readLine();
        newPlayer->setPlayerName(QString(getInitValue(localResponse,"name")));
        newPlayer->setPlayerID(QString(getInitValue(localResponse,"playerid")));
        newPlayer->setPlayerIP(QString(getInitValue(localResponse,"ip")));
        newPlayer->setPlayerModel(QString(getInitValue(localResponse,"model")));
        qDebug() << "initEnvironment() playerID:" << newPlayer->playerID;
        mySqueezeEnv.addPlayer(newPlayer);
        ui->playerComboBox->insertItem(i+1,newPlayer->playerName);
        queryStatus(newPlayer);
        responseListHandler(myResponse->readResponseWait(),true);
    }

    // Select player #0, and end the init part
    mySqueezeEnv.activePlayer = mySqueezeEnv.playerList.at(0);
    mySqueezeEnv.activePlayerSet=true;
    slaveMasterConnect();
    mySqueezeEnv.initDone=true;
    tcpSocket->blockSignals(false);
    updateActivePlayer(0);
    emit(initDone());

    qDebug()<< "********************************";
    qDebug()<< "**         initDone           **";
    qDebug()<< "********************************";

    // Request interesting lists from the server
    requestSqueezeList("status");
    qDebug() << "Request Favorites";
    QStringList favCmd;
    favCmd << "favorites" << "items";
    requestSqueezeList(favCmd);
    requestSqueezeList("radios");
    requestSqueezeList("apps");
    requestSqueezeList("albums",MAXITEMSPERREQUEST);
    requestSqueezeList("playlists");
    requestSqueezeList("musicfolder");
    progressTimer.start();
}


//----------------------------------------------
// squeezster::responseListHandler()
// Called when new server message has arrived
// Splits the response into individual messages
// and calls the responseHandler()
//----------------------------------------------
void squeezster::responseListHandler(QList<QByteArray> responseByteArrayList, bool init)
{
    QString responseString;
    qDebug() << "responseListHandler()";
    if (mySqueezeEnv.initDone || init) {

        while (!responseByteArrayList.isEmpty()) {
            responseString = QString(responseByteArrayList.takeFirst());
            responseHandler(responseString,init);
        }
    }
}


//----------------------------------------------
// squeezster::responseHandler()
// Delegate the server messaged to the correct
// function.
//----------------------------------------------
QString squeezster::responseHandler(QString responseString, bool init)
{
    qDebug() << "responseHandler()";// << responseString << mySqueezeEnv.initDone << init;
    QStringList subResponses;
    QListIterator<squeezePlayer*> players(mySqueezeEnv.playerList);
    squeezePlayer* player=0;
    QString command, tmpString;
    bool playerFound;

    playerFound = false;
    if((mySqueezeEnv.initDone || init) && !(responseString.isEmpty()))
    {
        subResponses=responseString.split(QRegExp("\n|\x20"),QString::SkipEmptyParts);
        QMutableListIterator<QString> i(subResponses);
        while (i.hasNext()) {
            tmpString=i.next();
            i.setValue(QUrl::fromPercentEncoding(tmpString.toUtf8()));
        }
        i.toBack();

        while (players.hasNext() && !playerFound) {
            player = players.next();
            if (QString(subResponses.first()) == player->playerID) {
                player = players.peekPrevious();
                //qDebug() << "Found player: " << QString(player->playerName);
                playerFound = true;
                subResponses.removeFirst();
            }
        }

        if (!playerFound) {
            // No player found in the response
            return(QString::null); }

        // Interpret command
        command = subResponses.takeFirst();
        qDebug() << "responseHandler: command" << command;
        if (command.toLower() == "status") {
            qDebug() << "responseHandler: status";
            updateStatus(player, &subResponses);
        }
        else if((command.toLower() == "radios") || (command.toLower() == "apps")) {
            qDebug() << "responseHandler: radios/apps";
            updateRadiosApps(command,&subResponses);
        }
        else if (command.toLower() == "albums") {
            qDebug() << "responseHandler: albums";
            updateXmlList(command,ui->albumsListWidget,&subResponses,"id",squeezster::updateIcons);
        }
        else if (command.toLower() == "playlists") {
            qDebug() << "responseHandler: playlists";
            updateXmlList(command,ui->playlistsListWidget,&subResponses,"id");
        }
        else if (command.toLower() == "titles") {
            qDebug() << "responseHandler: titles";
            updateXmlBrowserList(command,&subResponses);
        }
        else if (command.toLower() == "artists") {
            qDebug() << "responseHandler: artists";
            updateXmlBrowserList(command,&subResponses);
        }
        else if (command.toLower() == "musicfolder") {
            qDebug() << "responseHandler: musicfolder";
            updateXmlBrowserList(command,&subResponses,"id",ui->folderTreeWidget);
        }
        //---------------------------------------------
        // Cmd: favorites
        //---------------------------------------------
        else if (command.toLower() == "favorites") {
            qDebug() << "responseHandler: favorites";
            updateXmlBrowserList(command,&subResponses,"id",ui->favoritesTreeWidget);
        }
        //---------------------------------------------
        // Cmd: prefset
        //---------------------------------------------
        else if (command.toLower() == "prefset")
            updateMixer(player, &subResponses);
        //---------------------------------------------
        // Cmd: power
        //---------------------------------------------
        else if (command.toLower() == "power")
            queryStatus(player);
        //---------------------------------------------
        // Cmd: time
        //---------------------------------------------
        else if (command.toLower() == "time")
            queryStatus(player);
        //---------------------------------------------
        // Cmd: playlist
        //---------------------------------------------
        else if (command.toLower() == "playlist") {
            command=subResponses.takeFirst();

            if ((command.toLower()=="loadtracks") || (command.toLower()=="addtracks") ||
                    (command.toLower()=="inserttracks") || (command.toLower()=="deletetracks") ||
                    (command.toLower()=="load_done") || (command.toLower()=="delete") ||
                    (command.toLower()=="clear")) {

                queryStatus(player);
                requestSqueezeList("status");
            }

            if (command.toLower()=="newsong")
                queryStatus(player);
        }
        //---------------------------------------------
        // Others commands
        //---------------------------------------------
        else if (!responseBlacklist[command] && (responseWhitelist[command] || squeezster::treeItemParent.isValid())) {
            qDebug() << "responseHandler: xmlBrowserList" << command;
            updateXmlBrowserList(command,&subResponses); }
        else
            qDebug() << "responseHandler(): Doing nothing" << command;
    }

    return("");
}


//-----------------------------------------------------------------------------------
// squeezster::updateStatus:
// Reads a status message and updates the status of the players
//-----------------------------------------------------------------------------------
void squeezster::updateStatus(squeezePlayer *player, QStringList *parameters) {
    qDebug() << "updateStatus()"; // << parameters;
    QString keyVal,parameter;
    bool emitUiUpdate=mySqueezeEnv.playerChanged;
    bool songChanged=false;
    bool emitPlaylistUpdate=false;
    bool ok=false;
    quint16 pStartIndex=0;
    quint16 pEndIndex=0;

    player->isRemote=false;
    player->updateAlbumArt=false;
    mySqueezeEnv.playerChanged=false;

    // Check if the message is a status list, i.e. the current playlist.
    if(responseIsList(parameters, &pStartIndex, &pEndIndex)!=0) {
        // Update current playlist
        qDebug() << "Update current playlist." <<pStartIndex << pEndIndex;
        qDebug() << player->playerName << *parameters;
        // Make this list generic by adding count at the end:
        while(!parameters->isEmpty() && !ok)
        {
            parameter= parameters->takeFirst();
            QString(keyVal=getKeyValue(parameter,"playlist_tracks",&ok));
            if (ok)
                parameters->append(QString("count:"+keyVal));
        }
        parameters->prepend(QString::number(pEndIndex));
        parameters->prepend(QString::number(pStartIndex));
        //----------------------------------------------
        // Analyze and insert playlist
        //----------------------------------------------
        updateXmlList("status",ui->nowPlayingListWidget,parameters,"playlist index");

    } else { // The message contains the player status
        qDebug() << "Update player status";
        bool isActive = (player==mySqueezeEnv.activePlayer);
        while(!parameters->isEmpty())
        {
            parameter=parameters->takeFirst();
            keyVal=getKeyValue(parameter,"id",&ok);
            if(ok) {
                if(keyVal!=player->currentSongID && isActive) {
                    emitUiUpdate = true;
                    songChanged=true;
                }
                player->setCurrentSongID(keyVal);
                qDebug() << "updateStatus()" << "songID: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"title",&ok);
            if(ok) {
                player->setCurrentTitle(keyVal);
                qDebug() << "updateStatus()" << "title: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"artist",&ok);
            if(ok) {
                player->setCurrentArtist(keyVal);
                qDebug() << "updateStatus()" << "artist: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"album",&ok);
            if(ok) {
                qDebug() << "updateStatus()" << "album: " << keyVal;
                if(player->currentAlbum.isEmpty())
                    player->updateAlbumArt=true;
                if(keyVal!=player->currentAlbum && isActive) {
                    player->updateAlbumArt=true;}
                player->setCurrentAlbum(keyVal);
            }

            keyVal=getKeyValue(parameter,"duration",&ok);
            if(ok) {
                if(keyVal.toFloat()<1)
                    keyVal="3600";
                player->setCurrentDuration(keyVal.toFloat());
                if(isActive)
                    ui->progressBar->setMaximum((player->currentDuration?player->currentDuration:3600));
                qDebug() << "updateStatus()" << "duration: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"time",&ok);
            if(ok) {
                player->setCurrentTime(keyVal.toFloat());
                ui->progressBar->setValue(player->currentTime);
                qDebug() << "updateStatus()" << "time: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"genre",&ok);
            if(ok) {
                player->setCurrentGenre(keyVal);
                qDebug() << "updateStatus()" << "genre: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"mode",&ok);
            if(ok) {
                if((keyVal!=player->currentMode) && isActive)
                    emitUiUpdate = true;
                player->setCurrentMode(keyVal);
                qDebug() << "updateStatus()" << "mode: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"mixer volume",&ok);
            if(ok) {
                player->setCurrentVolume(keyVal.toInt());
                qDebug() << "updateStatus()" << "volume: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"sync_master",&ok);
            if(ok) {
                player->masterPlayerID=keyVal;
                qDebug() << "updateStatus()" << "set master: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"sync_slaves",&ok);
            if(ok) {
                player->slavePlayerID << keyVal;
                qDebug() << "updateStatus()" << "set slaves: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"playlist repeat",&ok);
            if(ok) {
                player->playlistRepeat=keyVal;
                if(keyVal==QString("0"))
                    ui->actionRepeat->setText("Repeat: Turn ON");
                else
                    ui->actionRepeat->setText("Repeat: Turn OFF");
            }

            keyVal=getKeyValue(parameter,"playlist shuffle",&ok);
            if(ok) {
                player->playlistShuffle=keyVal;
                if(keyVal==QString("0"))
                    ui->actionShuffle->setText("Shuffle: Turn ON");
                else
                    ui->actionShuffle->setText("Shuffle: Turn OFF");
            }

            keyVal=getKeyValue(parameter,"playlist mode",&ok);
            if(ok) {
                player->playlistMode=keyVal;
                qDebug() << "updateStatus()" << "playlist mode: " << keyVal;
            }

            keyVal=getKeyValue(parameter,"playlist_tracks",&ok);
            if(ok) {
                if(player->playlistTracks!=keyVal.toInt())
                    emitPlaylistUpdate=true;
                player->playlistTracks=keyVal.toInt();
            }

            keyVal=getKeyValue(parameter,"playlist_cur_index",&ok);
            if(ok) {
                if(keyVal.toInt()!=player->playlistIndex) {
                    emitUiUpdate=true;
                    player->prevPlaylistIndex=player->playlistIndex;
                }
                player->playlistIndex=keyVal.toInt();
            }

            keyVal=getKeyValue(parameter,"remote",&ok);
            if(ok) {
                player->isRemote=(keyVal=="1"?true:false);
            }

            keyVal=getKeyValue(parameter,"power",&ok);
            if(ok) {

                if(player->power!=(keyVal=="1"?true:false))
                    emitUiUpdate = true;
                player->power=(keyVal=="1"?true:false);
            }

            keyVal=getKeyValue(parameter,"artwork_url",&ok);
            if(ok) {
                if (!keyVal.startsWith("http",Qt::CaseInsensitive)) {
                    keyVal= QString(httpPrefixGen(settings)+settings.ipAddress()+':'+QString::number(settings.httpPort())+"/" +keyVal);
                }
                player->remoteAlbumArtURL=keyVal;
            }
        }

        if (isActive && (emitUiUpdate || player->updateAlbumArt)) {
            emitUiUpdate =false;
            qDebug() << "updateStatus()" << "Emit UI Update";
            emit playerStatusUpdated(player);
        }
        if (isActive && (emitPlaylistUpdate && mySqueezeEnv.initDone)) {
            emitPlaylistUpdate =false;
        }
    }
}


//-----------------------------------------------------------------------------------
// squeezster::updateRadiosApps:
// Decrypts a "radios" and "apps" messages
//-----------------------------------------------------------------------------------
void squeezster::updateRadiosApps(QString command, QStringList *parameters) {
    qDebug() << "updateRadiosApps()" << command << *parameters;
    quint16 pStartIndex, pEndIndex;

    // Radios and apps is not fully compatible with the "standard xml" protocol
    // Fix this by altering the list:

    if(responseIsList(parameters, &pStartIndex, &pEndIndex )) {
        qDebug() << "Is radios/apps list." << parameters;
        parameters->takeFirst(); // first index;
        parameters->takeFirst(); // second index;
        parameters->takeFirst(); // weight;
        QString countString=parameters->takeFirst(); // count
        parameters->append(countString);
        parameters->prepend(QString("title:"+command));
        parameters->prepend("100");
        parameters->prepend("0");
        parameters->prepend("items");
        if(command==QString("radios")) {
            ui->radiosTreeWidget->clear();
            updateXmlBrowserList("radios",parameters,"icon",ui->radiosTreeWidget);
        }
        else if(command==QString("apps")) {
            ui->appsTreeWidget->clear();
            updateXmlBrowserList("apps",parameters,"icon",ui->appsTreeWidget);
        }
    }
    else parameters->clear();
}


//----------------------------------------------
// squeezster::updateMixer
// Not implemented
//----------------------------------------------
void squeezster::updateMixer(squeezePlayer *player, QStringList *parameters) {qDebug() << "updateMixer()";};


//----------------------------------------------
// squeezster::queryStatus
// Send a status request to the server
//----------------------------------------------
void squeezster::queryStatus(squeezePlayer *player, bool subscribeEnable)
{
    qDebug() << "queryStatus()";
    QStringList parameters;
    parameters << "-" << "1" << STATUSTAGS;
    if(subscribeEnable)
        parameters << QString("subscribe:" + QString::number(SUBSCRIBEINTERVAL));
    QString command = "status";
    send(player,command,parameters,true);
};


//----------------------------------------------
// squeezster::slaveMasterConnect()
// Link master and slaves in a synchronized
// Squeezebox system
//----------------------------------------------
void squeezster::slaveMasterConnect() {
    qDebug() << "slaveMasterConnect()";
    // Add master and slaves
    QMutableListIterator<squeezePlayer*> player(mySqueezeEnv.playerList);
    QMutableListIterator<squeezePlayer*> master(mySqueezeEnv.playerList);
    while (player.hasNext())
    {
        bool masterFound = false;
        if (!(player.peekNext()->masterPlayerID.isEmpty()) &&
                (player.peekNext()->masterPlayerID != player.peekNext()->playerID)) {
            while (master.hasNext() && !masterFound) {
                if (player.peekNext()->masterPlayerID == master.next()->playerID) {
                    master.peekPrevious()->slavePlayers << player.peekNext();
                    masterFound = true;
                    player.peekNext()->masterPlayer=master.peekPrevious();
                    player.peekNext()->isSlave=true;
                    qDebug() << player.peekNext()->playerName << "found master:" << master.peekPrevious()->playerName;
                }
            }
        }
        player.next();
    }
}


//----------------------------------------------
// squeezster::updateUI(squeezePlayer *player)
// Update the UI to reflect the player status
//----------------------------------------------
void squeezster::updateUI(squeezePlayer *player)
{
    qDebug() << "updateUI()";
    bool isActive = (mySqueezeEnv.activePlayer == player);
    //if ((!mySqueezeEnv.initDone) || isActive) {
    if (isActive) {
        qDebug() << "updateUI(squeezePlayer *player)";
        QFontMetrics fontMetrics = ui->titleLabel->fontMetrics();
        ui->titleLabel->setText(fontMetrics.elidedText(player->currentTitle,Qt::ElideMiddle,ui->titleLabel->width()));
        fontMetrics = ui->artistLabel->fontMetrics();
        ui->artistLabel->setText(fontMetrics.elidedText(player->currentArtist,Qt::ElideMiddle,ui->artistLabel->width()));
        fontMetrics = ui->albumLabel->fontMetrics();
        ui->albumLabel->setText(fontMetrics.elidedText(player->currentAlbum,Qt::ElideMiddle,ui->albumLabel->width()));
        ui->volumeSlider->setValue(player->currentVolume);
        ui->progressBar->setMaximum((player->currentDuration?player->currentDuration:3600));
        ui->progressBar->setValue(player->currentTime);
        //---------------------------------------------
        // Update Icons to reflect status
        //---------------------------------------------
        if (mySqueezeEnv.activePlayer->power)
            ui->powerButton->setIcon(QPixmap(POWERGREENPIXMAPFILE));
        else
            ui->powerButton->setIcon(QPixmap(POWERREDPIXMAPFILE));

        updateNowPlayingIcons();

        // Menu text for shuffle
        if(player->playlistShuffle==QString("0"))
            ui->actionShuffle->setText("Shuffle: Turn ON");
        else
            ui->actionShuffle->setText("Shuffle: Turn OFF");

        // Menu text for repeat
        if(player->playlistRepeat==QString("0"))
            ui->actionRepeat->setText("Repeat: Turn ON");
        else
            ui->actionRepeat->setText("Repeat: Turn OFF");

        //---------------------------------------------
        // If valid album art, queue http request
        //---------------------------------------------
        qDebug() << "This Player:" << player->playerName;
        qDebug() << "Albumart URL" << player->updateAlbumArt;
        qDebug() << "Remote" << player->isRemote;
        QString newUrl;

        if (!player->isRemote) {
            newUrl=QString(httpPrefixGen(settings)+settings.ipAddress()+':'+QString::number(settings.httpPort())+
                           "/music/" + player->currentSongID + LOCALALBUMARTSUFFIX);
        }
        else {
            newUrl=player->remoteAlbumArtURL;
        }

        if((newUrl != player->currentAlbumArtURL) || (ui->albumArtLabel->icon().isNull())) {
            if (newUrl.isEmpty()) {
                qDebug() << "UpdateUI: Setting empty albumArtLabel";
                ui->albumArtLabel->setIcon(emptyAlbumArtIcon);
            } else {
                qDebug() << "UpdateUI: Starting to download" << newUrl;
                downloadImage(ui->albumArtLabel,newUrl);
                player->setCurrentAlbumArtURL(newUrl);
            }
            player->updateAlbumArt=false;
        }
    }
}


//----------------------------------------------
// squeezster::updateNowPlayingIcons()
// Update status icons of the current song
//----------------------------------------------
void squeezster::updateNowPlayingIcons()
{
    squeezePlayer* player = mySqueezeEnv.activePlayer;
    qDebug() << "squeezster::updateNowPlayingIcons()";

    // Set previous song icon to bullet
//    if ((ui->nowPlayingListWidget->count()>player->prevPlaylistIndex) && (player->prevPlaylistIndex!=player->playlistIndex))
//    {
//        ui->nowPlayingListWidget->item(player->prevPlaylistIndex)->setIcon(smallBulletIcon);
//    }
    int i;
    for(i=0; i<ui->nowPlayingListWidget->count(); i++) {
        ui->nowPlayingListWidget->item(i)->setIcon(smallBulletIcon);
        }
    // Update current song icon
    if (ui->nowPlayingListWidget->count()>player->playlistIndex)
    {
        if (player->power) {
            if (player->currentMode == "play")
                ui->nowPlayingListWidget->item(player->playlistIndex)->setIcon(playIcon);
            else if (player->currentMode == "pause")
                ui->nowPlayingListWidget->item(player->playlistIndex)->setIcon(pauseIcon);
            else if (player->currentMode == "stop")
                ui->nowPlayingListWidget->item(player->playlistIndex)->setIcon(stopIcon);
        } else {
            ui->nowPlayingListWidget->item(player->playlistIndex)->setIcon(QIcon(POWERREDPIXMAPFILE));
        }
        ui->nowPlayingListWidget->scrollToItem(ui->nowPlayingListWidget->item(player->playlistIndex));
    }
}


//----------------------------------------------
// squeezster::requestCurrentPlaylist()
// Ask the server to return the current playlist
//----------------------------------------------
void squeezster::requestCurrentPlaylist() {
    qDebug() << "requestCurrentPlaylist()";
    QStringList parameters;

    parameters << QString("0") << QString::number(mySqueezeEnv.activePlayer->playlistTracks) << STATUSTAGS; //"tags:aldxK";
    send(mySqueezeEnv.activePlayer,"status",parameters);
}


//---------------------------------------------------
// squeezster::requestSqueezeList
// Generic function to request a list from the server
//----------------------------------------------------
void squeezster::requestSqueezeList(const QString command, quint16 count, quint16 start, quint16 batch, QString tags) {
    QStringList cmdList;
    qDebug() << "squeezster::requestSqueezeList" << command;
    cmdList << command;
    requestSqueezeList(cmdList, count, start, batch, tags);
}


//--------------------------------------------------
// squeezster::requestSqueezeList()
// Request an element list based on "command".
// quint16 count: Total number of available elements
// quint16 start: First element in list
// quint16 batch: Number of elements per request
// QString tags: Ask for certain info, "tags"
//---------------------------------------------------
void squeezster::requestSqueezeList(const QStringList command, quint16 count, quint16 start, quint16 batch, QString tags) {
    qDebug() << "requestSqueezePlaylist()" << command;
    quint16 modCount;
    QString newTags;
    QStringList parameters;
    if (batch==0)
        modCount=count;
    else {
        if(count>batch)
            modCount=batch;
        else
            modCount=count;
    }
    parameters << QString::number(start) << QString::number(count);
    if (tags.isEmpty()) {
        if (command.first()==QString("albums"))
            newTags = ALBUMTAGS;
        else
            newTags = STATUSTAGS;
    }
    else
        newTags = tags;

    sendList(command,modCount,start,newTags);

}


//----------------------------------------------
// squeezster::responseIsList()
// Check if the respone is an elements list
//----------------------------------------------
quint16 squeezster::responseIsList (QStringList *parameters, quint16* pStartIndex, quint16* pEndIndex)
{
    QString tmpString;
    bool isNumberArg1, isNumberArg2;
    quint16 startIndex=0,endIndex=0;
    qDebug() << "responseIsList()" << *pStartIndex << *pEndIndex;

    if (parameters->size() > 1) {
        tmpString = parameters->at(0);
        startIndex=tmpString.toInt(&isNumberArg1);
        tmpString = parameters->at(1);
        endIndex = tmpString.toInt(&isNumberArg2);
    }

    *pStartIndex=startIndex;
    *pEndIndex=endIndex;

    if(!parameters->isEmpty() && isNumberArg1 && isNumberArg2) {
        qDebug() << "Is responselist";
        return (*pEndIndex); }
    else
    {
        qDebug() << "Is NOT responselist";
        return(0); }
}


//----------------------------------------------
// squeezster::setVolume(int index)
// Set the volume of the active player
//----------------------------------------------
void squeezster::setVolume(int index) {
    quint16 roundedIndex = ((int)index/5)*5;
    if (mySqueezeEnv.activePlayer->currentVolume != roundedIndex) {
        qDebug() << "setVolume() " << QString::number(roundedIndex);
        send(mySqueezeEnv.activePlayer,"mixer",(QStringList("volume") << QString::number(roundedIndex)));
    }
}

//----------------------------------------------
// squeezster::setNewProgress(int index)
// Jump inside a song
//----------------------------------------------
void squeezster::setNewProgress(int index) {
    qDebug() << "setNewProgress() " << QString::number(index);
    quint16 roundedIndex = ((int)index/5)*5;
    send(mySqueezeEnv.activePlayer,"time",QStringList(QString::number(roundedIndex)));
}


//--------------------------------------------------------
// squeezster::getNumberOfPlayers(QString responseString)
// Filter out the count value from a string
//--------------------------------------------------------
qint16 squeezster::getNumberOfPlayers(QString responseString)
{
    qDebug() << "squeezster::getNumberOfPlayers(QString responseString)";
    QString localString = responseString.split("count:",QString::SkipEmptyParts).last();
    return(localString.toInt());
}


//----------------------------------------------
// squeezster::getKeyValue()
// Return the value of a keypair
//----------------------------------------------
QString squeezster::getKeyValue(QString keyPair, QString keyName, bool *ok)
{
    //qDebug() << "getKeyValue()";
    keyName += ":"; // Separator ':'
    if (keyPair.startsWith(keyName)) {
        QString tmpString= QString(keyPair.split(keyName,QString::SkipEmptyParts).last());
        if(ok!=0)
            *ok = true;
        return(QUrl::fromPercentEncoding(tmpString.toUtf8())); }
    else {
        if(ok!=0)
            *ok = false;
        return(QString::null);
    }
}


//----------------------------------------------
// squeezster::splitKeyPair()
// Divide a keypair into key and value
//----------------------------------------------
void squeezster::splitKeyPair(QString keyPair, QString* keyName, QString* keyVal) {
    *keyName=keyPair.split(':',QString::SkipEmptyParts).first();
    *keyVal=keyPair.split(*keyName+':',QString::SkipEmptyParts).last();
}


//----------------------------------------------
// squeezster::getInitValue()
// This functions finds and returns a keyvalue
// from a response
//----------------------------------------------
QString squeezster::getInitValue(QString responseString, QString keyName)
{
    //qDebug() << "getInitValue()";
    bool keyFound = false;
    QString keyPair,tmpString;
    QStringList localStringList = responseString.split(" ",QString::SkipEmptyParts);
    QStringListIterator iString(localStringList);
    keyName += "%3A"; // Separator ':'
    while (iString.hasNext() && !keyFound) {
        if (iString.next().contains(keyName)) {
            keyFound = true;
            keyPair = iString.peekPrevious();
        }
    }
    if (keyFound) {
        tmpString = QString(keyPair.split(keyName,QString::SkipEmptyParts).last());
        return(QUrl::fromPercentEncoding(tmpString.toUtf8())); }
    else {
        qDebug() << "getInitValue() Couldn't find " << keyName << "in" << responseString;
        return(QString::null); }
}


//----------------------------------------------
// squeezster::displayError()
// Present connection errors
//----------------------------------------------
void squeezster::displayError(QAbstractSocket::SocketError socketError)
{
    switch (socketError) {
    case QAbstractSocket::RemoteHostClosedError:
        break;
    case QAbstractSocket::HostNotFoundError:
        QMessageBox::information(this, tr("squeezster"),
                                 tr("The host was not found. Please check the "
                                    "host name and port settings."));
        constructConnDialog();
        break;
    case QAbstractSocket::ConnectionRefusedError:
        QMessageBox::information(this, tr("squeezster"),
                                 tr("The connection was refused by the peer. "
                                    "Make sure the fortune server is running, "
                                    "and check that the host name and port "
                                    "settings are correct."));
        break;
    default:
        QMessageBox::information(this, tr("squeezster Client"),
                                 tr("The following error occurred: %1.")
                                 .arg(squeezster::tcpSocket->errorString()));
    }
}


//----------------------------------------------
// squeezster::updateActivePlayer(int index)
// Request status for active player, and
// subscribe on status
//----------------------------------------------
void squeezster::updateActivePlayer(int index) {
    squeezePlayer* oldPlayer;
    oldPlayer = mySqueezeEnv.activePlayer;

    qDebug() << "squeezster::updateActivePlayer(int index). New index: " << QString::number(index);

    mySqueezeEnv.activePlayer=mySqueezeEnv.playerList.at(index);
    // Indicate change of player
    mySqueezeEnv.playerChanged=true;

    ui->playerComboBox->setCurrentIndex(index);


    // Request status subscription
    if (mySqueezeEnv.initDone) {
        queryStatus(mySqueezeEnv.activePlayer,true);
        qDebug() << "squeezster::updateActivePlayer(int index). Subscribe to "+mySqueezeEnv.activePlayer->playerName;
    }

    // Disable subscription of old player
    if(mySqueezeEnv.activePlayerSet==true) {
        qDebug() << "squeezster::updateActivePlayer(int index). Old index: " << oldPlayer->playerName;
        send(oldPlayer,"status",QStringList("subscribe:-")); // unsubscribe;
    }
    mySqueezeEnv.activePlayerSet= true;
}


//----------------------------------------------
// squeezster::sendPlay()
// Send play command
//----------------------------------------------
void squeezster::sendPlay() {
    qDebug() << "sendPlay() ";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"play");
        mySqueezeEnv.activePlayer->currentMode="play";
        updateUI(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendPause()
// Send plause command
//----------------------------------------------
void squeezster::sendPause() {
    qDebug() << "sendPause()";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"pause");
        if(mySqueezeEnv.activePlayer->currentMode==QString("pause"))
            mySqueezeEnv.activePlayer->currentMode="play";
        else
            mySqueezeEnv.activePlayer->currentMode="pause";
        updateUI(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendStop()
// Send stop command
//----------------------------------------------
void squeezster::sendStop() {
    qDebug() << "sendStop()";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"stop");
        mySqueezeEnv.activePlayer->currentMode="stop";
        updateUI(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendSkipBack()
// Send skipback command
//----------------------------------------------
void squeezster::sendSkipBack() {
    qDebug() << "sendBackward()";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"playlist",(QStringList("index") << "-1"));
        queryStatus(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendSkipForward()
// Send skipforward command
//----------------------------------------------
void squeezster::sendSkipForward() {
    qDebug() << "sendSkipForward() ";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"playlist",(QStringList("index") << "+1"));
        queryStatus(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendPlayistIndex()
// Ask server to play index in playlist
//----------------------------------------------
void squeezster::sendPlaylistIndex(quint16 index) {
    qDebug() << "sendPlaylistIndex() ";
    if(mySqueezeEnv.initDone) {
        send(mySqueezeEnv.activePlayer,"playlist",(QStringList("index") << QString::number(index)));
        updateUI(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendPower()
// Turn on or off player
//----------------------------------------------
void squeezster::sendPower() {
    qDebug() << "sendPower() ";
    if(mySqueezeEnv.initDone) {
        if (mySqueezeEnv.activePlayer->power)
            send(mySqueezeEnv.activePlayer,"power",(QStringList("0")));
        else
            send(mySqueezeEnv.activePlayer,"power",(QStringList("1")));
        mySqueezeEnv.activePlayer->power=!mySqueezeEnv.activePlayer->power;
        updateUI(mySqueezeEnv.activePlayer);
    }
}


//----------------------------------------------
// squeezster::sendPlaylistClear()
// Clear current playlist
//----------------------------------------------
void squeezster::sendPlaylistClear() {
    qDebug() << "sendPlaylistClear() ";
    if(mySqueezeEnv.initDone) {
        QString command = "playlist";
        QStringList parameters("clear");
        send(mySqueezeEnv.activePlayer, command, parameters);
    }
}


//--------------------------------------------------
// squeezster::sendList()
// Ask the server to return list based on "command"
//--------------------------------------------------
void squeezster::sendList(QStringList command, quint16 count, quint16 start, QString tags) {
    qDebug() << "sendList()" << command;
    if(mySqueezeEnv.initDone) {
        QString cmd = command.takeFirst();
        command << QString::number(start) << QString::number(count-1) << tags;
        send(mySqueezeEnv.activePlayer,cmd,command);
    }
}


//---------------------------------------------------------
// Send() functions
//---------------------------------------------------------
QString squeezster::send(const squeezePlayer *player, const QString command, const QStringList parameters,bool force)
{
    QStringList stringList = QStringList(player->playerID);
    stringList << command << parameters;
    return(send(stringList,force));
}

QString squeezster::send(const QStringList commands,bool force)
{
    qDebug() << "Send()(StringList)";
    if(mySqueezeEnv.initDone || force) {
        QByteArray sendByteArray = "";
        QStringListIterator str(commands);
        if (!commands.isEmpty()) {
            while (str.hasNext()) {
                sendByteArray += " ";
                sendByteArray += QUrl::toPercentEncoding(str.next(),":");
            }
            qDebug() << sendByteArray;
            if(!outTS->device()->isOpen())
                requestConnect();
            qDebug() << "Send() sending:" << sendByteArray.trimmed();
            *outTS << sendByteArray.trimmed();
            *outTS << "\n";
            outTS->flush();
        }
        return(QString(sendByteArray));
    } else
        return(QString());
}


//----------------------------------------------
// squeezster::sessionOpened()
// Currently not used
//----------------------------------------------
void squeezster::sessionOpened()
{
    // Save the used configuration
    qDebug() << "Enter sessionOpened";
    QNetworkConfiguration config = networkSession->configuration();
    QString id;
    if (config.type() == QNetworkConfiguration::UserChoice)
        id = networkSession->sessionProperty(QLatin1String("UserChoiceConfiguration")).toString();
    else
        id = config.identifier();
    QSettings sessionSettings(QSettings::UserScope, QLatin1String("Trolltech"));
    sessionSettings.beginGroup(QLatin1String("QtNetwork"));
    sessionSettings.setValue(QLatin1String("DefaultNetworkConfiguration"), id);
    sessionSettings.endGroup();
}


//----------------------------------------------
// squeezster::updateProgress()
// Updates the elapsed time of the song
//----------------------------------------------
void squeezster::updateProgress()
{
    quint16 time;
    qDebug() << "updateProgress()";
    time = ui->progressBar->value()+PROGRESSINTERVAL/1000;
    qDebug() << "updateProgress()" << "New Time:" << QString::number(time) << "Max Time:" << QString::number(ui->progressBar->maximum());
    if ((time <= ui->progressBar->maximum()) && mySqueezeEnv.activePlayer->currentMode==QString("play")) {
        ui->progressBar->setValue(time);
    } else if(mySqueezeEnv.activePlayer->currentMode==QString("stop")){
        ui->progressBar->setValue(1);
    }
}


//----------------------------------------------
// squeezster::downloadImage()
// Issues a http download request of an image
//----------------------------------------------
void squeezster::downloadImage(QTreeWidgetItem* item, QString url) {

    qDebug() << "downloadImage(): Starting to download" << url;
    if(!url.startsWith("http")) // E.g. radio
        url=QString(httpPrefixGen(settings)+settings.ipAddress()+":"+QString::number(settings.httpPort())+"/" + url);
    treeWidgetItemNetworkRequest treeWidgetItemRequest(item);
    treeWidgetItemRequest.setUrl(url);
    networkManager->get(treeWidgetItemRequest);
    isBusy("downloadImage<QTreeWidgetItem*>");
}

void squeezster::downloadImage(xmlListItem* item, QString url) {

    qDebug() << "downloadImage(): Starting to download" << url;
    listWidgetItemNetworkRequest listWidgetItemRequest(item);
    listWidgetItemRequest.setUrl(url);
    networkManager->get(listWidgetItemRequest);
    isBusy(QString("downloadImage<xmlListItem*>:"+url));
}

void squeezster::downloadAlbumImage(xmlListItem* item)
{
    qDebug() << "downloadAlbumImage()";
    xmlListItem* xmlItem = static_cast<xmlListItem*>(item);
    if(!QString(xmlItem->property("artwork_track_id")).isEmpty()) {
        downloadAlbumImage(item,xmlItem->property("artwork_track_id"),settings);
        //qDebug() << "downloadImage(): Starting to download artwork_id:" << xmlItem->property("artwork_track_id");
    }
    else {
        //qDebug() << "downloadAlbumImage(xmlListItem* item): No proper artwork found";
        item->setIcon(emptyAlbumArtIcon);
        //isBusy();
    }
}

void squeezster::downloadAlbumImage(xmlListItem* item, QString id, UserSettings settings)
{
    qDebug() << "downloadAlbumImage()";
    QString url =QString(httpPrefixGen(settings)+settings.ipAddress()+':'+QString::number(settings.httpPort())+"/music/"+id+SMALLLOCALALBUMARTSUFFIX);
    QNetworkRequest listWidgetItemRequest(url);
    QVariant variant;
    qVariantSetValue(variant,item);
    QObject* object=new QObject();

    object->setProperty("item",variant);
    listWidgetItemRequest.setOriginatingObject(object);

    networkManager->get(listWidgetItemRequest);
    isBusy(QString("downloadImage<xmlListItem*>:"+url));
    // item->setIcon(QIcon(EMPTYALBUMARTPIXMAPFILE));
}


void squeezster::downloadImage(QPushButton* item, QString url) {

    qDebug() << "downloadImage(): Starting to download" << url;
    pushButtonNetworkRequest pushButtonRequest(item);
    pushButtonRequest.setUrl(url);
    networkManager->get(pushButtonRequest);
    isBusy("downloadImage<QPushButton*");
}


//----------------------------------------------
// squeezster::pixmapDownloaded()
// Called when a http request has finished.
// Assigns the image to the correct GUI item
//----------------------------------------------
void squeezster::pixmapDownloaded(QNetworkReply *reply)
{
    qDebug() << "pixmapDownloaded()";
    QNetworkRequest request = reply->request();
    QVariant variant;
    if(request.originatingObject())
        variant = request.originatingObject()->property("item");
    else
        variant = request.attribute((QNetworkRequest::Attribute)(1000));
    //qDebug() << "pixmapDownloaded()" << reply->url().toString() << "Variant" << variant.typeName();
    QByteArray jpegData = QByteArray(reply->readAll());
    QPixmap downloadPixmap; //= new QPixmap();
    QIcon newIcon;

    if (reply->error() != QNetworkReply::NoError) {
        qWarning() << "Error in" << reply->url() << ":" << reply->errorString();
        //return;
        if (QString(variant.typeName()) == "QLabel*")
        {     // QLabel
            variant.value<QLabel*>()->setPixmap(QPixmap(EMPTYALBUMARTPIXMAPFILE));
        } else if (variant.userType()==QMetaType::type("QPushButton*"))
        {     // QPushButton
            if(variant.canConvert(variant.type()))
                variant.value<QPushButton*>()->setIcon(emptyAlbumArtIcon);
            else {
                qWarning() << "squeezster::pixmapDownloaded(QNetworkReply *reply) Cannot convert:" << variant.typeName();
                isBusy(); return; }
        } else if (variant.userType()==QMetaType::type("xmlTreeItem*"))
        {
            variant.value<xmlTreeItem*>()->setIcon(emptyAlbumArtIcon);
        }
        else if (variant.userType()==QMetaType::type("xmlListItem*"))
        {
            if(variant.canConvert(variant.type()))
                variant.value<xmlListItem*>()->setIcon(emptyAlbumArtIcon);
            else {
                qWarning() << "squeezster::pixmapDownloaded(QNetworkReply *reply) Cannot convert:" << variant.typeName();
                isBusy(); return; }
        }
        else if (variant.userType()==QMetaType::type("QTreeWidgetItem*"))
        {
            variant.value<QTreeWidgetItem*>()->setIcon(0,emptyAlbumArtIcon);
        }
        else if (variant.userType()==QMetaType::type("QListWidgetItem*"))
        {
            variant.value<QListWidgetItem*>()->setIcon(emptyAlbumArtIcon);
        }
    }
    else {
        if (!downloadPixmap.loadFromData(jpegData)) {
            qWarning() << "Error loadFromData()";
            downloadPixmap=QPixmap(EMPTYALBUMARTPIXMAPFILE);
        } else {
            newIcon= QIcon(downloadPixmap);

            if (variant.userType()==QMetaType::type("QPushButton*"))
            {     // QPushButton
                variant.value<QPushButton*>()->setIcon(newIcon);
            }
            else if (variant.userType()==QMetaType::type("xmlTreeItem*"))
            {
                xmlTreeItem* newItem= variant.value<xmlTreeItem*>();
                qDebug() << "image" << newItem->property("image");
                newItem->setIcon(newIcon);
                //qDebug() << "pixmapDownloaded() Setting treeWidgetIcon";
            }
            else if (variant.userType()==QMetaType::type("xmlListItem*"))
            {
                xmlListItem* newItem= variant.value<xmlListItem*>();
                newItem->setIcon(newIcon);
                //qDebug() << "pixmapDownloaded() Setting listWidgetIcon";
            }
            else if (variant.userType()==QMetaType::type("QLabel*"))
            {     // QLabel
                variant.value<QLabel*>()->setPixmap(downloadPixmap);
            }
            else if (variant.userType()==QMetaType::type("QTreeWidgetItem*"))
            {
                QTreeWidgetItem* newItem= variant.value<QTreeWidgetItem*>();
                newItem->setIcon(0,newIcon);
                //qDebug() << "pixmapDownloaded() Setting treeWidgetIcon" << variant.value<QTreeWidgetItem*>()->text(xmlItemCol);
            }
            else if (variant.userType()==QMetaType::type("QListWidgetItem*"))
            {
                QListWidgetItem* newItem= variant.value<QListWidgetItem*>();
                newItem->setIcon(newIcon);
                //qDebug() << "pixmapDownloaded() Setting listWidgetIcon" << variant.value<QListWidgetItem*>()->text();
            }
            else {
                qDebug() <<  "Couldn't find type for pixmap assignment!!!";
            }
        }
    }
    reply->deleteLater();
    isBusy();
}


//----------------------------------------------
// squeezster::initTitlesTree()
// Initializes the titlesTreeWidget
//----------------------------------------------
void squeezster::initTitlesTree() {
    qDebug() << "squeezster::initTitlesTree()";

    ui->titlesTreeWidget->clear();
    char c;
    xmlTreeItem* newItem;
    newItem = new xmlTreeItem();
    newItem->setText(xmlItemCol,"All");
    newItem->addProperty("cmd","titles");
    newItem->addProperty("type","native");
    ui->titlesTreeWidget->addTopLevelItem(newItem);
    // 0-9
    newItem = new xmlTreeItem();
    newItem->setText(xmlItemCol,"Search 0-9");
    newItem->addProperty("search",QString("0123456789"));
    newItem->addProperty("cmd","titles");
    newItem->addProperty("type","native");
    ui->titlesTreeWidget->addTopLevelItem(newItem);
    for(c='A';c<'Z';c++) {
        newItem = new xmlTreeItem();
        newItem->setText(xmlItemCol,QString(QString("Search ")+QChar(c)));
        newItem->addProperty("search",QString(c));
        newItem->addProperty("cmd","titles");
        newItem->addProperty("type","native");
        ui->titlesTreeWidget->addTopLevelItem(newItem);
    }
}


//----------------------------------------------
// squeezster::initArtistsTree()
// Initializes the artistsTreeWidget
//----------------------------------------------
void squeezster::initArtistsTree() {
    qDebug() << "squeezster::initArtistsTree()";

    ui->artistsTreeWidget->clear();
    char c;
    xmlTreeItem* newItem;
    newItem = new xmlTreeItem();
    newItem->setText(xmlItemCol,"All");
    newItem->addProperty("cmd","artists");
    newItem->addProperty("type","native");
    ui->artistsTreeWidget->addTopLevelItem(newItem);
    // 0-9
    newItem = new xmlTreeItem();
    newItem->setText(xmlItemCol,"Search 0-9");
    newItem->addProperty("search",QString("0123456789"));
    newItem->addProperty("cmd","artists");
    newItem->addProperty("type","native");
    ui->artistsTreeWidget->addTopLevelItem(newItem);
    for(c='A';c<'Z';c++) {
        newItem = new xmlTreeItem();
        newItem->setText(xmlItemCol,QString(QString("Search ")+QChar(c)));
        newItem->addProperty("search",QString(c));
        newItem->addProperty("cmd","artists");
        newItem->addProperty("type","native");
        ui->artistsTreeWidget->addTopLevelItem(newItem);
    }
}


//----------------------------------------------
// squeezster::writeSettings()
// Writes the user settings to the system
//----------------------------------------------
void squeezster::writeSettings(const UserSettings newSettings)
{
    qDebug() << "squeezster::writeSettings(const UserSettings newSettings)";
    QSettings localSettings("Audiovance", "Squeezster");

    localSettings.beginGroup("Squeezster");
    localSettings.setValue("ipAddress", newSettings.ipAddress());
    localSettings.setValue("httpPort", newSettings.httpPort());
    localSettings.setValue("cliPort", newSettings.cliPort());
    localSettings.setValue("username", newSettings.username());
    localSettings.setValue("password", newSettings.password());
    localSettings.setValue("screensaverInhibited", newSettings.screensaverIsInhibited());
    localSettings.endGroup();

    qDebug() << "squeezster::writeSettings() newSettings.screensaverIsInhibited()" << newSettings.screensaverIsInhibited();
}


//----------------------------------------------------------
// squeezster::readSettings()
// Reads (from the system) or initializes the user settings
//----------------------------------------------------------
void squeezster::readSettings(UserSettings &newSettings)
{
    qDebug() << "squeezster::readSettings(UserSettings &newSettings)";

    QSettings localSettings("Audiovance", "Squeezster");

    localSettings.beginGroup("Squeezster");
    newSettings.setIpAddress(localSettings.value("ipAddress",IPADDRESS).toString());
    newSettings.setHttpPort(localSettings.value("httpPort",HTTPPORT).toUInt());
    newSettings.setCliPort(localSettings.value("cliPort",CLIPORT).toUInt());
    newSettings.setUsername(localSettings.value("username",QString("")).toString());
    newSettings.setPassword(localSettings.value("password",QString("")).toString());
    newSettings.setScreensaverIsInhibited(localSettings.value("screensaverInhibited",false).toBool());
    localSettings.endGroup();
}


//----------------------------------------------
// squeezster::savePlaylist()
// Saves the current playlist to the server
//----------------------------------------------
void squeezster::savePlaylist()
{
    qDebug() << "squeezster::savePlaylist()";
    bool ok;
    QString playlistName = QInputDialog::getText(this,tr("Save Playlist"),tr("Name:"),QLineEdit::Normal,"",&ok);
    QStringList parameters;
    parameters << "save" << playlistName;
    if(ok) {
        send(mySqueezeEnv.activePlayer,"playlist",parameters);
        requestSqueezeList("playlists");
    }
}


//----------------------------------------------
// squeezster::playlistItemClicked()
// Called when a song in the current playlist is
// clicked. Sends a request to play the song.
//----------------------------------------------
void squeezster::playlistItemClicked(QListWidgetItem* songItem)
{
    qDebug() << "squeezster::playlistItemClicked()";
    xmlListItem* item=static_cast<xmlListItem*>(songItem);
    qVariantSetValue(listItemParent,static_cast<xmlListItem*>(item));
    qVariantSetValue(itemSelected,static_cast<xmlListItem*>(item));
    itemSelectTimer.start();

    //if(item->playlistIndex()!= mySqueezeEnv.activePlayer->playlistIndex) {
        //mySqueezeEnv.activePlayer->prevPlaylistIndex=mySqueezeEnv.activePlayer->playlistIndex ;
        //mySqueezeEnv.activePlayer->playlistIndex=item->playlistIndex();
    //}
    sendPlaylistIndex(item->playlistIndex());
    queryStatus(mySqueezeEnv.activePlayer);
}


//-------------------------------------------------
// squeezster::treeItemClicked()
// Called then an item in a QTreeWidget is clicked.
// Handles the action.
//-------------------------------------------------
void squeezster::treeItemClicked(QTreeWidgetItem* treeItem) {

    xmlTreeItem* item = static_cast<xmlTreeItem*>(treeItem);
    qDebug() << "treeItemClicked()" << item->text() << item->property("xmlbrowser") << item->command() << item->isXmlBrowser();
    isBusy("treeItemClicked");
    bool ok;
    int i;
    QString searchString = "search:";
    QString string;
    QString command = item->command();
    QStringList parameters;
    qVariantSetValue(treeItemParent,static_cast<xmlTreeItem*>(item));
    itemSelected.setValue(item);
    itemSelectTimer.start();
    if (item->isExpanded()) {
        item->setExpanded(false);
    }
    else if(item->property("type")==QString("native")) {
        // Get artists and titles
        string=QString(item->property("search"));
        if(string.isEmpty())
            requestSqueezeList(command);
        else {
            for(i=0;i<string.length();i++) {
                qDebug() << "Fetching" << command << "search:" << string.at(i);
                searchString = QString("search:")+QChar(string.at(i));
                parameters << "0" << QString::number(MAXTREEITEMSPERREQUEST) << searchString;
                send(mySqueezeEnv.activePlayer,command,parameters);
                parameters.clear();
            }
        }
    }
    else if(item->command()==QString("musicfolder")) {
        if(item->typeFolder()) {
            qDebug() << "treeItemClicked() musicfolder" << item->typeFolder();
            parameters << "0" << QString::number(MAXTREEITEMSPERREQUEST);
            parameters << item->folder_id();
            send(mySqueezeEnv.activePlayer, command, parameters);
        } else if(item->typeTrack()) {
            qDebug() << "treeItemClicked() musicfolder typeTrack";
            parameters << "cmd:add" << item->track_id();
            send(mySqueezeEnv.activePlayer, "playlistcontrol", parameters);
        }
    }
    else if(item->isXmlBrowser() || item->hasItems()) // xmlbrowser e.g. radios, apps & spotify
    {
        qDebug() << "Going deeper in XML.";
        if (QString(item->property("type")).contains("search")) {
            searchString += QInputDialog::getText(this,tr("Search"),tr("Search string:"),QLineEdit::Normal,"",&ok);
            if (!ok)
                searchString.clear();
        }
        parameters << "items" << "0" << QString::number(MAXITEMSPERREQUEST);
        parameters << item->item_id();
        parameters << searchString ;
        send(mySqueezeEnv.activePlayer, command, parameters);
    }
    else if(!item->command().isEmpty()) {
        if(item->isAudio()) {
            qDebug() << "isAudio. Starting streams";
            parameters << "playlist" << "play" << item->item_id();
            send(mySqueezeEnv.activePlayer, command, parameters);
            //requestSqueezeList("status"); // requestCurrentPlaylist();
        }
    }

    else if(!item->property("item_id").isEmpty()) {
        parameters << "cmd:load" << item->item_id();
        send(mySqueezeEnv.activePlayer, "playlistcontrol", parameters);
    }
    else if(!item->property("artist_id").isEmpty()) {
        parameters << "cmd:load" << item->artist_id();
        send(mySqueezeEnv.activePlayer, "playlistcontrol", parameters);
    }
    else if(!item->property("title_id").isEmpty()) {
        parameters << "cmd:load" << item->title_id();
        send(mySqueezeEnv.activePlayer, "playlistcontrol", parameters);
    }
    else if(!item->property("cmd").isEmpty()) {
        parameters << "cmd:load" << item->playlist_id();
        send(mySqueezeEnv.activePlayer, "playlistcontrol", parameters);
    }
    else
        qDebug() << "TreeItemClicked: Doing nothing.";
    isBusy();
}


//----------------------------------------------
// squeezster::listItemClicked()
// Called then an item in a QListWidget is clicked.
// Handles the action.
//----------------------------------------------
void squeezster::listItemClicked(QListWidgetItem* listItem) {
    xmlListItem* item=static_cast<xmlListItem*>(listItem);
    qVariantSetValue(itemSelected,static_cast<xmlListItem*>(item));
    itemSelectTimer.start();
    QStringList parameters;
    qDebug() << "listItemClicked()" << listItem->text();
    parameters << "cmd:load";
    if(!item->property("album").isEmpty())
        parameters << item->album_id();
    else if(!item->property("playlist").isEmpty())
        parameters << item->playlist_id();
    send(mySqueezeEnv.activePlayer,"playlistcontrol",parameters);
}


//--------------------------------------------------
// squeezster::getFirstKey()
// Get the value assigned to a key in a stringlist
//--------------------------------------------------
QString squeezster::getFirstKey(QStringList parameters, QString keyName) {
    QString keyValue;
    while (!QString(keyValue=QString(getKeyValue(parameters.first(),keyName))).isEmpty());
    return(!keyValue.isEmpty()?keyValue:QString::null);
}


//----------------------------------------------
// squeezster::incrementScreenMode()
// Switch GUI modes
//----------------------------------------------
void squeezster::incrementScreenMode() {
    screenMode++;
    if (screenMode>2)
        screenMode=0;
    setupScreenMode();
}


//----------------------------------------------
// squeezster::itemUnselect()
// Called by a timer set when an item is clicked.
// Unselects the item, and thus gives feedback to
// the user that the click has been registered.
//----------------------------------------------
void squeezster::itemUnselect()
{
    qDebug() << "squeezster::itemUnselect()";
    if(itemSelected.isValid()) {
        if (itemSelected.userType()==QMetaType::type("xmlTreeItem*")) {
            itemSelected.value<xmlTreeItem*>()->setSelected(false);
        }
        else if(itemSelected.userType()==QMetaType::type("QTreeWidgetItem*")) {
            itemSelected.value<QTreeWidgetItem*>()->setSelected(false);
        }
        else if (itemSelected.userType()==QMetaType::type("xmlListItem*")) {
            itemSelected.value<xmlListItem*>()->setSelected(false);
        }
        else if (itemSelected.userType()==QMetaType::type("QListWidgetItem*")) {
            itemSelected.value<QListWidgetItem*>()->setSelected(false);
        }
        itemSelected.clear();
    }
}


//----------------------------------------------
// squeezster::showAlbums() etc.
// Switches the active widget in the
// library presentation (a stackedWidget)
//----------------------------------------------
void squeezster::showAlbums() {
    ui->stackedWidget->setCurrentIndex(0);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
}

void squeezster::showTitles() {
    ui->stackedWidget->setCurrentIndex(2);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
}

void squeezster::showArtists() {
    ui->stackedWidget->setCurrentIndex(3);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
}

void squeezster::showPlaylists() {
    ui->stackedWidget->setCurrentIndex(1);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
}


//----------------------------------------------
// squeezster::albumsListIconView()
// Switches the albums presentation between a
// list and icons.
//----------------------------------------------
void squeezster::albumsListIconView()
{
    int mode = ui->albumsListWidget->viewMode();
    if (mode==QListView::IconMode) {
        ui->albumsListWidget->setGridSize(QSize(60,60));
        ui->albumsListWidget->setIconSize(QSize(50,50));
        ui->albumsListWidget->setViewMode(QListView::ListMode);
        ui->actionAlbumsListIcons->setText("Album View: Icons");
    } else {
        ui->albumsListWidget->setGridSize(QSize(116,150));
        ui->albumsListWidget->setIconSize(QSize(112,112));
        ui->albumsListWidget->setViewMode(QListView::IconMode);
        ui->actionAlbumsListIcons->setText("Album View: List");
    }
    ui->squeezeTabWidget->setCurrentIndex(2);
}

//----------------------------------------------
// squeezster::clearCache()
// Clears the internal database
//----------------------------------------------
void squeezster::clearCache()
{
    //diskCache->clear();
    ui->albumsListWidget->clear();
    requestSqueezeList("albums",999);
    ui->radiosTreeWidget->clear();
    requestSqueezeList("radios",999);
    ui->appsTreeWidget->clear();
    requestSqueezeList("apps",999);
    ui->folderTreeWidget->clear();
    requestSqueezeList("musicfolder",999);
}



//----------------------------------------------
// squeezster::slotShuffle()
// Enable/disable shuffle
//----------------------------------------------
void squeezster::slotShuffle()
{
    QStringList parameters;
    parameters << "shuffle";
    if(mySqueezeEnv.activePlayer->playlistShuffle == QString("0")) {
        parameters << "1";
    }
    else {
        parameters << "0";
    }

    send(mySqueezeEnv.activePlayer,"playlist",parameters);
    queryStatus(mySqueezeEnv.activePlayer);
}


//----------------------------------------------
// squeezster::slotRepeat()
// Enable/disable repeat
//----------------------------------------------
void squeezster::slotRepeat()
{
    QStringList parameters;
    parameters << "repeat";
    if(mySqueezeEnv.activePlayer->playlistRepeat == QString("0")) {
        parameters << "2";}
    else {
        parameters << "0"; }

    send(mySqueezeEnv.activePlayer,"playlist",parameters);
    queryStatus(mySqueezeEnv.activePlayer);
}


//----------------------------------------------
// squeezster::isBusy(const QString busyString)
// Set busy indicator
//----------------------------------------------
void squeezster::isBusy(const QString busyString)
{
    QString str;
    qDebug() << "isBusy()";
    if (!busyString.isEmpty())
        squeezster::busyList<<busyString;
    else
        if (!squeezster::busyList.isEmpty())
            str=squeezster::busyList.takeFirst();
   // if(!busyList.isEmpty())
   //     qDebug() << "Busy:" << busyList.first() << QString::number(busyList.count());
#if defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, (busyList.isEmpty()?Qt::Unchecked:Qt::Checked));
#endif
}


//----------------------------------------------
// squeezster::constructConnDialog()
//----------------------------------------------
void squeezster::constructConnDialog()
{
    if(manager.isOnline()) {
        connectionDialog* connectionUI=new connectionDialog(this,settings);
        connect(connectionUI,SIGNAL(connectionSettingsUpdated(UserSettings)),this,SLOT(requestConnect(UserSettings)));
        connectionUI->exec();
    }
}

//----------------------------------------------
// squeezster::slotOptions()
//----------------------------------------------
void squeezster::slotOptions()
{
    OptionsDialog* myOptionsDialog =new OptionsDialog();
    myOptionsDialog->setScreensaverInhibitedState(settings.screensaverIsInhibited());
    myOptionsDialog->show();
    connect(myOptionsDialog,SIGNAL(clearCache()),this,SLOT(clearCache()));
    connect(myOptionsDialog,SIGNAL(ssaverInhibit(bool)),this,SLOT(screensaverInhibitSlot(bool)));

}

//----------------------------------------------
// squeezster::screensaverInhibitSlot()
//----------------------------------------------
void squeezster::screensaverInhibitSlot(bool ssInhibit)
{
    qDebug() << "squeezster::screensaverInhibitSlot(bool ssInhibit)=" << ssInhibit;
    // Delete inhibiter and construct new screensaver
    if(settings.screensaverIsInhibited()) {
        delete screenSaverInhibiter;
    }

    qDebug() << "squeezster::screensaverInhibitSlot() settings.screensaverInhibit=" << settings.screensaverIsInhibited();
    if (ssInhibit)
    {
        qDebug() << "squeezster::screensaverInhibitSlot(: Enabling inhibiter";
        screenSaverInhibiter = new QSystemScreenSaver();
        screenSaverInhibiter->setScreenSaverInhibit();
    }
    settings.setScreensaverIsInhibited(ssInhibit);
    writeSettings(settings);
}


//----------------------------------------------
// squeezster::slotMusicLibraryView()
//----------------------------------------------
void squeezster::slotMusicLibraryView()
{
    qDebug() << "squeezster::slotMusicLibraryView()";
    libraryViewDialog* myLibraryViewDialog;
    myLibraryViewDialog = new libraryViewDialog();
    connect(myLibraryViewDialog, SIGNAL(albums()),this,SLOT(showAlbums()));
    connect(myLibraryViewDialog, SIGNAL(titles()),this,SLOT(showTitles()));
    connect(myLibraryViewDialog, SIGNAL(playlists()),this,SLOT(showPlaylists()));
    connect(myLibraryViewDialog, SIGNAL(artists()),this,SLOT(showArtists()));

    myLibraryViewDialog->initButtons(ui->stackedWidget->currentIndex());
    myLibraryViewDialog->show();
}

//----------------------------------------------
// squeezster::treeContextMenuHandler(QPoint point)
//----------------------------------------------
void squeezster::treeContextMenuHandler(QPoint point) {
    QTreeWidget* treeWidget = static_cast<QTreeWidget*>(this->sender());
    QPoint widgetPos = treeWidget->mapFromGlobal(point);
    // Adjust for N900 fullscreen:
    if(!this->isFullScreen())
        widgetPos.ry()+=54;
    QMenu contextMenu;
    contextMenu.addActions(treeWidget->actions());
    QAction* selectedAction = contextMenu.exec(point);

    qDebug() << this->objectName() << this->sender()->objectName();
    qDebug() << "treeContextMenuHandler(QPoint point). Pos:" << QString::number(point.y());
    qDebug() << "treeContextMenuHandler(QPoint point). Adjusted Pos:" << QString::number(widgetPos.y());

    if (selectedAction) {
        xmlTreeItem* treeItem=static_cast<xmlTreeItem*>(treeWidget->itemAt(widgetPos));
        if (treeItem) {
            qDebug() << treeItem->name() << "Item isaudio" << QString::number(treeItem->isAudio()) << "Command:" << treeItem->command();
            QString command;
            QStringList parameters;
            // Browsing in musicfolder:
            if(treeItem->command()==QString("musicfolder")) {
                command = "playlistcontrol";

                if(selectedAction->text().toLower()==QString("play"))
                    parameters << "cmd:load";
                else if(selectedAction->text().toLower()==QString("add to playlist"))
                    parameters << "cmd:add";
                if(treeItem->typeFolder())
                    parameters << treeItem->folder_id();
                else
                    parameters << treeItem->track_id();
            } else {
                // "Item", i.e. favorite or radio
                command = treeItem->command();
                if(selectedAction->text().toLower()==QString("play"))
                    parameters << "playlist" << "play";
                else if(selectedAction->text().toLower()==QString("add to playlist"))
                    parameters << "playlist" << "add";
                else if(selectedAction->text().toLower()==QString("add to favorites"))
                    parameters << "favorites" << "add";

                parameters << treeItem->item_id();
            }

            send(mySqueezeEnv.activePlayer, command, parameters);
            //requestSqueezeList("status");
        }
    }
}


//----------------------------------------------
// squeezster::listContextMenuHandler(QPoint point)
//----------------------------------------------
void squeezster::listContextMenuHandler(QPoint point) {
    QListWidget* listWidget;
    listWidget = static_cast<QListWidget*>(this->sender());
    QPoint widgetPos = listWidget->mapFromGlobal(point);
    QMenu contextMenu;
    contextMenu.addActions(listWidget->actions());

    if(!this->isFullScreen())
        widgetPos.ry()+=54;

    qDebug() << this->objectName() << this->sender()->objectName();
    qDebug() << "listContextMenuHandler(QPoint point). Pos:" << QString::number(point.y());
    qDebug() << "listContextMenuHandler(QPoint point). Adjusted Pos:" << QString::number(widgetPos.y());

    QAction* selectedAction = contextMenu.exec(point);
    xmlListItem* listItem = static_cast<xmlListItem*>(listWidget->itemAt(widgetPos));
    if (selectedAction) {
        QString command;
        QStringList parameters;
        if(selectedAction->text().toLower()==QString("clear playlist")) {
            send(mySqueezeEnv.activePlayer, "playlist", QStringList("clear")); }
        else if(QString(selectedAction->text().toLower())=="delete track") {
            parameters << "delete" << QString::number(listWidget->row(listItem));
            send(mySqueezeEnv.activePlayer, "playlist", parameters); }
        else if(QString(selectedAction->text().toLower())=="add to favorites") {
            parameters << "add" << listItem->item_id();;
            send(mySqueezeEnv.activePlayer, "favorites", parameters); }
        else {
            if(QString(selectedAction->text().toLower())=="add to playlist") {
                command    =  "playlistcontrol";
                parameters  << "cmd:add";
            }
            else if(QString(selectedAction->text().toLower())=="play") {
                command    = "playlistcontrol";
                parameters << "cmd:load";
            }

            if(!listItem->property("playlist").isEmpty())
                parameters << listItem->playlist_id();
            else if(!listItem->property("album").isEmpty())
                parameters << listItem->album_id();
            send(mySqueezeEnv.activePlayer, command, parameters);
        }
       // requestSqueezeList("status");
        qDebug() << "Action" << selectedAction->text() << listWidget->itemAt(widgetPos)->text();
        qDebug() << "SelectedItems" << listWidget->selectedItems();
        qDebug() << "Current Item" << listWidget->currentItem();
    }
}


//----------------------------------------------
//
//----------------------------------------------
void squeezster::initUI()
{
    qDebug() << "initUI";

    screenMode = 0;
    connect(ui->albumArtLabel,SIGNAL(clicked()),this,SLOT(incrementScreenMode()));
    connect(ui->actionOptions,SIGNAL(triggered()),this,SLOT(slotOptions()));
//    connect(ui->actionClearCache,SIGNAL(triggered()),this,SLOT(clearCache()));
    connect(ui->actionShuffle,SIGNAL(triggered()),this,SLOT(slotShuffle()));
    connect(ui->actionRepeat,SIGNAL(triggered()),this,SLOT(slotRepeat()));
    connect(ui->actionConnectionSettings,SIGNAL(triggered()),this,SLOT(constructConnDialog()));
    connect(ui->playerComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(updateActivePlayer(int)));
    connect(ui->playButton, SIGNAL(clicked()), this, SLOT(sendPlay()));
    connect(ui->powerButton, SIGNAL(clicked()), this, SLOT(sendPower()));
    connect(ui->pauseButton, SIGNAL(clicked()), this, SLOT(sendPause()));
    connect(ui->stopButton, SIGNAL(clicked()), this, SLOT(sendStop()));
    connect(ui->skipBackButton, SIGNAL(clicked()), this, SLOT(sendSkipBack()));
    connect(ui->skipFwdButton, SIGNAL(clicked()), this, SLOT(sendSkipForward()));
    connect(ui->volumeSlider, SIGNAL(sliderMoved(int)), this, SLOT(setVolume(int)));
    connect(ui->progressBar, SIGNAL(sliderMoved(int)), this, SLOT(setNewProgress(int)));
    connect(ui->nowPlayingListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(playlistItemClicked(QListWidgetItem*)));
    connect(ui->albumsListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(listItemClicked(QListWidgetItem*)));
    connect(ui->playlistsListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(listItemClicked(QListWidgetItem*)));
    connect(ui->radiosTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->appsTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->artistsTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->titlesTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->favoritesTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->folderTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->actionClearPlaylist,SIGNAL(triggered()),this,SLOT(sendPlaylistClear()));
    connect(ui->actionSavePlaylist,SIGNAL(triggered()),this,SLOT(savePlaylist()));
    connect(ui->actionAlbumsListIcons, SIGNAL(triggered()),this,SLOT(albumsListIconView()));
    connect(ui->actionMusicLibraryView, SIGNAL(triggered()),this,SLOT(slotMusicLibraryView()));

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(setupScreenMode()));

    ui->radiosTreeWidget->setColumnWidth(0,470);
    ui->appsTreeWidget->setColumnWidth(0,470);
    ui->folderTreeWidget->setColumnWidth(0,470);
    ui->titlesTreeWidget->setColumnWidth(0,470);
    ui->artistsTreeWidget->setColumnWidth(0,470);
    ui->albumLabel->hide();

    bulletIcon = QIcon(BULLETPIXMAPFILE);
    smallBulletIcon = QIcon(SMALLBULLETPIXMAPFILE);
    plusIcon =  QIcon(PLUSPIXMAPFILE);
    minusIcon = QIcon(MINUSPIXMAPFILE);
    powerIcon = QIcon(POWERPIXMAPFILE);
    playIcon = QIcon(PLAYPIXMAPFILE);
    pauseIcon = QIcon(PAUSEPIXMAPFILE);
    stopIcon = QIcon(STOPPIXMAPFILE);
    rewIcon = QIcon(REWPIXMAPFILE);
    fwdIcon = QIcon(FWDPIXMAPFILE);
    noteIcon= QIcon(WHITENOTEPIXMAPFILE);
    emptyAlbumArtIcon = QIcon(EMPTYALBUMARTPIXMAPFILE);

    //----------------------------------------------------------------------
    // Add context menus to  widgets
    //----------------------------------------------------------------------
    // albumsListWidget Actions
    ui->albumsListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* albumsPlayAction=new QAction(noteIcon,"Play",ui->albumsListWidget);
    QAction* albumsAddAction=new QAction(noteIcon,"Add to playlist",ui->albumsListWidget);
    ui->albumsListWidget->addAction(albumsPlayAction);
    ui->albumsListWidget->addAction(albumsAddAction);
    connect(ui->albumsListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // playlistsListWidget Actions
    ui->playlistsListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* playlistsPlayAction=new QAction(noteIcon,"Play",ui->playlistsListWidget);
    QAction* playlistsAddAction=new QAction(noteIcon,"Add to playlist",ui->playlistsListWidget);
    ui->playlistsListWidget->addAction(playlistsPlayAction);
    ui->playlistsListWidget->addAction(playlistsAddAction);
    connect(ui->playlistsListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // nowPlayingListWidget Actions
    ui->nowPlayingListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    //QAction* playlistInfoAction=new QAction(bulletIcon,"Song Info",ui->nowPlayingListWidget);
    QAction* playlistAddFavAction=new QAction(bulletIcon,"Add to favorites",ui->nowPlayingListWidget);
    QAction* playlistDeleteAction=new QAction(bulletIcon,"Delete Track",ui->nowPlayingListWidget);
    //ui->nowPlayingListWidget->addAction(playlistPlayAction);
    //ui->nowPlayingListWidget->addAction(playlistInfoAction);
    ui->nowPlayingListWidget->addAction(playlistAddFavAction);
    ui->nowPlayingListWidget->addAction(playlistDeleteAction);
    connect(ui->nowPlayingListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // radioTreeWidget Actions
    ui->radiosTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* radioPlayAction=new QAction(noteIcon,"Play",ui->radiosTreeWidget);
    QAction* radioAddAction=new QAction(noteIcon,"Add to playlist",ui->radiosTreeWidget);
    QAction* radioAddFavAction=new QAction(noteIcon,"Add to favorites",ui->radiosTreeWidget);
    radioPlayAction->setIconVisibleInMenu(true);
    radioAddAction->setIconVisibleInMenu(true);
    radioAddFavAction->setIconVisibleInMenu(true);
    ui->radiosTreeWidget->addAction(radioPlayAction);
    ui->radiosTreeWidget->addAction(radioAddAction);
    ui->radiosTreeWidget->addAction(radioAddFavAction);
    connect(ui->radiosTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // appTreeWidget Actions
    ui->appsTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* appsPlayAction=new QAction(noteIcon,"Play",ui->appsTreeWidget);
    QAction* appsAddAction=new QAction(noteIcon,"Add to playlist",ui->appsTreeWidget);
    QAction* appsAddFavAction=new QAction(noteIcon,"Add to favorites",ui->appsTreeWidget);
    appsPlayAction->setIconVisibleInMenu(true);
    appsAddAction->setIconVisibleInMenu(true);
    appsAddFavAction->setIconVisibleInMenu(true);
    ui->appsTreeWidget->addAction(appsPlayAction);
    ui->appsTreeWidget->addAction(appsAddAction);
    ui->appsTreeWidget->addAction(appsAddFavAction);
    connect(ui->appsTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // titlesTreeWidget Actions
    ui->titlesTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* titlesPlayAction=new QAction(noteIcon,"Play",ui->titlesTreeWidget);
    QAction* titlesAddAction=new QAction(noteIcon,"Add to playlist",ui->titlesTreeWidget);
    titlesPlayAction->setIconVisibleInMenu(true);
    titlesAddAction->setIconVisibleInMenu(true);
    ui->titlesTreeWidget->addAction(titlesPlayAction);
    ui->titlesTreeWidget->addAction(titlesAddAction);
    connect(ui->titlesTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // folderTreeWidget Actions
    ui->folderTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* folderPlayAction=new QAction(noteIcon,"Play",ui->folderTreeWidget);
    QAction* folderAddAction=new QAction(noteIcon,"Add to playlist",ui->folderTreeWidget);
    QAction* folderAddFavAction=new QAction(noteIcon,"Add to favorites",ui->folderTreeWidget);
    folderPlayAction->setIconVisibleInMenu(true);
    folderAddAction->setIconVisibleInMenu(true);
    ui->folderTreeWidget->addAction(folderPlayAction);
    ui->folderTreeWidget->addAction(folderAddAction);
    ui->folderTreeWidget->addAction(folderAddFavAction);
    connect(ui->folderTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // artistsTreeWidget Actions
    ui->artistsTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* artistsPlayAction=new QAction(noteIcon,"Play",ui->artistsTreeWidget);
    QAction* artistsAddAction=new QAction(noteIcon,"Add to playlist",ui->artistsTreeWidget);
    artistsPlayAction->setIconVisibleInMenu(true);
    artistsAddAction->setIconVisibleInMenu(true);
    ui->artistsTreeWidget->addAction(artistsPlayAction);
    ui->artistsTreeWidget->addAction(artistsAddAction);
    connect(ui->artistsTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // favoritesTreeWidget Actions
    ui->favoritesTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* favPlayAction=new QAction(noteIcon,"Play",ui->favoritesTreeWidget);
    QAction* favAddAction=new QAction(noteIcon,"Add to playlist",ui->favoritesTreeWidget);
    ui->favoritesTreeWidget->addAction(favPlayAction);
    ui->favoritesTreeWidget->addAction(favAddAction);
    connect(ui->favoritesTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    initTitlesTree();
    initArtistsTree();
}


//-----------------------------------------------------------------------------------
// squeezster::updateXmlBrowserList
// Generic xml message decoder
//-----------------------------------------------------------------------------------
void squeezster::updateXmlBrowserList(const QString command, QStringList *parameters, QString divider,QTreeWidget* parentTreeWidget) {
    qDebug() << "updateXmlBrowserList()" << command; // << *parameters;
    isBusy("updateXmlBrowserList()");
    QString keyVal,keyName,title;
    quint16 pStartIndex, pEndIndex, count=0;
    xmlTreeItem* newXmlTreeItem;
    xmlTreeItem* parentTreeWidgetItem;
    QList<QTreeWidgetItem*> *treeWidgetItemList;
    bool isTreeItem = false;
    bool isList = false;
    bool ok = false;

    //---------------------------------------
    // Check if parent is treeWidget or
    // treeWidgetItem
    //---------------------------------------
    treeWidgetItemList = new QList<QTreeWidgetItem*>();
    //qDebug() << "updateXmlBrowserList() Parent type:" <<treeItemParent.typeName();

    if(!parameters->isEmpty() && parameters->first()==QString("items"))
        parameters->takeFirst(); // Take first parameter, e.g. playlist or items

    if ((isList=responseIsList(parameters, &pStartIndex, &pEndIndex))) {
        if (!treeItemParent.isValid()) {
            // Root level
            qDebug() << "updateXmlBrowserList() parent=0" << parentTreeWidget;
        } else {
            qDebug() << "updateXmlBrowserList() parent!=0" << parentTreeWidget;
            if (treeItemParent.userType()==QMetaType::type("QTreeWidget*")) {
                parentTreeWidget=treeItemParent.value<QTreeWidget*>();
                parentTreeWidgetItem=0;
                qDebug() << "xml QTreeWidget:" << parentTreeWidget->objectName();
            } else { //if (treeItemParent.userType()()==QMetaType::type("xmlTreeItem*")) {
                isTreeItem =true;
                parentTreeWidget=0;
                parentTreeWidgetItem=treeItemParent.value<xmlTreeItem*>();
                qDebug() << "xml xmlTreeItem:" << parentTreeWidgetItem->text();
            }
        }
        if (!isTreeItem && pStartIndex==0)
            parentTreeWidget->clear();

        // Only update new parents. Otherwise use cached tree
        if (!isTreeItem || (parentTreeWidgetItem->childCount()==0) ) {
            keyName = "count";
            if(!parameters->isEmpty()) splitKeyPair(parameters->last(),&keyName,&keyVal);
            qDebug() << "updateXmlBrowserList()" << "count: " << keyVal;
            if((count=keyVal.toInt(&ok)) && ok)
                count = 0;
            if (isTreeItem)
                static_cast<xmlTreeItem*>(parentTreeWidgetItem)->addProperty("count",QString::number(count));
            qDebug() << "updateXmlBrowserList() Getting title" ;
            if(!parameters->isEmpty())
                parameters->takeFirst();
            // Get Title
            bool ok=false;
            keyName = "title";
            if(!parameters->isEmpty()) splitKeyPair(parameters->last(),&keyName,&keyVal);
            qDebug() << "updateXmlBrowserList()" << "title: " << keyVal;
            if(ok)
                qDebug() << "xmlBrowserList()" << command << "title: " << title;

            // Find first divider in message:
            ok = false;
            keyName=divider;
            while(!ok &&!parameters->isEmpty())
                keyVal=getKeyValue(parameters->takeFirst(),keyName,&ok);
            newXmlTreeItem=new xmlTreeItem();
            newXmlTreeItem->setIcon(); // Make sure icon is added
            newXmlTreeItem->addProperty("cmd",command);
            newXmlTreeItem->addProperty("title",title);
            qDebug() << "xmlBrowserList() found first divider?" << ok;


            while(ok && !parameters->isEmpty()) {
                // Find image in xml message
                if((keyName.toLower()==QString("image")) || (keyName.toLower()==QString("icon"))) {
                    // Load icon:
                    //qDebug() << "updateXmlBrowserList: Starting to download" << keyVal;
                    downloadImage(newXmlTreeItem,keyVal);
                }
                // Insert keyPair
                //qDebug() << "xmlBrowserList() inserting:" << keyName << keyVal;
                newXmlTreeItem->addProperty(keyName,keyVal);
                // Get new keys
                splitKeyPair(parameters->takeFirst(),&keyName,&keyVal);

                // Search fore next item divider
                if(keyName.toLower()==divider) {
                    // End of item:
                    if(!isTreeItem && newXmlTreeItem->isXmlBrowser())  // Radio top level
                        newXmlTreeItem->addProperty("hasitems","1");
                    else if(isTreeItem && newXmlTreeItem->property("hasitems").isEmpty())
                        newXmlTreeItem->addProperty("isaudio","1");
                    newXmlTreeItem->setIcon();
                    //-------------------------------------
                    // Inserting item in treeWidgetItemList
                    //-------------------------------------
                    *treeWidgetItemList << newXmlTreeItem;
                    // Construct new Item:
                    newXmlTreeItem=new xmlTreeItem();
                    newXmlTreeItem->addProperty("title",title);
                    newXmlTreeItem->addProperty("cmd",command);
                }
            } // parameters empty
            if(!isTreeItem && newXmlTreeItem->isXmlBrowser())  // Radio top level
                newXmlTreeItem->addProperty("hasitems","1");
            else if(isTreeItem && newXmlTreeItem->property("hasitems").isEmpty())
                newXmlTreeItem->addProperty("isaudio","1");
            newXmlTreeItem->setIcon();


            // If search result, sort out items beginning with the search letter:
            if(isTreeItem && !newXmlTreeItem->property("textkey").isEmpty() && !parentTreeWidgetItem->property("search").isEmpty()) {
                if(newXmlTreeItem->property("textkey").toLower()==parentTreeWidgetItem->property("search").toLower())
                    *treeWidgetItemList << newXmlTreeItem;
                else
                    delete newXmlTreeItem;
            } else *treeWidgetItemList << newXmlTreeItem;

            qDebug() << "Inserting in parent";
            if(isTreeItem)
                parentTreeWidgetItem->addChildren(*treeWidgetItemList);
            else
                parentTreeWidget->addTopLevelItems(*treeWidgetItemList);
        } // if !treeItem

        if(isTreeItem)
            (treeItemParent.value<xmlTreeItem*>())->setExpanded(true);

        //////////////////////////////////////
        // Load more list if not complete
        //////////////////////////////////////
        qDebug() << "List:" << count << QString::number(pEndIndex) << QString::number(pStartIndex+1) << QString::number(pEndIndex);
        if(count && (count > pEndIndex)) {
            QStringList newParam;
            if(isTreeItem)
                newParam << static_cast<xmlTreeItem*>(parentTreeWidgetItem)->property("item_id");
            newParam << QString::number(pEndIndex+1) << QString::number(std::min((int)count,pEndIndex+MAXITEMSPERREQUEST));
            send(mySqueezeEnv.activePlayer,command,newParam);
        }
    }
    qDebug() << "xmlBrowserList() exit";
    parameters->clear();
    treeItemParent.clear();
    isBusy();
}


//-----------------------------------------------------------------------------------
// squeezster::updateXmlList
// Generic xml message decoder
//-----------------------------------------------------------------------------------
void squeezster::updateXmlList(const QString listType, QListWidget* listParent, QStringList* itemList, const QString divider, xmlListFlags updateFlag) {
    qDebug() << "updateXmlList()" << divider;// << *itemList;
    QString keyVal,keyName;
    quint16 pStartIndex, pEndIndex, count=0;
    xmlListItem* newXmlListItem;

    bool isList = responseIsList(itemList, &pStartIndex, &pEndIndex);
    if (isList) {
        isBusy("updateXmlList()");
        bool ok = false;
        // Clear ListWidget
        if (pStartIndex==0 && listParent->isWidgetType())
            listParent->clear();
        keyName = divider;
        // Check if count is last, then read
        if(!itemList->isEmpty())
            keyVal=getKeyValue(itemList->last(),"count",&ok);
        else {isBusy();return;}
        count = keyVal.toUInt();

        // Find first divider:
        ok = false;
        while(!ok && !itemList->isEmpty())
            keyVal=getKeyValue(itemList->takeFirst(),keyName,&ok);

        qDebug() << "updateXmlList()" << "count:" << QString::number(count) << "found first divider" << ok;
        newXmlListItem=new xmlListItem(listParent);
        newXmlListItem->addProperty("type",listType);
        while(ok && !itemList->isEmpty()) {
            // Insert keyPair
            newXmlListItem->addProperty(keyName,keyVal);
            splitKeyPair(itemList->takeFirst(),&keyName,&keyVal);
            if(keyName.toLower()==divider) {
                // End of item:
                if (updateFlag==updateIcons) {downloadAlbumImage(newXmlListItem); }
                else newXmlListItem->setIcon();
                //qDebug() << "Inserting item in list.";
                //listParent->addItem(newXmlListItem);
                newXmlListItem=new xmlListItem(listParent);
                newXmlListItem->addProperty("type",listType);
            }
        } // itemList empty
        if (updateFlag==updateIcons) downloadAlbumImage(newXmlListItem);
        else newXmlListItem->setIcon();
        //////////////////////////////////////
        // Load more list if not complete
        //////////////////////////////////////
        qDebug() << "List:" << count << QString::number(pEndIndex) << QString::number(pStartIndex+1) << QString::number(count);
        if(count && (count > pEndIndex)) {
            QStringList newParam;
            newParam << QString::number(pEndIndex+1) << QString::number(std::min(int(count),pEndIndex+MAXITEMSPERREQUEST)) << (listType==QString("albums")?ALBUMTAGS:"");
            qDebug() << "Asking for more items";
            send(mySqueezeEnv.activePlayer,listType,newParam);
        }

        // Re-apply status icons for current song
        if(listParent==ui->nowPlayingListWidget)
            updateNowPlayingIcons();

        isBusy();
        listItemParent.clear();
    }
}

//----------------------------------------------
// HTTP prefix generator
//----------------------------------------------
QString squeezster::httpPrefixGen(UserSettings lsettings)
{
    QString prefix = "http://";
//    if(!lsettings.username().isEmpty())
//        prefix+=lsettings.username()+':'+lsettings.password()+'@';
    return(prefix);
}


//----------------------------------------------
// Server Discovery
//----------------------------------------------
void squeezster::serverDiscovery()
{
    qDebug() << "Server Discovery";

    //QByteArray datagram = "eIPAD\0NAME\0JSON\0";
    QByteArray datagram;
    datagram.resize(18);
    datagram[0] = 0xd;   // 0. Discovery
    datagram[1] = 0x00;  // 1. Reserved
    datagram[2] = 0x01;  // 2. Device ID
    datagram[3] = 0x11;  // 3. Firmware rev.
    datagram[4] = 0x00;
    datagram[5] = 0x00;
    datagram[6] = 0x00;
    datagram[7] = 0x00;
    datagram[8] = 0x00;
    datagram[9] = 0x00;
    datagram[10] = 0x00;
    datagram[11] = 0x00; // 4.-11. Reserved
    datagram[12] = 0x00;
    datagram[13] = 0x00;
    datagram[14] = 0x00;
    datagram[15] = 0x00;
    datagram[16] = 0x00;
    datagram[17] = 0x00; // 12.-17. MAC Address

    serverUdpSocket = new QUdpSocket(this);
    serverUdpSocket->writeDatagram(datagram.data(), datagram.size(),
                                   QHostAddress::Broadcast, DISCOVERYPORT_CS);


    clientUdpSocket = new QUdpSocket(this);
    clientUdpSocket->bind(DISCOVERYPORT_SC, QUdpSocket::ShareAddress);

    connect(clientUdpSocket, SIGNAL(readyRead()),
            this, SLOT(receiveUdpDatagram()));
}

void squeezster::receiveUdpDatagram()
{
    qDebug() << "receiveUdpDatagram()";

    QByteArray datagram;
    datagram.resize(clientUdpSocket->pendingDatagramSize());
    QHostAddress sender;
    quint16 senderPort;
    int i;

    while(clientUdpSocket->hasPendingDatagrams())
    {
        clientUdpSocket->readDatagram(datagram.data(), datagram.size(),
                                      &sender, &senderPort);

        char *data=datagram.data();
        for (i=0;i<17;i++)
            qDebug() << data++ << "\n" << sender.toString() << QString::number(senderPort);
    }
    disconnect(clientUdpSocket);
}

//----------------------------------------------
// Squeezster deconstructor
//----------------------------------------------
squeezster::~squeezster()
{
    qDebug() << "Exit main()";
    delete ui;
}

