#!/usr/bin/python
# -*- coding: utf-8 -*-

"""
	SqueezeControl: Control your Squeezeboxes from your N900.

	Copyright 2011 Mikkel Munch Mortensen <3xm@detfalskested.dk> 

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import sys, os, sqlite3
from PySide.QtCore import *
from PySide.QtGui import *
from PySide.QtMaemo5 import *

from pysqueezecenter.server import Server
from pysqueezecenter.player import Player

app_path = os.environ['HOME'] + '/.squeezecontrol/'
app_version = '0.1.2'

def format_seconds(seconds):
	seconds = int(seconds)
	minutes, hours = 0, None
	if seconds >= 60:
		minutes = seconds / 60
		seconds = seconds % 60
	if minutes >= 60:
		hours = minutes / 60
		minutes = minutes % 60
	output = str(seconds).zfill(2)
	if minutes != None:
		output = str(minutes).zfill(2) + ':' + output
	if hours != None:
		output = str(hours).zfill(2) + ':' + output
	return output

class PlayerWindow(QMainWindow):
	timer = None
	timer_count = 5
	elapsed = 0
	remaining = 0
	duration = 0
	track_info = ''
	
	def __init__(self, parent, player):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle(player.get_name())
		self.player = player
		self.render()
	
	def closeEvent(self, event):
		self.clear_timer()
	
	def can_pause(self):
		return self.player.get_track_path().startswith('file:')
	
	def can_skip_forwards(self):
		path = self.player.get_track_path()
		return path.startswith('file:') or path.startswith('lfm:')
	
	def can_skip_backwards(self):
		return self.player.get_track_path().startswith('file:')
	
	def update_display(self, force_update = False):
		if force_update or self.timer_count == 5 or self.remaining == 0:
			old_track_info = self.track_info
			self.track_info = self.player.get_track_title()
			album = self.player.get_track_album()
			if album != '':
				self.track_info += ' from %s' % album
			artist = self.player.get_track_artist()
			if artist != '':
				self.track_info += ' by %s' % artist
			self.track_info = self.track_info.decode('utf-8')
			if old_track_info != self.track_info:
				visible = self.can_skip_forwards()
				self.progress_bar.setVisible(visible)
				self.remaining_label.setVisible(visible)
			self.elapsed = self.player.get_time_elapsed()
			if self.remaining_label:
				self.remaining = self.player.get_time_remaining()
			if self.progress_bar:
				self.duration = self.player.get_track_duration()
			self.timer_count = 0
		else:
			self.timer_count += 1
			self.elapsed += 1
			self.remaining -= 1
		self.track_info_label.setText(self.track_info)
		self.elapsed_label.setText(format_seconds(self.elapsed))
		self.remaining_label.setText(format_seconds(self.remaining))
		if self.can_skip_forwards():
			position = int(self.elapsed / self.duration * 1000)
			self.progress_bar.setValue(position)
		self.update()
	
	def render(self):
		self.main_widget = QWidget(self)
		vbox = QVBoxLayout(self.main_widget)
		
		# Title and artist
		self.track_info_label = QLabel('')
		self.track_info_label.setWordWrap(True)
		self.track_info_label.setAlignment(Qt.AlignCenter)
		vbox.addWidget(self.track_info_label)
		
		# Time and progress
		time = QWidget(self.main_widget)
		hbox = QHBoxLayout(time)
		
		self.elapsed_label = QLabel('')
		hbox.addWidget(self.elapsed_label)
		self.progress_bar = QProgressBar()
		self.progress_bar.setTextVisible(False)
		self.progress_bar.setRange(0, 1000)
		hbox.addWidget(self.progress_bar)	
		self.remaining_label = QLabel('')
		hbox.addWidget(self.remaining_label)
		if not self.can_skip_forwards():
			self.progress_bar.setVisible(False)
			self.remaining_label.setVisible(False)
		vbox.addWidget(time)
		
		# Main controls
		controls = QWidget(self.main_widget)
		hbox = QHBoxLayout(controls)
		
		prev = QPushButton('<<')
		self.connect(prev, SIGNAL('clicked()'), self.prev)
		hbox.addWidget(prev)
		
		self.play_button = QPushButton('Playing' if self.player.get_mode() == 'play' else 'Paused')
		self.connect(self.play_button, SIGNAL('clicked()'), self.play_toggle)
		hbox.addWidget(self.play_button)
		
		next = QPushButton('>>')
		self.connect(next, SIGNAL('clicked()'), self.next)
		hbox.addWidget(next)
		vbox.addWidget(controls)
		
		# Volume slider
		vol = QSlider(Qt.Horizontal)
		vol.setValue(self.player.get_volume())
		vbox.addWidget(vol)
		vol.valueChanged.connect(self.change_vol)
		
		self.setCentralWidget(self.main_widget)
		
		# Update display and make a timer update it every now and then
		self.update_display()
		self.start_timer()
	
	def start_timer(self):
		if self.timer == None:
			self.timer = QTimer(self)
			self.timer.timeout.connect(self.update_display)
			self.timer.start(1000)
	
	def clear_timer(self):
		if self.timer != None:
			self.timer.stop()
			self.timer = None
	
	def test(self):
		print 'close'
	
	def prev(self):
		self.player.prev()
		self.update_display(True)
	
	def next(self):
		self.player.next()
		self.update_display(True)
	
	def play_toggle(self):
		self.player.toggle()
		mode = self.player.get_mode()
		self.play_button.setText('Playing' if mode == 'play' else 'Paused')
		if mode == 'play':
			self.start_timer()
		else:
			self.clear_timer()
	
	def change_vol(self):
		self.player.set_volume(self.sender().value())

class StartWindow(QMainWindow):
	def __init__(self):
		QMainWindow.__init__(self)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('SqueezeControl')
		
		# Make sure the app folder exists.
		if not os.path.exists(app_path):
			os.mkdir(app_path)
		
		# Create SQLite database and structure.
		self.db = sqlite3.connect(app_path + 'db.sqlite')
		self.cursor = self.db.cursor()
		self.cursor.execute('''
		create table if not exists settings
		(key, value)
		''')
		
	def get_setting(self, key):
		params = (key,)
		self.cursor.execute('select value from settings where key=?', params)
		result = self.cursor.fetchone()
		return result[0] if result else None
	
	def save_setting(self, key, value):
		params = (key,)
		self.cursor.execute('delete from settings where key=?', params)
		params = (key,value)
		self.cursor.execute('insert into settings values (?, ?)', params)
		self.db.commit()
	
	def render(self):
		container = QWidget(self)
		vbox = QVBoxLayout(container)
		
		for text, action in [
			('Select player', self.open_player_selection,),
			('Settings', self.open_settings,),
			('About', self.open_about,),
		]:
			button = QPushButton(text)
			self.connect(button, SIGNAL('clicked()'), action)
			vbox.addWidget(button)
		self.setCentralWidget(container)
	
	def open_player_selection(self):
		select_window = SelectPlayerWindow(self)
		select_window.show()
	
	def open_settings(self):
		settings_window = SettingsWindow(self)
		settings_window.show()
	
	def open_about(self):
		about_window = AboutWindow(self)
		about_window.show()

class AboutWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('About SqueezeControl %s' % app_version)
		self.render()
	
	def render(self):
		container = QWidget()
		vbox = QVBoxLayout(container)
		label = QLabel('This version of SqueezeControl was made while listening to cheesy piano music. A huge shoutout to JingleManSweep who made the pysqueezecenter wrapper library for Python. Also thanks to Dag Liodden -- developer of SqueezeControl for Android -- for letting me use the name "SqueezeControl".\n\nCopyright 2010, 2011 JingleManSweep and Mikkel Munch Mortensen. Released under the terms of the GNU General Public License version 3.')
		label.setWordWrap(True)
		vbox.addWidget(label)
		
		self.setCentralWidget(container)

class SettingsWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('Settings')
		self.main = parent
		self.fields = {}
		self.render()
	
	def render(self):
		container = QWidget(self)
		container.setMinimumSize(792,0)
		vbox = QVBoxLayout(container)
		vbox.setSpacing(25)
		
		label = QLabel('For SqueezeControl to work, it needs to know how to connect to your Squeezebox Server. Please fill out the fields below to match your server setup.')
		label.setWordWrap(True)

		vbox.addWidget(label)
		
		vbox.addWidget(self.render_field('server', 'Server name', 'E.g. an IP address or something like myservername.com'))
		vbox.addWidget(self.render_field('port', 'Server CLI port', 'Optional. Leave blank if you use the default port (9090).', validator = QIntValidator()))
		vbox.addWidget(self.render_field('username', 'Username', 'Optional. Leave blank if you haven\'t enabled server authenticion.'))
		vbox.addWidget(self.render_field('password', 'Password', 'Optional. Leave blank if you haven\'t enabled server authenticion.', echo_mode = QLineEdit.EchoMode.Password))
		
		button = QPushButton('Save')
		self.connect(button, SIGNAL('clicked()'), self.save_settings)
		vbox.addWidget(button)
		
		scroller = QScrollArea()
		scroller.setWidget(container)
		self.setCentralWidget(scroller)
	
	def render_field(self, option_name, label_text, help_text, validator = None, echo_mode = None):
		vwidget = QWidget()
		vbox = QVBoxLayout(vwidget)
		vbox.setContentsMargins(0, 0, 0, 0)
		vbox.setSpacing(0)
				
		hwidget = QWidget()
		hbox = QHBoxLayout(hwidget)
		hbox.setContentsMargins(0, 0, 0, 0)
		
		label = QLabel(label_text)
		label.setFixedWidth(160)
		hbox.addWidget(label)
		
		field = QLineEdit()
		if validator:
			field.setValidator(validator)
		if echo_mode:
			field.setEchoMode(echo_mode)
		option = self.main.get_setting(option_name)
		option = option if option else ''
		field.setText(str(option))
		hbox.addWidget(field)
		self.fields[option_name] = field
		
		vbox.addWidget(hwidget)
		
		help = QLabel(help_text)
		help.setAlignment(Qt.AlignRight)
		vbox.addWidget(help)
		
		return vwidget
	
	def save_settings(self):
		for setting in ['server', 'port', 'username', 'password']:
			self.main.save_setting(setting, self.fields[setting].text())
		info = QMaemo5InformationBox()
		info.information(self, 'Settings saved.', 2500)

class SelectPlayerWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.main = parent
		self.render()
	
	def render(self):
		settings = {}
		for value in ['server', 'port', 'username', 'password']:
			settings[value] = self.main.get_setting(value)
		if settings['server']:
			self.render_player_list(settings)
		else:
			self.setWindowTitle('No server set')
			container = QWidget(self)
			vbox = QVBoxLayout(container)
			label = QLabel('Please go back to the main screen, go to "Settings" and set what server to use.')
			label.setWordWrap(True)
			vbox.addWidget(label)
			self.setCentralWidget(container)
		
	def render_player_list(self, settings):
		try:
			settings['port'] = int(settings['port'])
		except:
			settings['port'] = 9090
		try:
			if not settings['port']:
				settings['port'] = 9090
			if settings['username']:
				self.sc = Server(hostname=settings['server'], port=settings['port'], username=str(settings['username']), password=str(settings['password']))
			else:
				self.sc = Server(hostname=settings['server'], port=settings['port'])
			self.sc.connect()
		except Exception:
			self.setWindowTitle('Connection error')
			label = QLabel('Unable to connect to Squeezebox server at %s. Is the server accessible via your current network? Did you enter the right server settings?' % settings['server'])
			label.setWordWrap(True)
			self.setCentralWidget(label)
		else:
			self.setWindowTitle('Select player')
			self.main_widget = QWidget(self)
			self.main_widget.setMinimumSize(792,0)
			vbox = QVBoxLayout(self.main_widget)
			
			plist = {}
			for p in self.sc.get_players():
				name = p.get_name()
				plist[name] = QPushButton(name)
				plist[name].player = p
				self.connect(plist[name], SIGNAL('clicked()'), self.select_player)
				vbox.addWidget(plist[name])
	
			vbox.setSizeConstraint(QLayout.SetMinAndMaxSize)
			self.scroller = QScrollArea(self)
			self.scroller.setWidget(self.main_widget)
			self.setCentralWidget(self.scroller)
	
	def select_player(self):
		player = self.sender().player
		player_window = PlayerWindow(self, player)
		player_window.show()

if __name__ == '__main__':
	app = QApplication(sys.argv)
	w = StartWindow()
	w.render()
	w.show()
	app.exec_()
	sys.exit()

