// This file is part of the SpeedCrunch project
// Copyright (C) 2005-2006 Johan Thelin <e8johan@gmail.com>
// Copyright (C) 2007-2008 Helder Correia <helder.pereira.correia@gmail.com>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.


#include "portraitportraitKeypad.hxx"

#include <QApplication>
#include <QGridLayout>
#include <QLocale>
#if QT_VERSION >= 0x040400 && defined(Q_WS_MAC) && !defined(QT_NO_STYLE_MAC)
#include <QMacStyle>
#endif
#include <QPushButton>
#include <QStyle>
#include <QStyleOptionButton>


struct portraitKeypad::Private
{
    // parent
    portraitKeypad * p;

    // attributes
    char radixChar;

    // widgets
    QPushButton * key0;
    QPushButton * key1;
    QPushButton * key2;
    QPushButton * key3;
    QPushButton * key4;
    QPushButton * key5;
    QPushButton * key6;
    QPushButton * key7;
    QPushButton * key8;
    QPushButton * key9;
    QPushButton * keyAcos;
    QPushButton * keyAdd;
    QPushButton * keyAns;
    QPushButton * keyAsin;
    QPushButton * keyAtan;
    QPushButton * keyC;
    QPushButton * keyCos;
    QPushButton * keyDiv;
    QPushButton * keyDot;
    QPushButton * keyEE;
    QPushButton * keyEq;
    QPushButton * keyExp;
    QPushButton * keyFact;
    QPushButton * keyLn;
    QPushButton * keyLPar;
    QPushButton * keyMul;
    QPushButton * keyPerc;
    QPushButton * keyPi;
    QPushButton * keyRaise;
    QPushButton * keyRPar;
    QPushButton * keySin;
    QPushButton * keySqrt;
    QPushButton * keySub;
    QPushButton * keyTan;
    QPushButton * keyX;
    QPushButton * keyXEq;

    // methods
    void createButtons();
    void disableButtonFocus();
    void layoutButtons();
    void setButtonTooltips();
    void setUpButtonPressedSignal();
    void sizeButtons();
};


// public

portraitKeypad::portraitKeypad( char radixChar, QWidget * parent )
  : QWidget( parent ), d( new portraitKeypad::Private )
{
  d->p = this;

  if ( radixChar == 'C' )
    d->radixChar = QLocale().decimalPoint().toAscii();
  else
    d->radixChar = radixChar;

  d->createButtons();
  d->sizeButtons();
  d->layoutButtons();
  d->setButtonTooltips();
  d->setUpButtonPressedSignal();
  d->disableButtonFocus();
  setLayoutDirection( Qt::LeftToRight );
}


portraitKeypad::~portraitKeypad()
{
  delete d;
}


char portraitKeypad::radixChar() const
{
  return d->radixChar;
}


// public slots

void portraitKeypad::setRadixChar( char c )
{
  if ( c == 'C' )
    c = QLocale().decimalPoint().toAscii();
  if ( d->radixChar != c )
  {
    d->radixChar = c;
    d->keyDot->setText( QString( c ) );
  }
}


void portraitKeypad::retranslateText()
{
  d->setButtonTooltips();;
}


// protected slots

void portraitKeypad::key0Pressed()     { emit buttonPressed( portraitKeypad::Key0         ); };
void portraitKeypad::key1Pressed()     { emit buttonPressed( portraitKeypad::Key1         ); };
void portraitKeypad::key2Pressed()     { emit buttonPressed( portraitKeypad::Key2         ); };
void portraitKeypad::key3Pressed()     { emit buttonPressed( portraitKeypad::Key3         ); };
void portraitKeypad::key4Pressed()     { emit buttonPressed( portraitKeypad::Key4         ); };
void portraitKeypad::key5Pressed()     { emit buttonPressed( portraitKeypad::Key5         ); };
void portraitKeypad::key6Pressed()     { emit buttonPressed( portraitKeypad::Key6         ); };
void portraitKeypad::key7Pressed()     { emit buttonPressed( portraitKeypad::Key7         ); };
void portraitKeypad::key8Pressed()     { emit buttonPressed( portraitKeypad::Key8         ); };
void portraitKeypad::key9Pressed()     { emit buttonPressed( portraitKeypad::Key9         ); };
void portraitKeypad::keyAcosPressed()  { emit buttonPressed( portraitKeypad::KeyAcos      ); };
void portraitKeypad::keyAddPressed()   { emit buttonPressed( portraitKeypad::KeyPlus      ); };
void portraitKeypad::keyAnsPressed()   { emit buttonPressed( portraitKeypad::KeyAns       ); };
void portraitKeypad::keyAsinPressed()  { emit buttonPressed( portraitKeypad::KeyAsin      ); };
void portraitKeypad::keyAtanPressed()  { emit buttonPressed( portraitKeypad::KeyAtan      ); };
void portraitKeypad::keyCosPressed()   { emit buttonPressed( portraitKeypad::KeyCos       ); };
void portraitKeypad::keyCPressed()     { emit buttonPressed( portraitKeypad::KeyClear     ); };
void portraitKeypad::keyDivPressed()   { emit buttonPressed( portraitKeypad::KeyDivide    ); };
void portraitKeypad::keyDotPressed()   { emit buttonPressed( portraitKeypad::KeyRadixChar ); };
void portraitKeypad::keyEEPressed()    { emit buttonPressed( portraitKeypad::KeyEE        ); };
void portraitKeypad::keyEqPressed()    { emit buttonPressed( portraitKeypad::KeyEquals    ); };
void portraitKeypad::keyExpPressed()   { emit buttonPressed( portraitKeypad::KeyExp       ); };
void portraitKeypad::keyFactPressed()  { emit buttonPressed( portraitKeypad::KeyFactorial ); };
void portraitKeypad::keyLnPressed()    { emit buttonPressed( portraitKeypad::KeyLn        ); };
void portraitKeypad::keyLParPressed()  { emit buttonPressed( portraitKeypad::KeyLeftPar   ); };
void portraitKeypad::keyMulPressed()   { emit buttonPressed( portraitKeypad::KeyTimes     ); };
void portraitKeypad::keyPercPressed()  { emit buttonPressed( portraitKeypad::KeyPercent   ); };
void portraitKeypad::keyPiPressed()    { emit buttonPressed( portraitKeypad::KeyPi        ); };
void portraitKeypad::keyRaisePressed() { emit buttonPressed( portraitKeypad::KeyRaise     ); };
void portraitKeypad::keyRParPressed()  { emit buttonPressed( portraitKeypad::KeyRightPar  ); };
void portraitKeypad::keySinPressed()   { emit buttonPressed( portraitKeypad::KeySin       ); };
void portraitKeypad::keySqrtPressed()  { emit buttonPressed( portraitKeypad::KeySqrt      ); };
void portraitKeypad::keySubPressed()   { emit buttonPressed( portraitKeypad::KeyMinus     ); };
void portraitKeypad::keyTanPressed()   { emit buttonPressed( portraitKeypad::KeyTan       ); };
void portraitKeypad::keyXPressed()     { emit buttonPressed( portraitKeypad::KeyX         ); };
void portraitKeypad::keyXEqPressed()   { emit buttonPressed( portraitKeypad::KeyXEquals   ); };


// private

void portraitKeypad::Private::createButtons()
{
  key0      = new QPushButton( "0",    p );
  key1      = new QPushButton( "1",    p );
  key2      = new QPushButton( "2",    p );
  key3      = new QPushButton( "3",    p );
  key4      = new QPushButton( "4",    p );
  key5      = new QPushButton( "5",    p );
  key6      = new QPushButton( "6",    p );
  key7      = new QPushButton( "7",    p );
  key8      = new QPushButton( "8",    p );
  key9      = new QPushButton( "9",    p );
  keyAcos   = new QPushButton( "acos", p );
  keyAdd    = new QPushButton( "+",    p );
  keyAns    = new QPushButton( "ans",  p );
  keyAsin   = new QPushButton( "asin", p );
  keyAtan   = new QPushButton( "atan", p );
  keyC      = new QPushButton( "C",    p );
  keyCos    = new QPushButton( "cos",  p );
  keyDiv    = new QPushButton( "/",    p );
  keyEE     = new QPushButton( "E",    p );
  keyEq     = new QPushButton( "=",    p );
  keyExp    = new QPushButton( "exp",  p );
  keyFact   = new QPushButton( "!",    p );
  keyLn     = new QPushButton( "ln",   p );
  keyLPar   = new QPushButton( "(",    p );
  keyMul    = new QPushButton( "*",    p );
  keyPerc   = new QPushButton( "%",    p );
  keyRaise  = new QPushButton( "^",    p );
  keyRPar   = new QPushButton( ")",    p );
  keySin    = new QPushButton( "sin",  p );
  keySub    = new QPushButton( "-",    p );
  keyTan    = new QPushButton( "tan",  p );
  keyXEq    = new QPushButton( "x=",   p );
  keyX      = new QPushButton( "x",    p );

  keyDot    = new QPushButton( QString( radixChar ),     p );
  keyPi     = new QPushButton( QString::fromUtf8( "π" ), p );
  keySqrt   = new QPushButton( QString::fromUtf8( "√" ), p );

  QFont f = key0->font();
  f.setBold( true );
  f.setPointSize( f.pointSize() + 3 );
  key0->setFont( f );
  key1->setFont( f );
  key2->setFont( f );
  key3->setFont( f );
  key4->setFont( f );
  key5->setFont( f );
  key6->setFont( f );
  key7->setFont( f );
  key8->setFont( f );
  key9->setFont( f );
  keyEq->setFont( f );
  keyMul->setFont( f );
  keyDiv->setFont( f );
  keyAdd->setFont( f );
  keySub->setFont( f );
  keyDot->setFont( f );
}


void portraitKeypad::Private::disableButtonFocus()
{
  key0->setFocusPolicy    ( Qt::NoFocus );
  key1->setFocusPolicy    ( Qt::NoFocus );
  key2->setFocusPolicy    ( Qt::NoFocus );
  key3->setFocusPolicy    ( Qt::NoFocus );
  key4->setFocusPolicy    ( Qt::NoFocus );
  key5->setFocusPolicy    ( Qt::NoFocus );
  key6->setFocusPolicy    ( Qt::NoFocus );
  key7->setFocusPolicy    ( Qt::NoFocus );
  key8->setFocusPolicy    ( Qt::NoFocus );
  key9->setFocusPolicy    ( Qt::NoFocus );
  keyAcos->setFocusPolicy ( Qt::NoFocus );
  keyAdd->setFocusPolicy  ( Qt::NoFocus );
  keyAns->setFocusPolicy  ( Qt::NoFocus );
  keyAsin->setFocusPolicy ( Qt::NoFocus );
  keyAtan->setFocusPolicy ( Qt::NoFocus );
  keyCos->setFocusPolicy  ( Qt::NoFocus );
  keyC->setFocusPolicy    ( Qt::NoFocus );
  keyDiv->setFocusPolicy  ( Qt::NoFocus );
  keyDot->setFocusPolicy  ( Qt::NoFocus );
  keyEE->setFocusPolicy   ( Qt::NoFocus );
  keyEq->setFocusPolicy   ( Qt::NoFocus );
  keyExp->setFocusPolicy  ( Qt::NoFocus );
  keyFact->setFocusPolicy ( Qt::NoFocus );
  keyLn->setFocusPolicy   ( Qt::NoFocus );
  keyLPar->setFocusPolicy ( Qt::NoFocus );
  keyMul->setFocusPolicy  ( Qt::NoFocus );
  keyPerc->setFocusPolicy ( Qt::NoFocus );
  keyPi->setFocusPolicy   ( Qt::NoFocus );
  keyRaise->setFocusPolicy( Qt::NoFocus );
  keyRPar->setFocusPolicy ( Qt::NoFocus );
  keySin->setFocusPolicy  ( Qt::NoFocus );
  keySqrt->setFocusPolicy ( Qt::NoFocus );
  keySub->setFocusPolicy  ( Qt::NoFocus );
  keyTan->setFocusPolicy  ( Qt::NoFocus );
  keyXEq->setFocusPolicy  ( Qt::NoFocus );
  keyX->setFocusPolicy    ( Qt::NoFocus );
}


void portraitKeypad::Private::layoutButtons()
{
  int layoutSpacing = 3;

#if QT_VERSION >= 0x040400 && defined(Q_WS_MAC) && !defined(QT_NO_STYLE_MAC)
  // Workaround for a layouting bug in QMacStyle, Qt 4.4.0. Buttons would overlap
  if ( qobject_cast<QMacStyle *>(p->style()) )
    layoutSpacing = -1;
#endif

  QHBoxLayout * layout = new QHBoxLayout( p );
  layout->setMargin( 3 );
  layout->setSpacing( layoutSpacing );

  layout->addWidget( key0,      3, 0 );
  layout->addWidget( key1,      2, 0 );
  layout->addWidget( key2,      2, 1 );
  layout->addWidget( key3,      2, 2 );
  layout->addWidget( key4,      1, 0 );
  layout->addWidget( key5,      1, 1 );
  layout->addWidget( key6,      1, 2 );
  layout->addWidget( key7,      0, 0 );
  layout->addWidget( key8,      0, 1 );
  layout->addWidget( key9,      0, 2 );
//  layout->addWidget( keyAcos,   2, 8 );
//  layout->addWidget( keyAdd,    3, 3 );
//  layout->addWidget( keyAns,    1, 6 );
//  layout->addWidget( keyAsin,   1, 8 );
//  layout->addWidget( keyAtan,   3, 8 );
//  layout->addWidget( keyC,      0, 4 );
//  layout->addWidget( keyCos,    2, 7 );
//  layout->addWidget( keyDiv,    0, 3 );
//  layout->addWidget( keyDot,    3, 1 );
//  layout->addWidget( keyEE,     1, 4 );
//  layout->addWidget( keyEq,     3, 2 );
//  layout->addWidget( keyExp,    0, 7 );
//  layout->addWidget( keyFact,   3, 5 );
//  layout->addWidget( keyLn,     0, 8 );
//  layout->addWidget( keyLPar,   2, 4 );
//  layout->addWidget( keyMul,    1, 3 );
//  layout->addWidget( keyPerc,   3, 4 );
//  layout->addWidget( keyPi,     0, 6 );
//  layout->addWidget( keyRaise,  1, 5 );
//  layout->addWidget( keyRPar,   2, 5 );
//  layout->addWidget( keySin,    1, 7 );
//  layout->addWidget( keySqrt,   0, 5 );
//  layout->addWidget( keySub,    2, 3 );
//  layout->addWidget( keyTan,    3, 7 );
//  layout->addWidget( keyX,      2, 6 );
//  layout->addWidget( keyXEq,    3, 6 );
}


void portraitKeypad::Private::setButtonTooltips()
{
  keyAcos->setToolTip( portraitKeypad::tr( "Inverse cosine"      ) );
  keyAns->setToolTip ( portraitKeypad::tr( "The last result"     ) );
  keyAsin->setToolTip( portraitKeypad::tr( "Inverse sine"        ) );
  keyAtan->setToolTip( portraitKeypad::tr( "Inverse tangent"     ) );
  keyC->setToolTip   ( portraitKeypad::tr( "Clear expression"    ) );
  keyCos->setToolTip ( portraitKeypad::tr( "Cosine"              ) );
  keyEE->setToolTip  ( portraitKeypad::tr( "Scientific notation" ) );
  keyExp->setToolTip ( portraitKeypad::tr( "Exponential"         ) );
  keyLn->setToolTip  ( portraitKeypad::tr( "Natural logarithm"   ) );
  keySin->setToolTip ( portraitKeypad::tr( "Sine"                ) );
  keySqrt->setToolTip( portraitKeypad::tr( "Square root"         ) );
  keyTan->setToolTip ( portraitKeypad::tr( "Tangent"             ) );
  keyXEq->setToolTip ( portraitKeypad::tr( "Assign variable x"   ) );
  keyX->setToolTip   ( portraitKeypad::tr( "The variable x"      ) );
}


void portraitKeypad::Private::setUpButtonPressedSignal()
{
  connect( key0,     SIGNAL( clicked() ), p, SLOT( key0Pressed()    ) );
  connect( key1,     SIGNAL( clicked() ), p, SLOT( key1Pressed()    ) );
  connect( key2,     SIGNAL( clicked() ), p, SLOT( key2Pressed()    ) );
  connect( key3,     SIGNAL( clicked() ), p, SLOT( key3Pressed()    ) );
  connect( key4,     SIGNAL( clicked() ), p, SLOT( key4Pressed()    ) );
  connect( key5,     SIGNAL( clicked() ), p, SLOT( key5Pressed()    ) );
  connect( key6,     SIGNAL( clicked() ), p, SLOT( key6Pressed()    ) );
  connect( key7,     SIGNAL( clicked() ), p, SLOT( key7Pressed()    ) );
  connect( key8,     SIGNAL( clicked() ), p, SLOT( key8Pressed()    ) );
  connect( key9,     SIGNAL( clicked() ), p, SLOT( key9Pressed()    ) );
  connect( keyAcos,  SIGNAL( clicked() ), p, SLOT( keyAcosPressed() ) );
  connect( keyAdd,   SIGNAL( clicked() ), p, SLOT( keyAddPressed()  ) );
  connect( keyAns,   SIGNAL( clicked() ), p, SLOT( keyAnsPressed()  ) );
  connect( keyAsin,  SIGNAL( clicked() ), p, SLOT( keyAsinPressed() ) );
  connect( keyAtan,  SIGNAL( clicked() ), p, SLOT( keyAtanPressed() ) );
  connect( keyC,     SIGNAL( clicked() ), p, SLOT( keyCPressed()    ) );
  connect( keyCos,   SIGNAL( clicked() ), p, SLOT( keyCosPressed()  ) );
  connect( keyDiv,   SIGNAL( clicked() ), p, SLOT( keyDivPressed()  ) );
  connect( keyEE,    SIGNAL( clicked() ), p, SLOT( keyEEPressed()   ) );
  connect( keyEq,    SIGNAL( clicked() ), p, SLOT( keyEqPressed()   ) );
  connect( keyExp,   SIGNAL( clicked() ), p, SLOT( keyExpPressed()  ) );
  connect( keyFact,  SIGNAL( clicked() ), p, SLOT( keyFactPressed() ) );
  connect( keyLn,    SIGNAL( clicked() ), p, SLOT( keyLnPressed()   ) );
  connect( keyLPar,  SIGNAL( clicked() ), p, SLOT( keyLParPressed() ) );
  connect( keyMul,   SIGNAL( clicked() ), p, SLOT( keyMulPressed()  ) );
  connect( keyPerc,  SIGNAL( clicked() ), p, SLOT( keyPercPressed() ) );
  connect( keyRaise, SIGNAL( clicked() ), p, SLOT( keyRaisePressed()) );
  connect( keyRPar,  SIGNAL( clicked() ), p, SLOT( keyRParPressed() ) );
  connect( keySin,   SIGNAL( clicked() ), p, SLOT( keySinPressed()  ) );
  connect( keySub,   SIGNAL( clicked() ), p, SLOT( keySubPressed()  ) );
  connect( keyTan,   SIGNAL( clicked() ), p, SLOT( keyTanPressed()  ) );
  connect( keyXEq,   SIGNAL( clicked() ), p, SLOT( keyXEqPressed()  ) );
  connect( keyX,     SIGNAL( clicked() ), p, SLOT( keyXPressed()    ) );
  connect( keyDot,   SIGNAL( clicked() ), p, SLOT( keyDotPressed()  ) );
  connect( keyPi,    SIGNAL( clicked() ), p, SLOT( keyPiPressed()   ) );
  connect( keySqrt,  SIGNAL( clicked() ), p, SLOT( keySqrtPressed() ) );
}


void portraitKeypad::Private::sizeButtons()
{
  // The same font in all buttons, so just pick one
  QFontMetrics fm = key0->fontMetrics();

  int maxWidth = fm.width( keyAcos->text() );
  int textHeight = qMax( fm.lineSpacing(), 14 );
  //int ls = fm.lineSpacing();

  QStyle::ContentsType type = QStyle::CT_ToolButton;
  QStyleOptionButton opt;
  opt.initFrom(keyAcos);
  QSize size = keyAcos->style()->sizeFromContents( type, &opt,
    QSize( int(maxWidth), int(textHeight) ).expandedTo( QApplication::globalStrut() ), keyAcos );

#ifdef Q_WS_X11
  // we would like to use the button size as indicated by the widget style,
  // but in some cases, e.g. KDE's Plastik or Oxygen, another few pixels
  // (typically 5) are added as the content margin, thereby making the button
  // incredibly wider than necessary
  // workaround: take only the hinted height, adjust the width ourselves (with our own margin)
  maxWidth += 6;
  int hh = size.height();
  int ww = hh * 162 / 100; // use golden ratio
  size = QSize(qMax(ww, maxWidth), hh);
#endif

  // limit the size of the buttons
  key0->setMaximumSize     ( size ); key0->setMinimumSize     ( size );
  key1->setMaximumSize     ( size ); key1->setMinimumSize     ( size );
  key2->setMaximumSize     ( size ); key2->setMinimumSize     ( size );
  key3->setMaximumSize     ( size ); key3->setMinimumSize     ( size );
  key4->setMaximumSize     ( size ); key4->setMinimumSize     ( size );
  key5->setMaximumSize     ( size ); key5->setMinimumSize     ( size );
  key6->setMaximumSize     ( size ); key6->setMinimumSize     ( size );
  key7->setMaximumSize     ( size ); key7->setMinimumSize     ( size );
  key8->setMaximumSize     ( size ); key8->setMinimumSize     ( size );
  key9->setMaximumSize     ( size ); key9->setMinimumSize     ( size );
  keyAcos->setMaximumSize  ( size ); keyAcos->setMinimumSize  ( size );
  keyAdd->setMaximumSize   ( size ); keyAdd->setMinimumSize   ( size );
  keyAns->setMaximumSize   ( size ); keyAns->setMinimumSize   ( size );
  keyAsin->setMaximumSize  ( size ); keyAsin->setMinimumSize  ( size );
  keyAtan->setMaximumSize  ( size ); keyAtan->setMinimumSize  ( size );
  keyC->setMaximumSize     ( size ); keyC->setMinimumSize     ( size );
  keyCos->setMaximumSize   ( size ); keyCos->setMinimumSize   ( size );
  keyDiv->setMaximumSize   ( size ); keyDiv->setMinimumSize   ( size );
  keyDot->setMaximumSize   ( size ); keyDot->setMinimumSize   ( size );
  keyEE->setMaximumSize    ( size ); keyEE->setMinimumSize    ( size );
  keyEq->setMaximumSize    ( size ); keyEq->setMinimumSize    ( size );
  keyExp->setMaximumSize   ( size ); keyExp->setMinimumSize   ( size );
  keyFact->setMaximumSize  ( size ); keyFact->setMinimumSize  ( size );
  keyLn->setMaximumSize    ( size ); keyLn->setMinimumSize    ( size );
  keyLPar->setMaximumSize  ( size ); keyLPar->setMinimumSize  ( size );
  keyMul->setMaximumSize   ( size ); keyMul->setMinimumSize   ( size );
  keyPerc->setMaximumSize  ( size ); keyPerc->setMinimumSize  ( size );
  keyPi->setMaximumSize    ( size ); keyPi->setMinimumSize    ( size );
  keyRaise->setMaximumSize ( size ); keyRaise->setMinimumSize ( size );
  keyRPar->setMaximumSize  ( size ); keyRPar->setMinimumSize  ( size );
  keySin->setMaximumSize   ( size ); keySin->setMinimumSize   ( size );
  keySqrt->setMaximumSize  ( size ); keySqrt->setMinimumSize  ( size );
  keySub->setMaximumSize   ( size ); keySub->setMinimumSize   ( size );
  keyTan->setMaximumSize   ( size ); keyTan->setMinimumSize   ( size );
  keyXEq->setMaximumSize   ( size ); keyXEq->setMinimumSize   ( size );
  keyX->setMaximumSize     ( size ); keyX->setMinimumSize     ( size );
}
