//    smssend
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "smssender.h"
#include <iostream>
#include <QRegExp>
#include <QMessageService>
#include <QDateTime>
#include <QUuid>
#include <time.h>

using namespace QtMobility;

SmsSender::SmsSender(int argc, char *argv[]) :
        QCoreApplication(argc,argv)
{
    m_LogFile = NULL;
    m_Save = true;
    m_El = NULL;
    m_Send = false;
    m_Flash = false;
    //For delivery report to work the program should be listening until the
    //message is delivered:
    //+CDS: 6,<the SMS ref num above>,,,<date & time>,<date & time>,0 or 48
    // last character - 0 being successful, 48 delivery failed
    //This is not possible with a CLI program (should exit as the sms is sent).
    //m_DeliveryReport = false;
    m_Validity = 167;
    m_Argc = argc - 1;
    for (int i=1; i<argc; i++)
        m_Argv.append(argv[i]);
}

SmsSender::~SmsSender()
{
    if (m_LogFile != NULL){
        m_LogFile->close();
        delete m_LogFile;
    }

    if (m_El != NULL)
        g_object_unref(m_El);
}

void SmsSender::consoleOutput(QString output, bool newline)
{
    std::cout << qPrintable(output);
    if (newline)
        std::cout << std::endl;
    else
        std::cout.flush();
}

void SmsSender::printInfo(){
    consoleOutput(QString("smssend version %1").arg(QCoreApplication::applicationVersion()));
    consoleOutput("copyright(c) 2011 by Paolo Iommarini <sakya_tg@yahoo.it>");
}

void SmsSender::printUsage()
{
    consoleOutput("Usage:");
    consoleOutput("  smssend [options] -n +39xxxxxx:+39xxxxxx -m \"Text\"");
    consoleOutput("Options:");
    consoleOutput("  -c         : compose the message with conversation");
    consoleOutput("  -C         : send the message with conversation");
    consoleOutput("");
    consoleOutput("Following options are valid only if used without -c or -C");
    consoleOutput("  -l logfile : log pnatd messages to logfile");
    consoleOutput("  -s         : don't save message in the sent folder");
    consoleOutput("  -f         : send a flash sms");
    //consoleOutput("  -r         : request delivery report");
    consoleOutput("  -v VP      : set the sms validity period (0-255)");
    consoleOutput("               Default value is 167 (1 day)");
    consoleOutput("                 0 - 143 (VP + 1) x 5 mins");
    consoleOutput("               144 - 167 12 Hours + ((VP-143) x 30 mins)");
    consoleOutput("               168 - 196 (VP-166) x 1 day");
    consoleOutput("               197 - 255 (VP-192) x 1 week");
}

int SmsSender::exec()
{
    int res = errNoError;
    m_Compose = false;

    printInfo();
    if (m_Argc == 0){
        printUsage();
        return errArguments;
    }

    args currArg = SmsSender::argNone;
    QStringList numbers;
    QString message;

    foreach (QString param, m_Argv){
        if (param.startsWith("-")){
            currArg = SmsSender::argNone;
            if (param == "-n")
                currArg = SmsSender::argNumber;
            else if(param == "-m")
                currArg = SmsSender::argText;
            else if(param == "-l")
                currArg = SmsSender::argLog;
            else if (param == "-c")
                m_Compose = true;
            else if (param == "-C"){
                m_Compose = true;
                m_Send = true;
            }else if (param == "-s")
                m_Save = false;
            else if (param == "-f")
                m_Flash = true;
            /*else if (param == "-r")
                m_DeliveryReport = true;*/
            else if (param == "-v")
                currArg = SmsSender::argValidity;
            else{
                consoleOutput(QString("ERR: unrecognized argument: %1").arg(param));
                return errArguments;
            }
        }else{
            switch (currArg){
                case SmsSender::argNumber:
                    {
                        QRegExp exp("\\+?[0-9]*");
                        foreach (QString num, param.split(":")){
                            if (!exp.exactMatch(num)){
                                consoleOutput(QString("WARNING: Invalid receiver number %1").arg(num));
                                continue;
                            }
                            if (!numbers.contains(num))
                                numbers.append(num);
                        }
                    }
                    break;
                case SmsSender::argText:
                    message = param;
                    break;
                case SmsSender::argLog:
                    {
                        m_LogFile = new QFile(param);
                        if (!m_LogFile->open(QIODevice::Truncate | QIODevice::WriteOnly)){
                            consoleOutput(QString("ERR: cannot open file: %1").arg(param));
                            return errLogfileOpenFailed;
                        }
                    }
                    break;
                case SmsSender::argValidity:
                    {
                        bool ok = false;
                        m_Validity = param.toInt(&ok);
                        if (!ok || m_Validity < 0 || m_Validity > 255){
                            consoleOutput(QString("WARNING: Invalid validity period %1").arg(m_Validity));
                            m_Validity = 167;
                        }
                    }
                    break;
                default:
                    break;
            }

            currArg = SmsSender::argNone;
        }
    }

    bool ok = true;
    if (numbers.isEmpty()){
        consoleOutput("ERR: No receiver number specified");
        ok = false;
        res = errArguments;
    }

    if (message.isEmpty()){
        consoleOutput("ERR: No message text specified");
        ok = false;
        res = errArguments;
    }

    if (ok){
        if (m_Save)
            m_El = rtcom_el_new();

        QStringList messageBlocks;
        if (m_Compose || message.length() <= 160)
            messageBlocks.append(message);
        else{
            int parts = message.length() / 160;
            if (message.length() % 160 > 0)
                parts++;
            consoleOutput("WARNING: SMS exceding 160 characters");
            consoleOutput(QString("         Splitted in %1 parts").arg(parts));
            for (int i=0; i<parts - 1; i++)
                messageBlocks.append(message.mid(160*i, 160));
        }

        if (m_Flash)
            consoleOutput("Sendind flash SMS:");
        else
            consoleOutput("Sendind SMS:");

        bool sent;
        int totalMsg = numbers.count() * messageBlocks.count();
        int i = 0;
        foreach(QString number, numbers){
            i++;
            consoleOutput(QString("[%1/%2]").arg(i).arg(totalMsg), false);
            foreach (QString msg, messageBlocks){
                if (m_Compose)
                    sent = compose(number, msg);
                else
                    sent = send(number, msg);

                if (!sent)
                    res = errSmsFailed;
            }
        }
    }

    return res;
}

bool SmsSender::compose(QString number, QString messageText)
{
    //Implementation using QtMobility (opens conversation after the message is moved to the sent folder):
    QMessage message;
    QMessageService service;

    message.setType(QMessage::Sms);
    message.setTo(QMessageAddress(QMessageAddress::Phone, number));
    message.setBody(messageText);

    if (m_Send){
        if (service.send(message)){
            consoleOutput(".");
            consoleOutput(QString("OK: sms queued for sending to %1").arg(number));
            return true;
        }
    }else{
        if (service.compose(message)){
            consoleOutput(".");
            return true;
        }
    }
    consoleOutput(QString("ERR: Failed to queue sms for sending to %1").arg(number));
    return false;
}

bool SmsSender::send(QString number, QString messageText)
{
    //Implementation using pnatd (must be run as root)
    //to run it as user:
    // chmod 4555 /usr/bin/pnatd
    bool res = true;
    Pty* pty = new Pty();
    QStringList args;
    if (!pty->run("pnatd", args)){
        consoleOutput(QString("ERR: Failed to run pnatd"));
        res = false;
    }

    if (res){
        writeCmd(pty, "AT\r", "OK");
        consoleOutput(".", false);
    }

    //AT+CPIN="0000" <-- TODO: send if the SIM is pin-protected
    if (res){
        if (!writeCmd(pty, "AT+CMGF=1\r", "OK")){
            consoleOutput(QString("\nERR: Failed to send at+cmgf command"));
            res = false;
        }else
            consoleOutput(".", false);
    }

    if (res){
        int opt = 17;
        /*if (m_DeliveryReport)
            opt += 32;*/

        int dcs = 0;
        if (m_Flash)
            dcs = 16;

        if (!writeCmd(pty, QString("AT+CSMP=%1,%2,0,%3\r").arg(opt).arg(m_Validity).arg(dcs), "OK")){
            consoleOutput(QString("\nERR: Failed to send at+csmp command"));
            res = false;
        }else
            consoleOutput(".", false);
    }

    if (res){
        if (!writeCmd(pty, QString("AT+CMGS=\"%1\"\r").arg(number).toLatin1(), ">")){
            consoleOutput(QString("\nERR: Failed to send at+cmgs command"));
            res = false;
        }else
            consoleOutput(".", false);
    }

    if (res){
        if (!writeCmd(pty, messageText, messageText)){
            consoleOutput(QString("\nERR: Failed to send text message"));
            res = false;
        }else
            consoleOutput(".", false);
    }

    if (res){
        res = writeCmd(pty, QString("%1").arg(QChar(26)).toLatin1(), "OK", -1);
        if (!res){
            consoleOutput("FAILED");
        }else{
            consoleOutput("OK");
            if (m_Save){
                if (!saveMsgToSentFolder(number, messageText))
                    consoleOutput("WARNING: Failed to save sent sms");
            }
        }
    }
    delete pty;
    return res;
}

bool SmsSender::writeCmd(Pty* p, QString cmd, QString okReply, int msectimeout)
{
    if (m_LogFile != NULL)
        m_LogFile->write(cmd.toLatin1());

    QString fullRes;
    if (!p->writeCmd(&cmd, &fullRes, msectimeout, &okReply)){
        consoleOutput(QString("ERR: failed to write to pty"));
        return false;
    }
    if (m_LogFile != NULL)
        m_LogFile->write(fullRes.toLatin1());

    if (!okReply.isEmpty()){
        QStringList lines = fullRes.split(QRegExp("\r|\r\n"), QString::SkipEmptyParts);
        QString res = lines.last().trimmed();
        bool ok = res == okReply;
        //if (!ok)
        //    consoleOutput(QString("ERR: pty reply \"%1\"").arg(fullRes));
        return ok;
    }else{
        return true;
    }
}

bool SmsSender::saveMsgToSentFolder(QString number, QString message)
{
    bool result = false;
    QString partNumber = number.right(7);

    RTComElEvent* ev = rtcom_el_event_new();

    RTCOM_EL_EVENT_SET_FIELD(ev, service, g_strdup(QString("RTCOM_EL_SERVICE_SMS").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, event_type, g_strdup(QString("RTCOM_EL_EVENTTYPE_SMS_OUTBOUND").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, start_time, time(NULL));
    RTCOM_EL_EVENT_SET_FIELD(ev, end_time, time(NULL));
    RTCOM_EL_EVENT_SET_FIELD(ev, is_read, TRUE);
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_ebook_uid, 0);
    RTCOM_EL_EVENT_SET_FIELD(ev, local_uid, g_strdup(QString("ring/tel/ring").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, local_name, g_strdup(QString("<SelfHandle>").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_uid, number.toLatin1().data());
    RTCOM_EL_EVENT_SET_FIELD(ev, remote_name, g_strdup(QString("").toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, group_uid, g_strdup(partNumber.toLatin1().data()));
    RTCOM_EL_EVENT_SET_FIELD(ev, free_text, g_strdup(message.toUtf8().data()));

    int id = rtcom_el_add_event(m_El, ev, NULL);
    if (id >= 0){
        QUuid uid = QUuid::createUuid();
        rtcom_el_add_header(m_El, id, QString("message-token").toLatin1().data(), uid.toString().toLatin1().data(), NULL);
        result = true;
    }

    rtcom_el_event_free(ev);

    return result;
}
