/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtCore/QString>
#include <QtTest/QtTest>

#include "coordinates/geocoordinate.h"
#include "map/mapcommon.h"

#include "coordinates/scenecoordinate.h"

const double X = 12.345678;
const double Y = -89.765432;

const double ORIGIN = 1000;
const double DELTA = 500;

class TestSceneCoordinate : public QObject
{
    Q_OBJECT

private Q_SLOTS:
    void azimuthTo();
    void azimuthTo_data();
    void constructors();
    void conversion();
    void conversion_data();
    void isNull();
    void operators();
    void settersAndGetters();
};

// for formatting the output of double valuest into the test log
namespace QTest {
    template<>
    char *toString(const double &number)
    {
        QByteArray ba;
        ba += QByteArray::number(number, 'f', 9);
        return qstrdup(ba.data());
    }
}

void TestSceneCoordinate::azimuthTo()
{
    QFETCH(SceneCoordinate, to);
    QFETCH(qreal, expectedAzimuth);

    SceneCoordinate from(ORIGIN, ORIGIN);

    QCOMPARE(from.azimuthTo(to), expectedAzimuth);
}

void TestSceneCoordinate::azimuthTo_data()
{
    QTest::addColumn<SceneCoordinate>("to");
    QTest::addColumn<qreal>("expectedAzimuth");

    QTest::newRow("N") <<  SceneCoordinate(ORIGIN, ORIGIN - DELTA) << 0.0;
    QTest::newRow("NE") <<  SceneCoordinate(ORIGIN + DELTA, ORIGIN - DELTA) << 45.0;
    QTest::newRow("E") <<  SceneCoordinate(ORIGIN + DELTA, ORIGIN) << 90.0;
    QTest::newRow("SE") <<  SceneCoordinate(ORIGIN + DELTA, ORIGIN + DELTA) << 135.0;
    QTest::newRow("S") <<  SceneCoordinate(ORIGIN, ORIGIN + DELTA) << 180.0;
    QTest::newRow("SW") <<  SceneCoordinate(ORIGIN - DELTA, ORIGIN + DELTA) << 225.0;
    QTest::newRow("W") <<  SceneCoordinate(ORIGIN - DELTA, ORIGIN) << 270.0;
    QTest::newRow("NW") <<  SceneCoordinate(ORIGIN - DELTA, ORIGIN - DELTA) << 315.0;
}

void TestSceneCoordinate::constructors()
{
    SceneCoordinate coordinate;
    QVERIFY(coordinate.isNull());

    SceneCoordinate coordinate2(X, Y);
    QCOMPARE(coordinate2.x(), X);
    QCOMPARE(coordinate2.y(), Y);

    // NOTE: constructor with conversion from GeoCoordinate is tested in conversion() test slot
}

void TestSceneCoordinate::conversion()
{
    QFETCH(GeoCoordinate, geoCoordinate);
    QFETCH(SceneCoordinate, result);

    SceneCoordinate sceneCoordinate(geoCoordinate);

    // Comparison is done using only the integer parts because data type of the scene coordinate
    // values is double so the result is not exact pixel value but is one containing a fractional
    // part. Also the rounding errors below one pixel are insignificant.

    QCOMPARE(int(sceneCoordinate.x()), int(result.x()));
    QCOMPARE(int(sceneCoordinate.y()), int(result.y()));
}

void TestSceneCoordinate::conversion_data()
{
    QTest::addColumn<GeoCoordinate>("geoCoordinate");
    QTest::addColumn<SceneCoordinate>("result");

    QTest::newRow("top left pixel") << GeoCoordinate(OSM_MAX_LATITUDE, MIN_LONGITUDE)
                                    << SceneCoordinate(0, 0);

    const double ONE_SCENE_PIXEL_WIDTH_IN_DEGREES = 0.00000536441802978516;
    const double LAST_SCENE_HORIZONTAL_PIXEL_LONGITUDE = MAX_LONGITUDE
                                                         - ONE_SCENE_PIXEL_WIDTH_IN_DEGREES;
    QTest::newRow("bottom right pixel")
            << GeoCoordinate(-OSM_MAX_LATITUDE, LAST_SCENE_HORIZONTAL_PIXEL_LONGITUDE)
            << SceneCoordinate(OSM_MAP_MAX_PIXEL_X, OSM_MAP_MAX_PIXEL_Y);

    QTest::newRow("southeast corner with 180 degrees longitude")
            << GeoCoordinate(-OSM_MAX_LATITUDE, MAX_LONGITUDE)
            << SceneCoordinate(OSM_MAP_MIN_PIXEL_X, OSM_MAP_MAX_PIXEL_Y);

    QTest::newRow("southeast corner just little over west edge of the map")
            << GeoCoordinate(-OSM_MAX_LATITUDE, MIN_LONGITUDE - ONE_SCENE_PIXEL_WIDTH_IN_DEGREES)
            << SceneCoordinate(OSM_MAP_MAX_PIXEL_X, OSM_MAP_MAX_PIXEL_Y);
}

void TestSceneCoordinate::isNull()
{
    SceneCoordinate coordinate;
    QVERIFY(coordinate.isNull());
    coordinate.setX(1);
    QVERIFY(!coordinate.isNull());

    SceneCoordinate coordinate2;
    QVERIFY(coordinate2.isNull());
    coordinate2.setY(1);
    QVERIFY(!coordinate.isNull());

    SceneCoordinate coordinate3;
    QVERIFY(coordinate3.isNull());
    coordinate3.setX(1);
    coordinate3.setY(1);
    QVERIFY(!coordinate.isNull());
}

void TestSceneCoordinate::operators()
{
    // operator*=
    SceneCoordinate coordinate(100, 30);
    coordinate *= 3;
    QCOMPARE(coordinate.x(), 300.0);
    QCOMPARE(coordinate.y(), 90.0);

    // operator+=
    SceneCoordinate coordinate2(110, 40);
    coordinate2 += SceneCoordinate(40, 75);
    QCOMPARE(coordinate2.x(), 150.0);
    QCOMPARE(coordinate2.y(), 115.0);

    // operator-=
    SceneCoordinate coordinate3(120, 50);
    coordinate3 -= SceneCoordinate(45, 80);
    QCOMPARE(coordinate3.x(), 75.0);
    QCOMPARE(coordinate3.y(), -30.0);

    // operator+
    SceneCoordinate coordinate4(130, 60);
    SceneCoordinate coordinate5(25, 10);
    SceneCoordinate result = coordinate4 + coordinate5;
    QCOMPARE(result.x(), 155.0);
    QCOMPARE(result.y(), 70.0);
    QVERIFY(&result != &coordinate4);
    QVERIFY(&result != &coordinate5);

    // operator-
    SceneCoordinate coordinate6(140, 70);
    SceneCoordinate coordinate7(15, 5);
    SceneCoordinate result2 = coordinate6 - coordinate7;
    QCOMPARE(result2.x(), 125.0);
    QCOMPARE(result2.y(), 65.0);
    QVERIFY(&result2 != &coordinate6);
    QVERIFY(&result2 != &coordinate7);

    // operator*
    SceneCoordinate coordinate8(10, 20);
    SceneCoordinate result3 = 4 * coordinate8;
    QCOMPARE(result3.x(), 40.0);
    QCOMPARE(result3.y(), 80.0);
    QVERIFY(&result3 != &coordinate8);
}

void TestSceneCoordinate::settersAndGetters()
{
    SceneCoordinate coordinate;
    QCOMPARE(coordinate.x(), (double)0);
    QCOMPARE(coordinate.y(), (double)0);

    coordinate.setX(X);
    coordinate.setY(Y);

    QCOMPARE(coordinate.x(), X);
    QCOMPARE(coordinate.y(), Y);
}

QTEST_APPLESS_MAIN(TestSceneCoordinate);

#include "testscenecoordinate.moc"
