/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef LOCATIONSEARCHPANEL_H
#define LOCATIONSEARCHPANEL_H

#include <QtGui>

#include "panelbase.h"

class ExtendedListItemDelegate;
class GeoCoordinate;
class ImageButton;
class Location;
class LocationListView;
class SearchHistoryListView;

/**
 * @brief Location search panel
 *
 * @author Jussi Laitinen - jussi.laitinen (at) ixonos.com
 * @author Sami Rämö - sami.ramo (at) ixonos.com
 */
class LocationSearchPanel : public PanelBase
{
    Q_OBJECT

public:
    /**
     * @brief Default constructor
     *
     * @param parent
     */
    LocationSearchPanel(QWidget *parent = 0);

    /**
     * @brief Destructor
     *
     * Writes search history to settings.
     */
    ~LocationSearchPanel();

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
protected:
    /**
    * @brief Re-implemented from QWidget::hideEvent()
    *
    * Calls clearListsSelections()
    *
    * @param event
    */
    void hideEvent(QHideEvent *event);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
private:
    /**
    * @brief Reads search history from settings.
    */
    void readSettings();

    /**
    * @brief Set text for header
    *
    * @param count Search result count
    */
    void setHeaderText(int count);

private slots:
    /**
    * @brief Prepends search history list view with search.
    *
    * If search history limit is reached, oldest search is removed from list view.
    *
    * @param searchString search string to add
    * @param dateTime date and time of search (if empty, current date and time is added)
    */
    void prependSearchHistory(QString searchString, QDateTime dateTime = QDateTime());

    /**
    * @brief Clears lists' selections.
    *
    * Does call setRouteButtonDisabled().
    */
    void clearListsSelections();

    /**
     * @brief show / hide empty panel label
     *
     * @param show true if empty panel should be shown
     */
    void showEmptyPanel(bool show);

    /**
    * @brief Shows location list view.
    *
    * @param locationItemsCount location items count
    */
    void showLocationListView(int locationItemsCount);

    /**
    * @brief Shows search history list view.
    */
    void showSearchHistoryListView();

    /**
    * @brief Populates location list view.
    *
    * @param locations list of Location objects
    */
    void populateLocationListView(const QList<Location> &locations);

    /**
    * @brief Routes to selected location.
    *
    * Emits routeToLocation if location is selected from list.
    */
    void routeToSelectedLocation();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signal for location item clicked.
    *
    * @param swBound south-west bound GeoCoordinate
    * @param neBound north-east bound GeoCoordinate
    */
    void locationItemClicked(const GeoCoordinate &swBound, const GeoCoordinate &neBound);

    /**
    * @brief Signal for requesting searching location.
    */
    void requestSearchLocation();

    /**
    * @brief Signal for routing to location.
    *
    * @param coordinates location's geo coordinates
    */
    void routeToLocation(const GeoCoordinate &coordinates);

    /**
    * @brief Signal for route waypoint item clicked.
    *
    * @param coordinate waypoint item's coordinate
    */
    void routeWaypointItemClicked(const GeoCoordinate &coordinate);

    /**
    * @brief Signal is emitted when search history item is clicked.
    *
    * @param searchString search string used
    */
    void searchHistoryItemClicked(const QString &searchString);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    QLabel *m_noSearchLabel;                        ///< Text label for empty panel
    QLabel *m_resultsLabel;                         ///< Location list label

    ImageButton *m_clearLocationListButton;         ///< Clear location list button
    ImageButton *m_routeButton;                     ///< Route to location button
    SearchHistoryListView *m_searchHistoryListView; ///< Search history list view
    LocationListView *m_locationListView;           ///< Search results list view
};

#endif // LOCATIONSEARCHPANEL_H
