/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QLayout>
#include <QPainter>
#include <QPropertyAnimation>
#include <QStateMachine>

#include "panelcommon.h"

#include "listitemcontextbuttonbar.h"

ListItemContextButtonBar::ListItemContextButtonBar(QWidget *parent) :
    QWidget(parent),
    m_waitForOpen(false),
    m_contextButtons(0),
    m_newContextButtons(0),
    m_state(StateHidden)
{
    qDebug() << __PRETTY_FUNCTION__;

    // --- BAR LOOK ---
    const int BAR_HEIGHT = 79;
    setFixedHeight(BAR_HEIGHT);

    const int BUTTON_WIDTH = 74;
    const int BUTTON_COUNT = 4;
    const int HORIZONTAL_MARGIN = 10;
    setFixedWidth(BUTTON_COUNT * BUTTON_WIDTH + 2 * HORIZONTAL_MARGIN);

    m_backgroundLeft = new QPixmap(":/res/images/list_item_context_button_bar_left.png");
    m_backgroundMiddle = new QPixmap(":/res/images/list_item_context_button_bar_tile.png");
    m_backgroundRight = new QPixmap(":/res/images/list_item_context_button_bar_right.png");

    // --- LAYOUT ---
    m_barLayout = new QHBoxLayout();
    setLayout(m_barLayout);
    m_barLayout->setContentsMargins(0, 0, 0, 0);

    // --- SET POSITION ---
    // center this widget horizontally to middle of the panel contents area and set outside of
    // the view
    const int FROM_PANEL_CONTENTS_LEFT = PANEL_WIDTH / 2 - width() / 2;
    move(PANEL_TAB_BAR_WIDTH + PANEL_BAR_WIDTH + FROM_PANEL_CONTENTS_LEFT, -size().height());

    // --- ANIMATION ---
    const int ANIMATION_DURATION_MS = 150;

    m_animation = new QPropertyAnimation(this, "pos", this);
    m_animation->setDuration(ANIMATION_DURATION_MS);

    connect(m_animation, SIGNAL(finished()),
            this, SLOT(onAnimationFinished()));

    // target values for animations
    m_animation->setStartValue(pos());
    const int Y = 0;
    m_animation->setEndValue(QPoint(pos().x(), Y));
}

ListItemContextButtonBar::~ListItemContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_backgroundLeft)
        delete m_backgroundLeft;

    if (m_backgroundMiddle)
        delete m_backgroundMiddle;

    if (m_backgroundRight)
        delete m_backgroundRight;
}

void ListItemContextButtonBar::changeButtons()
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_ASSERT(m_animation->state() == QAbstractAnimation::Stopped);

    if (!isVisible())
        show();

    // Remove old buttons (if any)
    if (m_contextButtons) {
        m_barLayout->removeWidget(m_contextButtons);
        m_contextButtons->setParent(0);
    }

    m_contextButtons = m_newContextButtons;
    m_newContextButtons = 0;

    m_barLayout->addWidget(m_contextButtons, 0, Qt::AlignHCenter);
}

void ListItemContextButtonBar::hideContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_state = StateClosing;
    m_animation->setDirection(QAbstractAnimation::Backward);
    m_animation->start();
}

void ListItemContextButtonBar::onAnimationFinished()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_animation->direction() == QAbstractAnimation::Backward) {
        m_state = StateHidden;
        if (m_newContextButtons) {
            changeButtons();
            if (m_waitForOpen) {
                m_waitForOpen = false;
                showContextButtonBar();
            }
        }
    } else {
        m_state = StateVisible;
    }
}

void ListItemContextButtonBar::onListItemSelectionChanged(bool itemIsSelected)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (itemIsSelected) {
        if (m_newContextButtons)
            m_waitForOpen = true;
        else if (m_state != StateVisible)
            showContextButtonBar();
    } else {
        if (m_newContextButtons)
            m_waitForOpen = false;
        else if (m_state != StateHidden)
            hideContextButtonBar();
    }
}

void ListItemContextButtonBar::paintEvent(QPaintEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(event);

    QPainter painter(this);

    const int TOP = 0;
    const int LEFT = 0;
    painter.drawPixmap(TOP, LEFT, *m_backgroundLeft);
    painter.drawTiledPixmap(m_backgroundLeft->width(), TOP,
                            width() - m_backgroundLeft->width() - m_backgroundRight->width(),
                            height(),
                            *m_backgroundMiddle);
    painter.drawPixmap(width() - m_backgroundRight->width(), TOP, *m_backgroundRight);
}

void ListItemContextButtonBar::setContextButtons(QWidget *contextButtons)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_newContextButtons = contextButtons;
    m_waitForOpen = false;

    if (m_state != StateHidden)
        hideContextButtonBar();
    else
        changeButtons();
}

void ListItemContextButtonBar::showContextButtonBar()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_contextButtons->layout()->count() > 0) {
        m_state = StateOpening;
        m_animation->setDirection(QAbstractAnimation::Forward);
        m_animation->start();
    }
}
