/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com
       Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>
#include <QDesktopServices>
#include <QNetworkAccessManager>
#include <QNetworkDiskCache>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QPixmap>
#include <QUrl>

#include "error.h"
#include "mapcommon.h"
#include "network/networkaccessmanager.h"

#include "mapfetcher.h"

const int MAX_PARALLEL_DOWNLOADS = 2; ///< Max simultaneous parallel downloads
const int NOT_FOUND = -1; ///< Return value if matching request is not found from the list

MapFetcher::MapFetcher(NetworkAccessManager *manager, QObject *parent)
    : QObject(parent)
    , m_pendingRequestsSize(0)
    , m_fetchMapImagesTimerRunning(false)
    , m_manager(manager)
{
    qDebug() << __PRETTY_FUNCTION__;

    QNetworkDiskCache *diskCache = new QNetworkDiskCache(this);
    diskCache->setCacheDirectory(QDesktopServices::storageLocation(
            QDesktopServices::CacheLocation));
    m_manager->setCache(diskCache);

    connect(m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(
            downloadFinished(QNetworkReply*)));
}

QUrl MapFetcher::buildURL(int zoomLevel, QPoint tileNumbers)
{
    qDebug() << __PRETTY_FUNCTION__;

    /**
    * @brief Map server string for building actual URL
    *
    * %1 zoom level
    * %2 x index
    * %3 y index
    *
    * NOTE: If the URL is changed, then the parseURL method must be changed to match
    *       the new URL structure
    *
    * @var MAP_SERVER_URL
    */
    const QString MAP_SERVER_URL = QString("http://tile.openstreetmap.org/mapnik/%1/%2/%3.png");

    return QString(MAP_SERVER_URL)
            .arg(zoomLevel).arg(tileNumbers.x()).arg(tileNumbers.y());
}

void MapFetcher::checkNextRequestFromCache()
{
    qDebug() << __PRETTY_FUNCTION__;

    int i = newestRequestIndex(false);

    if (i != NOT_FOUND) {
        QUrl url = m_pendingRequests[i].url;
        if (!url.isEmpty() && url.isValid()) {
            if (loadImageFromCache(url)) {
                // was found, remove from the list
                m_pendingRequests.removeAt(i);
            }
            else {
                // didn't found from cache so mark cache checked and leave to queue
                m_pendingRequests[i].cacheChecked = true;

                if (m_currentDownloads.size() < MAX_PARALLEL_DOWNLOADS)
                    startNextDownload();
            }
        }
    }

    // schedule checking of the next request if the list is not empty
    if (newestRequestIndex(false) != NOT_FOUND)
        QTimer::singleShot(0, this, SLOT(checkNextRequestFromCache()));
    else
        m_fetchMapImagesTimerRunning = false;
}

void MapFetcher::downloadFinished(QNetworkReply *reply)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_currentDownloads.contains(reply)) {

        if (reply->error() == QNetworkReply::NoError) {
            QImage image;
            QUrl url = reply->url();

            if (!image.load(reply, 0))
                image = QImage();

            int zoomLevel;
            int x;
            int y;
            parseURL(url, &zoomLevel, &x, &y);

            emit mapImageReceived(zoomLevel, x, y, QPixmap::fromImage(image));
        }
        else {
            emit error(ErrorContext::SITUARE, SituareError::MAP_IMAGE_DOWNLOAD_FAILED);
        }

        m_currentDownloads.removeAll(reply);
        reply->deleteLater();
        startNextDownload();
    }
}

void MapFetcher::enqueueFetchMapImage(int zoomLevel, int x, int y)
{
    qDebug() << __PRETTY_FUNCTION__;

    QUrl url = buildURL(zoomLevel, QPoint(x, y));

    // ignore request if it is already downloading
    foreach (QNetworkReply *reply, m_currentDownloads) {
        if (reply->url() == url)
            return;
    }

    // check if new request is already in the list and move it to the begin of the list...
    bool found = false;
    for (int i = 0; i < m_pendingRequests.size(); i++) {
        if (m_pendingRequests[i].url == url) {
            m_pendingRequests.move(i, 0);
            found = true;
            break;
        }
    }
    // ...or add new request to the begining of the list
    if (!found) {
        MapTileRequest request(url);
        m_pendingRequests.prepend(request);
    }

    limitPendingRequestsListSize();

    if (!m_fetchMapImagesTimerRunning) {
        m_fetchMapImagesTimerRunning = true;
        QTimer::singleShot(0, this, SLOT(checkNextRequestFromCache()));
    }
}

void MapFetcher::limitPendingRequestsListSize()
{
    qDebug() << __PRETTY_FUNCTION__;

    while (m_pendingRequests.size() > m_pendingRequestsSize) {
        m_pendingRequests.removeLast();
    }
}

bool MapFetcher::loadImageFromCache(const QUrl &url)
{
    qDebug() << __PRETTY_FUNCTION__;

    bool imageFound = false;

    QAbstractNetworkCache *cache = m_manager->cache();

    if (cache) {

        int zoomLevel;
        int x;
        int y;
        parseURL(url, &zoomLevel, &x, &y);
        int originalZoomLevel = zoomLevel;

        // try to fetch requested and upper level images until found or MAX_UPPER_LEVELS
        // limit is reached
        const int MAX_UPPER_LEVELS = 4;
        do {
            QIODevice *cacheImage = cache->data(buildURL(zoomLevel, QPoint(x, y)));
            if (cacheImage) {
                QPixmap pixmap;
                if (pixmap.loadFromData(cacheImage->readAll())) {
                    imageFound = true;
                    emit mapImageReceived(zoomLevel, x, y, pixmap);
                }

                delete cacheImage;
            }
        } while (!imageFound
                 && ((originalZoomLevel - zoomLevel) < MAX_UPPER_LEVELS)
                 && translateIndexesToUpperLevel(zoomLevel, x, y));

        // check expiration if image was found from requested level
        if (imageFound && (originalZoomLevel == zoomLevel)) {
            // check if image is expired
            QNetworkCacheMetaData metaData = cache->metaData(url);
            if ((metaData.expirationDate().isValid()) && (url.isValid())) {

                if (metaData.expirationDate() < QDateTime::currentDateTime()) {
                    cache->remove(url);
                    return false;
                }
            }
        }

        // if image was found, but from upper level, return false
        if (imageFound && (originalZoomLevel != zoomLevel))
            return false;
    }

    return imageFound;
}

bool MapFetcher::translateIndexesToUpperLevel(int &zoomLevel, int &x, int &y)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (zoomLevel > OSM_MIN_ZOOM_LEVEL) {
        zoomLevel--;
        x /= 2;
        y /= 2;

        return true;
    }

    return false;
}

int MapFetcher::newestRequestIndex(bool cacheChecked)
{
    qDebug() << __PRETTY_FUNCTION__;

    for (int i = 0; i < m_pendingRequests.size(); i++) {
        if (m_pendingRequests[i].cacheChecked == cacheChecked) {
            return i;
        }
    }

    return NOT_FOUND;
}

void MapFetcher::parseURL(const QUrl &url, int *zoom, int *x, int *y)
{
    qDebug() << __PRETTY_FUNCTION__;

    QString path = url.path();
    QStringList pathParts = path.split("/", QString::SkipEmptyParts);

    int size = pathParts.size();

    // Example URL: "http://tile.openstreetmap.org/mapnik/14/9354/4263.png"
    const int MIN_PATH_SPLITTED_PARTS = 4;
    const int ZOOM_INDEX = size - 3;
    const int X_INDEX = size - 2;
    const int Y_INDEX = size - 1;
    const int FILE_EXTENSION_LENGTH = 4;

    if (size >= MIN_PATH_SPLITTED_PARTS) {
        *zoom = (pathParts.at(ZOOM_INDEX)).toInt();
        *x = (pathParts.at(X_INDEX)).toInt();
        QString yString = pathParts.at(Y_INDEX);
        yString.chop(FILE_EXTENSION_LENGTH);
        *y = yString.toInt();
    }
}

void MapFetcher::setDownloadQueueSize(int size)
{
    qDebug() << __PRETTY_FUNCTION__ << "size:" << size;

    m_pendingRequestsSize = size;
    limitPendingRequestsListSize();
}

void MapFetcher::startNextDownload()
{
    qDebug() << __PRETTY_FUNCTION__;

    int i = newestRequestIndex(true);

    if (i != NOT_FOUND) {
        QUrl url = m_pendingRequests.takeAt(i).url;

        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", "Situare");
        QNetworkReply *reply = m_manager->get(request);

        m_currentDownloads.append(reply);
    }
}
