/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef GPSLOCATIONITEM_H
#define GPSLOCATIONITEM_H

#include <QGraphicsItem>

/**
  * @brief Class for indicating current position and accuracy accuired from GPS on the map.
  *
  * Actual position spot image is a child item of this class and this class handles drawing the
  * accuracy ring. GPS fix accuracy is indicated by the radius of the circle around the position
  * item.
  */
class GPSLocationItem : public QGraphicsItem
{
public:
    /**
      * @brief Constructor
      */
    GPSLocationItem();

/*******************************************************************************
 * BASE CLASS INHERITED AND REIMPLEMENTED MEMBER FUNCTIONS
 ******************************************************************************/
public:
    /**
    * @brief Implements the bounding rect
    *
    * @returns Bounding rect of the item
    */
    QRectF boundingRect() const;

    /**
    * @brief Implements the painter
    */
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
      * @brief Enable / disable item
      *
      * Actual enabling is done when next position update is received, so the old value before
      * hiding the item is not used.
      *
      * @param enable True if item should be enabled, otherwise false
      */
    void setEnabled(bool enable);

    /**
      * @brief Update position item
      *
      * @param scenePosition Scene position
      * @param accuracy Accuracy of the GPS fix
      * @param sceneResolution Scene horizontal resolution at given coordinate latitude level
      */
    void updateItem(QPointF scenePosition, qreal accuracy, qreal sceneResolution);

private:
    /**
    * @brief Set new radius for the accuracy ring
    *
    * Calls setBoundingRect() if the radius changes
    *
    * @param accuracy Accuracy (in meters)
    * @param sceneResolution Scene resolution (meters/pixel)
    */
    void setAccuracyRingRadius(qreal accuracy, qreal sceneResolution);

    /**
    * @brief Sets new bounding rect for this item based on the accuracy ring radius
    *
    * @param radius Radius of the accuracy ring
    */
    void setBoundingRect(qreal radius);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_showOnNextUpdate;    ///< should the item be shown when next update arrives

    qreal m_radius;             ///< accuracy ring radius in scene pixels

    QGraphicsPixmapItem *m_pixmapItem;  ///< red led item

    QRectF m_boundingRect;              ///< item's bounding rect
};

#endif // GPSLOCATIONITEM_H
