/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#include <QDebug>

#include "coordinates/scenecoordinate.h"
#include "friendgroupitem.h"
#include "friendlocationitem.h"
#include "mapcommon.h"
#include "mapscene.h"
#include "user/user.h"

#include "frienditemshandler.h"

FriendItemsHandler::FriendItemsHandler(MapScene *mapScene, QObject *parent)
    : QObject(parent),
      m_mapScene(mapScene),
      m_zoomLevel(0)
{
    qDebug() << __PRETTY_FUNCTION__;
}

void FriendItemsHandler::addFriendItem(User *friendData)
{
    qDebug() << __PRETTY_FUNCTION__;

    FriendLocationItem *item = new FriendLocationItem(friendData->userId());

    item->setProfileImage(friendData->profileImage(), friendData->profileImageUrl());
    item->setPos(SceneCoordinate(friendData->coordinates()).toPointF());
    m_friendItems.append(item);
    m_mapScene->addItem(item);

    connect(item, SIGNAL(locationItemClicked(QList<QString>)),
            this, SIGNAL(locationItemClicked(QList<QString>)));
}

void FriendItemsHandler::checkAllFriendsForCollidingFriends()
{
    qDebug() << __PRETTY_FUNCTION__;

    QLinkedList<FriendLocationItem *>::iterator iter = m_friendItems.begin();
    while (iter != m_friendItems.end()) {
        // check only friends which are not part of group already
        if (!(*iter)->isPartOfGroup()) {
            checkFriendForCollidingFriends(*iter);
        }
        iter++;
    }
}

void FriendItemsHandler::checkFriendForCollidingFriends(FriendLocationItem *item)
{
    // checkGroupsForCollisions() is used for checking if groups collide with another
    // groups or friend items, so this method doesn't have to check against groups

    qDebug() << __PRETTY_FUNCTION__;

    FriendGroupItem *group = 0;

    // loop through all friend items
    QLinkedList<FriendLocationItem *>::iterator iter = m_friendItems.begin();
    while (iter != m_friendItems.end()) {
        // but don't check myself and friends which are already part of a group
        if (item != *iter && !(*iter)->isPartOfGroup()) {
            if (collides(item, *iter)) {
                if (!group) {
                    group = new FriendGroupItem(item);
                    m_mapScene->addItem(group);
                    m_friendGroupItems.append(group);

                    connect(group, SIGNAL(locationItemClicked(QList<QString>)),
                            this, SIGNAL(locationItemClicked(QList<QString>)));
                }
                group->joinFriend(*iter);
            }
        }
        iter++;
    }
}

bool FriendItemsHandler::collides(BaseLocationItem *item1, BaseLocationItem *item2)
{
    QRect item1Rect = item1->sceneTransformedBoundingRect(m_zoomLevel);
    QRect item2Rect = item2->sceneTransformedBoundingRect(m_zoomLevel);

    if (item1Rect.intersects(item2Rect))
        return true;

    if (item1Rect.left() < (OSM_MAP_MIN_PIXEL_X + item1Rect.width() / 2)) {
        QRect translated = item1Rect.translated(OSM_MAP_PIXELS_X, 0);
        if (translated.intersects(item2Rect))
            return true;
    } else if (item1Rect.right() > (OSM_MAP_MAX_PIXEL_X  - item1Rect.width() / 2)) {
        QRect translated = item1Rect.translated(-OSM_MAP_PIXELS_X, 0);
        if (translated.intersects(item2Rect))
            return true;
    }

    return false;
}

void FriendItemsHandler::deleteFriendItem(FriendLocationItem *item)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapScene->removeItem(item);
    delete item;
}

void FriendItemsHandler::destructGroups()
{
    foreach (FriendGroupItem *group, m_friendGroupItems)
        group->dropFriends();

    qDeleteAll(m_friendGroupItems);
    m_friendGroupItems.clear();
}

void FriendItemsHandler::friendImageReady(User *user)
{
   qDebug() << __PRETTY_FUNCTION__;

   foreach (FriendLocationItem *friendItem, m_friendItems) {
       if (user->userId() == friendItem->userId()) {
           friendItem->setProfileImage(user->profileImage(), user->profileImageUrl());
           break;
       }
   }
}

void FriendItemsHandler::friendListUpdated(QList<User *> &friendsList)
{
    qDebug() << __PRETTY_FUNCTION__;

    destructGroups();

    // loop through friend items and find matching friend data. If matching data
    // is not found then remove item
    QLinkedList<FriendLocationItem *>::iterator iter = m_friendItems.begin();
    while (iter != m_friendItems.end()) {
        bool found = false;
        foreach (User * friendData, friendsList) {
            if (friendData->userId() == (*iter)->userId()) {
                found = true;
                break;
            }
        }
        if (!found) {
            // data for friend item was not found so item must be deleted
            deleteFriendItem(*iter);
            iter = m_friendItems.erase(iter);
        }
        else {
            iter++;
        }
    }

    // loop through new friend data, find matching friend items and update them, or add new items
    // if old items are not found
    foreach (User * friendData, friendsList) {
        bool found = false;
        foreach (FriendLocationItem *friendItem, m_friendItems) {
            if (friendData->userId() == friendItem->userId()) {
                // friend item was found so update the data
                updateFriendItem(friendItem, friendData);
                found = true;
                break;
            }
        }
        if (!found) {
            // friend item was not found so new item must be added
            addFriendItem(friendData);
        }
    }

    checkAllFriendsForCollidingFriends();
}

void FriendItemsHandler::refactorFriendItems(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_zoomLevel = zoomLevel;

    destructGroups();
    checkAllFriendsForCollidingFriends();
}

void FriendItemsHandler::updateFriendItem(FriendLocationItem *friendItem, User *friendData)
{
    qDebug() << __PRETTY_FUNCTION__;

    // update position
    friendItem->setPos(SceneCoordinate(friendData->coordinates()).toPointF());

    // update image
    if (friendItem->profileImageUrl() != friendData->profileImageUrl())
        friendItem->setProfileImage(friendData->profileImage(), friendData->profileImageUrl());
}
