/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QFile>
#include <QDir>
#include <QApplication>
#include <QDebug>
#include <QTimer>

#include "error.h"
#include "gpscommon.h"
#include "gpsposition.h"
#include "gpspositionprivate.h"

GPSPositionPrivate::GPSPositionPrivate(QObject *parent)
    : QObject(parent),
      m_gpsSource(0),
      m_initialized(false),
      m_running(false),
      m_updateInterval(DEFAULT_UPDATE_INTERVAL)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_parent = static_cast<GPSPosition*>(parent);
}

void GPSPositionPrivate::setPowerSave(bool enabled)
{
    qDebug() << __PRETTY_FUNCTION__;
}

bool GPSPositionPrivate::isInitialized() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_initialized;
}

void GPSPositionPrivate::setMode(GPSPosition::Mode mode, const QString &filePath)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_gpsSource) {
        disconnect(m_gpsSource, 0, 0, 0);
        delete m_gpsSource;
    }

    if (mode == GPSPosition::Default) {
        m_gpsSource = QGeoPositionInfoSource::createDefaultSource(this);

        if (!m_gpsSource) {
            m_initialized = false;
            emit m_parent->error(ErrorContext::SITUARE, SituareError::GPS_INITIALIZATION_FAILED);
            return;
        }
    }
    else if (mode == GPSPosition::Simulation) {
        QNmeaPositionInfoSource *nmeaSource = new QNmeaPositionInfoSource(
                QNmeaPositionInfoSource::SimulationMode, this);
        QFile *logFile = new QFile(filePath, this);

        nmeaSource->setDevice(logFile);
        m_gpsSource = nmeaSource;
    }

    if (m_gpsSource) {
        m_initialized = true;
        connect(m_gpsSource, SIGNAL(positionUpdated(const QGeoPositionInfo &)),
                this, SLOT(positionUpdated(const QGeoPositionInfo &)));
        connect(m_gpsSource, SIGNAL(updateTimeout()), this, SLOT(updateTimeout()));

        m_gpsSource->setUpdateInterval(m_updateInterval);
    }
}

void GPSPositionPrivate::start()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_initialized && !isRunning()) {
        m_gpsSource->startUpdates();
        m_running = true;
    }
}

void GPSPositionPrivate::stop()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_initialized && isRunning()) {
        m_gpsSource->stopUpdates();
        m_running = false;
    }
}

bool GPSPositionPrivate::isRunning() const
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_running;
}

GeoCoordinate GPSPositionPrivate::lastPosition() const
{
    QGeoCoordinate coordinate = m_gpsSource->lastKnownPosition().coordinate();
    return GeoCoordinate(coordinate.latitude(), coordinate.longitude());
}

void GPSPositionPrivate::requestLastPosition()
{
    qDebug() << __PRETTY_FUNCTION__;

    QGeoCoordinate coordinate = m_gpsSource->lastKnownPosition().coordinate();

    if (coordinate.isValid()) {
        emit m_parent->position(GeoCoordinate(coordinate.latitude(), coordinate.longitude()),
                                accuracy(m_gpsSource->lastKnownPosition()));
    }
}

void GPSPositionPrivate::requestUpdate()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!isRunning())
        start();

    m_gpsSource->requestUpdate();
}

void GPSPositionPrivate::positionUpdated(const QGeoPositionInfo &positionInfo)
{
    qDebug() << __PRETTY_FUNCTION__ << positionInfo;

    const QGeoCoordinate &coordinate = positionInfo.coordinate();

    if (coordinate().isValid()) {
        emit m_parent->position(GeoCoordinate(coordinate.latitude(), coordinate.longitude()),
                                accuracy(positionInfo));
    }
}

void GPSPositionPrivate::updateTimeout()
{
    qDebug() << __PRETTY_FUNCTION__;

    emit m_parent->timeout();
}

void GPSPositionPrivate::setUpdateInterval(int interval)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (m_updateInterval != interval) {
        m_updateInterval = interval;
        m_gpsSource->setUpdateInterval(m_updateInterval);
    }
}

qreal GPSPositionPrivate::accuracy(const QGeoPositionInfo &positionInfo)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (positionInfo.hasAttribute(QGeoPositionInfo::HorizontalAccuracy))
        return positionInfo.attribute(QGeoPositionInfo::HorizontalAccuracy);
    else
        return GPS_ACCURACY_UNDEFINED;
}
