/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/


#ifndef SCENECOORDINATE_H
#define SCENECOORDINATE_H

#include <QDebug>
#include <QMetaType>

class GeoCoordinate;

/**
* @brief Scene coordinate
*
* @author Sami Rämö - sami.ramo@ixonos.com
*/
class SceneCoordinate
{
public:
    /**
    * @brief Constructs a null coordinate
    */
    SceneCoordinate();

    /**
    * @brief Constructs a coordinate with given x and y
    *
    * @param x X value
    * @param y Y value
    */
    SceneCoordinate(double x, double y);

    /**
    * @brief Constructs a coordinate with values converted from the given GeoCoordinate
    *
    * Uses convertFrom() method.
    *
    * @param coordinate Geological coordinate
    */
    SceneCoordinate(const GeoCoordinate &coordinate);

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
    * @brief Returns the azimuth from this coordinate to other coordinate
    *
    * Zero is pointing to north. Returned value is from 0 to 360.
    *
    * @param to Target coordinate
    * @returns Azimuth in degrees
    */
    qreal azimuthTo(const SceneCoordinate &to) const;

    /**
    * @brief Check if coordinate is (0.0, 0.0)
    *
    * @returns True if both X and Y are 0.0, otherwise false
    */
    bool isNull() const;

    /**
    * @brief Sets the latitude
    *
    * @param x X value
    */
    void setX(double x);

    /**
    * @brief Sets the longitude
    *
    * @param y Y value
    */
    void setY(double y);

    /**
    * @brief Convert to QPointF
    *
    * @returns a QPointF object
    */
    QPointF toPointF() const;

    /**
    * @brief Returns the x value
    *
    * @returns x
    */
    double x() const;

    /**
    * @brief Returns the y value
    *
    * @returns y
    */
    double y() const;

private:
    /**
     * @brief Convert values from GeoCoordinate
     *
     * Does run normalize() for the x value after the conversion to make sure that the result
     * is inside the allowed map pixel values.
     *
     * In horizontal direction:
     *    -180º equals scene pixel 0 (first scene pixel)
     *    +180º equals -180º
     *
     *    scene has 2^18 * 256 - 1 = 67108864 pixels per side
     *    one pixel width is 360º / 67108864 = 0.00000536441802978516º
     *    so the last scene pixel is 180º - 0.00000536441802978516º = 179.99999463558197021484º
     *
     * @param coordinate Geological coordinate
     */
    void convertFrom(const GeoCoordinate &coordinate);

    /**
     * @brief Translate integer part of the given value between min and max
     *
     * If given value is not inside the given range (min <= value <= max), then the allowed range
     * is adder or subtracted until the value does fit in the range. Only integer part is compared.
     *
     * @param value Value to be normalized
     * @param min Minimum allowed value
     * @param max Maximum allowed value
     */
    void normalize(double &value, int min, int max);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    double m_x;         ///< X value
    double m_y;         ///< Y value

/*******************************************************************************
 * OPERATORS
 ******************************************************************************/
public:
    /**
    * @brief Operator for creating QVariant
    */
    operator QVariant() const;

    /**
    * @brief Multiplies this coordinate's values by the given factor, and returns a reference
    *        to this coordinate.
    */
    SceneCoordinate & operator*=(double factor);

    /**
    * @brief Adds the given coordinate to this coordinate and returns a reference to this
             coordinate
    */
    SceneCoordinate & operator+=(const SceneCoordinate &coordinate);

    /**
    * @brief Subtracts the given coordinate from this coordinate and returns a reference to this
             coordinate.
    */
    SceneCoordinate & operator-=(const SceneCoordinate &coordinate);

    /**
    * @brief Returns a SceneCoordinate object that is the sum of the coordinates.
    */
    const SceneCoordinate operator+(const SceneCoordinate &other) const;

    /**
    * @brief Returns a SceneCoordinate object that is formed by subtracting the coordinates.
    */
    const SceneCoordinate operator-(const SceneCoordinate &other) const;
};

/**
* @brief Operator for writing the coordinate to QDebug stream.
*/
QDebug operator<<(QDebug dbg, const SceneCoordinate &coordinate);

/**
* @brief Returns a SceneCoordinate object that is multiplied by the given factor.
*/
const SceneCoordinate operator*(double factor, const SceneCoordinate &coordinate);

Q_DECLARE_METATYPE(SceneCoordinate)

#endif // SCENECOORDINATE_H
