/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QtTest/QtTest>

#include "map/mapcommon.h"
#include "map/maptile.h"

const QPointF UNDEFINED_POSITION(UNDEFINED, UNDEFINED);

class TestMapTile : public QObject
{
    Q_OBJECT
private slots:
    void zoomLevel();
    void tileNumber();
    void position_data();
    void position();
    void positionUnsetValues();
    void sceneLevel_data();
    void sceneLevel();
};

/**
* @brief Test setting zoomlevel
*/
void TestMapTile::zoomLevel()
{
    MapTile mapTile;
    mapTile.setZoomLevel(13, 14);
    QVERIFY(mapTile.zoomLevel() == 13);
}

/**
* @brief Test setting tile numbers
*/
void TestMapTile::tileNumber()
{
    MapTile mapTile;
    mapTile.setTileNumber(QPoint(24, 17));
    QVERIFY(mapTile.tileNumber() == QPoint(24, 17));
}

/**
* @brief Data for position test
*/
void TestMapTile::position_data()
{
    QTest::addColumn<int>("zoomLevel");
    QTest::addColumn<QPoint>("tileNumber");
    QTest::addColumn<QPointF>("result");

    QTest::newRow("allowed values") << 16 << QPoint(24, 17) << QPointF(24576, 17408); //ok
    QTest::newRow("x position negative") << 16 << QPoint(-1, 0) << QPointF(-1024, 0); //ok
    QTest::newRow("y position negative") << 16 << QPoint(0, -1) << UNDEFINED_POSITION; //fail

    QTest::newRow("min zoom") << 0 << QPoint(0, 0) << QPointF(0, 0); //ok
    QTest::newRow("min zoom - 1") << -1 << QPoint(0, 0) << UNDEFINED_POSITION; //fail
    QTest::newRow("min zoom & x out of world (east)") << 2 << QPoint(4, 0) << QPointF(4*256*(1<<16), 0); //ok
    QTest::newRow("min zoom & y out of range (upper limit)") << 0 << QPoint(0, 1) << UNDEFINED_POSITION; //fail
    QTest::newRow("min zoom, x & y out of range (upper limit)") << 0 << QPoint(1, 1) << UNDEFINED_POSITION; //fail

    QTest::newRow("max zoom") << 18 << QPoint(2, 3) << QPointF(512, 768); //ok
    QTest::newRow("max zoom + 1") << 19 << QPoint(2, 3) << UNDEFINED_POSITION; //fail
    QTest::newRow("max zoom & x out of world (east)") << 18 << QPoint(262144, 0) << QPointF(256*(1<<18), 0); //ok
    QTest::newRow("max zoom & y out of range (upper limit)") << 18 << QPoint(0, 262144) << UNDEFINED_POSITION; //fail
    QTest::newRow("max zoom, x & y out of range(upper limit) ") << 18 << QPoint(262144, 262144) << UNDEFINED_POSITION; //fail
}

/**
* @brief Test position correctness
*/
void TestMapTile::position()
{
    QFETCH(int, zoomLevel);
    QFETCH(QPoint, tileNumber);
    QFETCH(QPointF, result);

    MapTile mapTile;
    mapTile.setZoomLevel(zoomLevel, 14);
    mapTile.setTileNumber(tileNumber);
    QCOMPARE(mapTile.pos(), result);
}

/**
* @brief Test position correctness when some values are not set
*/
void TestMapTile::positionUnsetValues()
{
    MapTile mapTile;

    // zoom level and tile numbers unset
    QCOMPARE(mapTile.pos(), UNDEFINED_POSITION);

    // only tile numbers set
    mapTile.setTileNumber(QPoint(24, 17));
    QCOMPARE(mapTile.pos(), UNDEFINED_POSITION);

    // both set
    mapTile.setZoomLevel(16, 14);
    QCOMPARE(mapTile.pos(), QPointF(24576, 17408));

    // only zoom level set
    MapTile anotherMapTile;
    anotherMapTile.setZoomLevel(14, 14);
    QCOMPARE(anotherMapTile.pos(), UNDEFINED_POSITION);
}

/**
  * @brief Data for test of setting zValues
  */
void TestMapTile::sceneLevel_data()
{
    QTest::addColumn<int>("tileZoomLevel");
    QTest::addColumn<int>("viewZoomLevel");
    QTest::addColumn<qreal>("result");

    QTest::newRow("tile zoom 15, view zoom 17") << 15 << 17 << 15.0 + 19;
    QTest::newRow("tile zoom 15, view zoom 16") << 15 << 16 << 15.0 + 19;
    QTest::newRow("tile zoom 15, view zoom 15") << 15 << 15 << 15.0 + 19;
    QTest::newRow("tile zoom 15, view zoom 14") << 15 << 14 << 13.5 + 19;
    QTest::newRow("tile zoom 15, view zoom 13") << 15 << 13 << 11.5 + 19;
    QTest::newRow("tile zoom 15, view zoom 12") << 15 << 12 << 9.5 + 19;
    QTest::newRow("tile zoom 18, view zoom 1") << 18 << 1 << -15.5 + 19;
    QTest::newRow("tile zoom 18, view zoom 0") << 18 << 0 << -17.5 + 19;
}

/**
  * @brief Test setting zValues (drawing order)
  */
void TestMapTile::sceneLevel()
{
    QFETCH(int, tileZoomLevel);
    QFETCH(int, viewZoomLevel);
    QFETCH(qreal, result);

    MapTile tile;
    tile.setZoomLevel(tileZoomLevel, 14);
    tile.setSceneLevel(viewZoomLevel);
    QCOMPARE(tile.zValue(), result);
}

QTEST_MAIN(TestMapTile)
#include "testmaptile.moc"
