/*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QGraphicsScene>
#include <QtTest/QtTest>

#include "common.h"
#include "map/mapcommon.h"
#include "map/mapengine.h"

const int DEFAULT_TEST_ZOOMLEVEL = 5;
const QPoint DEFAULT_TEST_SCENECOORDINATE = QPoint(12345,54321);
const QPoint NEW_TEST_SCENECOORDINATE = QPoint(50000,40000);

class TestMapEngine: public QObject
{
    Q_OBJECT

private slots:
    void convertTileNumberToSceneCoordinate();
    void convertLatLonToSceneCoordinate_data();
    void convertLatLonToSceneCoordinate();
    void setLocationNewTilesCount();
    void setLocationRemovedTilesCount();
    void zoomOutRemovedTilesCount();
    void zoomInRemovedTilesCount();
    void usingLastLocation();

signals:
    void setCenterPosition(QPoint); // a way to call a private slot
};

/**
* @brief Test converting tile numbers to scene coordinates
*
* Different zoom levels are also tested
*/
void TestMapEngine::convertTileNumberToSceneCoordinate()
{
    QCOMPARE(MapEngine::convertTileNumberToSceneCoordinate(18, QPoint(0,0)), QPoint(0,0));
    QCOMPARE(MapEngine::convertTileNumberToSceneCoordinate(18, QPoint(1,2)), QPoint(256,512));
    QCOMPARE(MapEngine::convertTileNumberToSceneCoordinate(16, QPoint(3,4)), QPoint(3072,4096));
}

/**
  * @brief Test data for converting latitude and longitude coordinates to scene coordinates
  */
void TestMapEngine::convertLatLonToSceneCoordinate_data()
{
    QTest::addColumn<QPointF>("coordinate");
    QTest::addColumn<QPoint>("result");

    QTest::newRow("top left") << QPointF(MIN_LONGITUDE, MAX_LATITUDE) << QPoint(0, 0);

    int x = (1 << MAX_MAP_ZOOM_LEVEL) * TILE_SIZE_X;
    int y = (1 << MAX_MAP_ZOOM_LEVEL) * TILE_SIZE_Y;
    QTest::newRow("bottom right") << QPointF(MAX_LONGITUDE, MIN_LATITUDE) << QPoint(x, y);
}

/**
* @brief Test converting real world cordinates to scene coordinates
* @todo Implement
*/
void TestMapEngine::convertLatLonToSceneCoordinate()
{
    QFETCH(QPointF, coordinate);
    QFETCH(QPoint, result);

    QCOMPARE(MapEngine::convertLatLonToSceneCoordinate(coordinate), result);
}

void TestMapEngine::setLocationNewTilesCount()
{
    MapEngine engine;
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(QPoint)), &engine, SLOT(setCenterPosition(QPoint)));

    QSignalSpy fetchImageSpy(&engine, SIGNAL(fetchImage(int,int,int)));
    QTest::qWait(1000);
    fetchImageSpy.clear();

    // first test, scene is empty so all tiles should be downloaded
    emit setCenterPosition(QPoint(1220*16, 1220*16));
    QTest::qWait(1000);
    QCOMPARE(fetchImageSpy.count(), 5*4);
    fetchImageSpy.clear();

    // move one tile to east and south, only most right one column and most bottom one row tiles
    // should be downloaded
    emit setCenterPosition(QPoint((1220+TILE_SIZE_X)*16, (1220+TILE_SIZE_Y)*16));
    QTest::qWait(1000);
    QCOMPARE(fetchImageSpy.count(), 4 + 4);
    fetchImageSpy.clear();
}

void TestMapEngine::setLocationRemovedTilesCount()
{
    MapEngine engine;
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(QPoint)), &engine, SLOT(setCenterPosition(QPoint)));

    const int maxItemsCount = 40;

    emit setCenterPosition(QPoint(1220*16, 1220*16));
    QTest::qWait(1000);
    emit setCenterPosition(QPoint(2220*16, 2220*16));
    QTest::qWait(1000);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(QPoint(520*16, 2220*16));
    QTest::qWait(1000);
    emit setCenterPosition(QPoint(2220*16, 520*16));
    QTest::qWait(1000);

    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::zoomInRemovedTilesCount()
{
    MapEngine engine;
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(QPoint)), &engine, SLOT(setCenterPosition(QPoint)));

    const int maxItemsCount = 40;

    emit setCenterPosition(QPoint(1220*16, 1220*16));
    QTest::qWait(1000);
    QTest::qWait(1000);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(QPoint(520*16, 2220*16));
    QTest::qWait(1000);
    emit setCenterPosition(QPoint(2220*16, 520*16));
    QTest::qWait(1000);

    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::zoomOutRemovedTilesCount()
{
    MapEngine engine;
    engine.viewResized(QSize(800, 480));
    connect(this, SIGNAL(setCenterPosition(QPoint)), &engine, SLOT(setCenterPosition(QPoint)));

    const int maxItemsCount = 40;

    emit setCenterPosition(QPoint(1220*16, 1220.23*16));
    QTest::qWait(1000);
    emit setCenterPosition(QPoint(2220*16, 2220.23*16));
    QTest::qWait(1000);
    QVERIFY(engine.scene()->items().count() <= maxItemsCount);

    emit setCenterPosition(QPoint(520*16, 2220*16));
    QTest::qWait(1000);
    emit setCenterPosition(QPoint(2220*16, 520*16));
    QTest::qWait(1000);

    QVERIFY(engine.scene()->items().count() <= maxItemsCount);
}

void TestMapEngine::usingLastLocation()
{
    // Create mapengine and start monitoring zoomLevelChanged-signal
    MapEngine *mapengine = new MapEngine;
    QSignalSpy mapEngineSpy(mapengine, SIGNAL(zoomLevelChanged(int)));
    connect(this, SIGNAL(setCenterPosition(QPoint)), mapengine, SLOT(setCenterPosition(QPoint)));
    QVERIFY (mapEngineSpy.isValid());
    QCOMPARE (mapEngineSpy.count(), 0);

    // Write new zoomlevel and location to settings
    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    settings.setValue(MAP_LAST_ZOOMLEVEL, DEFAULT_TEST_ZOOMLEVEL);
    settings.setValue(MAP_LAST_POSITION,
                      mapengine->convertSceneCoordinateToLatLon(DEFAULT_TEST_ZOOMLEVEL,
                                                                DEFAULT_TEST_SCENECOORDINATE));

    // Call mapengines init() and verify that signal will be send
    mapengine->init();
    QCOMPARE (mapEngineSpy.count(), 1);

    // Remove zoomlevel and location from settings, call init() again and check that
    // signals is send again
    settings.remove(MAP_LAST_ZOOMLEVEL);
    settings.remove(MAP_LAST_POSITION);
    mapengine->init();
    QCOMPARE(mapEngineSpy.count(), 2);

    // Check parameters of sended signals
    QList<QVariant> parameters = mapEngineSpy.takeFirst();
    QVERIFY(parameters.at(0).toInt() == DEFAULT_TEST_ZOOMLEVEL);

    parameters = mapEngineSpy.takeFirst();
    QVERIFY(parameters.at(0).toInt() == DEFAULT_START_ZOOM_LEVEL);

    // Call set location with know parameter to get location changed
    // Store new location and zoomlevel to settings
    emit setCenterPosition(NEW_TEST_SCENECOORDINATE);
    delete mapengine;

    // Read settings and verify that zoomlevel is correct
    MapEngine testengine;// = new mapEngine;
    QPointF LatLonLocation =
            settings.value(MAP_LAST_POSITION, ERROR_VALUE_NOT_FOUND_ON_SETTINGS).toPointF();
    QCOMPARE(LatLonLocation, testengine.convertSceneCoordinateToLatLon(DEFAULT_TEST_ZOOMLEVEL,
                                                                      NEW_TEST_SCENECOORDINATE));
}

QTEST_MAIN(TestMapEngine)
#include "testmapengine.moc"
