/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jukka Saastamoinen - jukka.saastamoinen@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Katri Kaikkonen - katri.kaikkonen@ixonos.com
       Henri Lampela - henri.lampela@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include "common.h"
#include "imagebutton.h"
#include "../user/user.h"
#include "userinfo.h"

const int BACKGROUND_WIDTH = 240;           ///< Width for item
const int BACKGROUND_TOP_HEIGHT = 16;       ///< Height for item top
const int BACKGROUND_BOTTOM_HEIGHT = 15;    ///< Height for item bottom
const int ICON_HEIGHT = 24;                 ///< Icon height
const int ICON_WIDTH = 24;                  ///< Icon width
const int MARGIN = 5;                       ///< Icon margin
const int LINE_LENGTH = 17;                 ///< Line length
const int MOUSE_PRESS_AREA_WIDTH = 20;      ///< Area width for item height toggling
const int MOUSE_PRESS_AREA_HEIGHT = 20;     ///< Area height for item height toggling

/**
* @var LABEL_MAX_WIDTH
*
* @brief All label's maximum width
*/
const int LABEL_MAX_WIDTH = BACKGROUND_WIDTH - 3 * MARGIN - ICON_WIDTH + 130;

UserInfo::UserInfo(QWidget *parent)
    : QWidget(parent),
      m_expanded(false),
      m_messageUpdateVerified(false),
      m_updateLocation(0)
{
    qDebug() << __PRETTY_FUNCTION__;

    QVBoxLayout *verticalLayout = new QVBoxLayout(this);
    verticalLayout->setContentsMargins(MARGIN * 2, 0, MARGIN * 2, MARGIN * 2);
    verticalLayout->setSpacing(0);
    setLayout(verticalLayout);

    QFormLayout *infoLayout = new QFormLayout();
    infoLayout->setMargin(0);
    infoLayout->setSpacing(0);

    QHBoxLayout *buttonLayout = new QHBoxLayout();
    buttonLayout->setMargin(0);
    buttonLayout->setSpacing(0);

    QLabel *envelopeLabel = new QLabel();
    envelopeLabel->setPixmap(QPixmap(":/res/images/envelope.png"));
    envelopeLabel->setContentsMargins(0, 0, MARGIN, 0);
    envelopeLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *compassLabel = new QLabel();
    compassLabel->setPixmap(QPixmap(":/res/images/compass.png"));
    compassLabel->setContentsMargins(0, 0, MARGIN, 0);
    compassLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);
    QLabel *clockLabel = new QLabel();
    clockLabel->setPixmap(QPixmap(":/res/images/clock.png"));
    clockLabel->setContentsMargins(0, 0, MARGIN, 0);
    clockLabel->setFixedSize(ICON_WIDTH + MARGIN, ICON_HEIGHT);

    m_findButton = new ImageButton();

    m_nameLabel = new QLabel();
    m_nameLabel->setWordWrap(true);

    m_statusTextLabel = new QLabel();
    m_statusTextLabel->setWordWrap(true);
    m_locationLabel = new QLabel();
    m_locationLabel->setWordWrap(true);
    m_updatedLabel = new QLabel();
    m_updatedLabel->setWordWrap(true);

    ImageButton *updateFriendsButton = new ImageButton(this, ":/res/images/refresh.png",
                                                             ":/res/images/refresh_s.png");
    ImageButton *updateStatusMessageButton = new ImageButton(this, ":/res/images/send_position.png",
                                                                   ":/res/images/send_position_s.png");

    buttonLayout->addWidget(updateFriendsButton);
    buttonLayout->addWidget(updateStatusMessageButton);

    infoLayout->addRow(envelopeLabel, m_statusTextLabel);
    infoLayout->addRow(compassLabel, m_locationLabel);
    infoLayout->addRow(clockLabel, m_updatedLabel);

    verticalLayout->addWidget(m_findButton, 0, Qt::AlignHCenter);
    verticalLayout->addWidget(m_nameLabel, 0, Qt::AlignHCenter);
    verticalLayout->addLayout(infoLayout);
    verticalLayout->addLayout(buttonLayout);

    connect(updateStatusMessageButton,SIGNAL(clicked()),
            this,SLOT(messageUpdate()));

    connect(updateFriendsButton,SIGNAL(clicked()),
            this, SIGNAL(refreshUserData()));

    connect(m_findButton, SIGNAL(clicked()),
            this, SLOT(findButtonClicked()));

    setFixedWidth(BACKGROUND_WIDTH);

    this->setFont(NOKIA_FONT_SMALL);
    m_nameLabel->setFont(NOKIA_FONT_NORMAL);
    QPalette itemPalette = palette();
    itemPalette.setColor(QPalette::Foreground, COLOR_GRAY);
    setPalette(itemPalette);
    QPalette namePalette = m_nameLabel->palette();
    namePalette.setColor(QPalette::Foreground, Qt::white);
    m_nameLabel->setPalette(namePalette);

    m_backgroundTopImage.load(":/res/images/user_info_item_top.png");
    m_backgroundMiddleImage.load(":/res/images/user_info_item_middle.png");
    m_backgroundBottomImage.load(":/res/images/user_info_item_bottom.png");

    restoreUnsendMessage();
}

UserInfo::~UserInfo()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);

    if (!m_backupMessage.isEmpty()) {        
        settings.setValue(USER_UNSEND_MESSAGE, m_backupMessage.toAscii());
        settings.setValue(USER_UNSEND_MESSAGE_PUBLISH, m_backupFacebookPublishPolicity);
    } else {
        settings.remove(USER_UNSEND_MESSAGE);
        settings.remove(USER_UNSEND_MESSAGE_PUBLISH);
    }
}

void UserInfo::setAddress(const QString &address)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_locationLabel->setText(address);
}

void UserInfo::setCoordinates(const QPointF &coordinates)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_coordinates = coordinates;
}

void UserInfo::setMessageText(const QString &text)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_messageText = text;
    m_expandedMessageText.clear();
    QString temp = "";
    for(int i=0;i < text.length();i++) {
        if(fontMetrics().width(temp.append(text.at(i))) > 170) {
            temp.append("\n");
            if(temp.startsWith(QString(" ")))
                temp.remove(0, 1);

            m_expandedMessageText.append(temp);
            temp.clear();
        }
    }
    m_expandedMessageText.append(temp);
    setText(false);
}

void UserInfo::setProfileImage(const QPixmap &image)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_findButton->setButtonIcon(image);
}

void UserInfo::setTime(const QString &time)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_updatedLabel->setText(time);
}

void UserInfo::setUserName(const QString &name)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_userName = name;
    setText(false);
}

void UserInfo::setText(bool expanded)
{
    qDebug() << __PRETTY_FUNCTION__;

    if (expanded) {
        m_statusTextLabel->setText(m_expandedMessageText);
    }
    else {
        m_nameLabel->setText(shortenText(m_nameLabel, m_userName, LABEL_MAX_WIDTH));
        m_statusTextLabel->setText(shortenText(m_statusTextLabel, m_messageText,
                                               LABEL_MAX_WIDTH));
    }
}

void UserInfo::backupUpdateLocationDialogData(const QString &status, bool publish)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_backupMessage = status;
    m_backupFacebookPublishPolicity = publish;

    m_messageUpdateVerified = false;
}

void UserInfo::clearUpdateLocationDialogData()
{
    qDebug() << __PRETTY_FUNCTION__;

    m_backupMessage.clear();
    m_backupFacebookPublishPolicity = false;
}

void UserInfo::findButtonClicked()
{
    qDebug() << __PRETTY_FUNCTION__;

    emit findUser(m_coordinates);
}

void UserInfo::mouseReleaseEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << event->pos();

    if ((abs(m_mousePosition.y() - event->pos().y()) <= MOUSE_PRESS_AREA_WIDTH) &&
        (abs(m_mousePosition.x() - event->pos().x()) <= MOUSE_PRESS_AREA_HEIGHT)) {
        if (m_expanded) {
            setText(false);
            m_expanded = false;
        }
        else {
            setText(true);
            m_expanded = true;
        }
    }
}

void UserInfo::mousePressEvent(QMouseEvent *event)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << event->pos();

    m_mousePosition = event->pos();
}

void UserInfo::messageUpdate()
{
    qDebug() << __PRETTY_FUNCTION__;

    delete m_updateLocation;
    m_updateLocation = new UpdateLocationDialog(m_backupMessage, m_backupFacebookPublishPolicity,
                                                this);

    connect(this, SIGNAL(reverseGeoReady(QString)),
            m_updateLocation, SLOT(setAddress(QString)));

    connect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
            this, SIGNAL(statusUpdate(QString,bool)));

    connect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
            this, SLOT(backupUpdateLocationDialogData(QString,bool)));

    connect(m_updateLocation, SIGNAL(finished(int)),
            this, SLOT(updateLocationDialogFinished(int)));

    m_updateLocation->show();

    emit requestReverseGeo();
}

void UserInfo::paintEvent(QPaintEvent *aPaintEvent)
{
    qDebug() << __PRETTY_FUNCTION__ << " " << aPaintEvent->rect();

    QPainter painter(this);

    QRect topRect = QRect(0, MARGIN, BACKGROUND_WIDTH, BACKGROUND_TOP_HEIGHT);
    QRect middleRect = QRect(0, topRect.bottom(), BACKGROUND_WIDTH,
                             height() - BACKGROUND_TOP_HEIGHT - BACKGROUND_BOTTOM_HEIGHT);
    QRect bottomRect = QRect(topRect.left(), middleRect.bottom(), BACKGROUND_WIDTH,
                             BACKGROUND_BOTTOM_HEIGHT);

    painter.drawPixmap(topRect, m_backgroundTopImage);
    painter.drawPixmap(middleRect, m_backgroundMiddleImage);
    painter.drawPixmap(bottomRect, m_backgroundBottomImage);
}

void UserInfo::restoreUnsendMessage()
{
    qDebug() << __PRETTY_FUNCTION__;

    QSettings settings(DIRECTORY_NAME, FILE_NAME);
    m_backupMessage = settings.value(USER_UNSEND_MESSAGE, EMPTY).toString();
    m_backupFacebookPublishPolicity =
            settings.value(USER_UNSEND_MESSAGE_PUBLISH, false).toBool();
}

QString UserInfo::shortenText(const QLabel *label, const QString &text, int textMaxWidth)
{
    qDebug() << __PRETTY_FUNCTION__;

    QFontMetrics labelMetrics = label->fontMetrics();
    QString textParam = text;
    int index = textParam.indexOf('\n');
    int textWidth = fontMetrics().width(textParam);

        if (index > 0) {
            textParam.truncate(index);
            textParam.append("...");
        }
            if (textWidth > 150)
                textParam.insert(LINE_LENGTH, QString("\n"));

   return labelMetrics.elidedText(textParam, Qt::ElideRight, textMaxWidth);
}

void UserInfo::verifyMessageUpdateFailure(const int error)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(error);

    if (!m_messageUpdateVerified) {

        if (m_messageText != m_backupMessage && !m_backupMessage.isEmpty())
            emit notificateUpdateFailing(tr("Location update failed, please try again"), false);
    }

    m_messageUpdateVerified = true;
}

void UserInfo::updateLocationDialogFinished(int reason)
{
    qDebug() << __PRETTY_FUNCTION__;

    Q_UNUSED(reason);

    if (m_updateLocation) {
        disconnect(this, SIGNAL(reverseGeoReady(QString)),
                m_updateLocation, SLOT(setAddress(QString)));

        disconnect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
                this, SIGNAL(statusUpdate(QString,bool)));

        disconnect(m_updateLocation, SIGNAL(statusUpdate(QString,bool)),
                this, SLOT(backupUpdateLocationDialogData(QString,bool)));

        disconnect(m_updateLocation, SIGNAL(finished(int)),
                this, SLOT(updateLocationDialogFinished(int)));

        m_updateLocation->deleteLater();
        m_updateLocation = 0;
    }

}
