/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Jussi Laitinen - jussi.laitinen@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef NETWORKACCESSMANAGER_H
#define NETWORKACCESSMANAGER_H

#include <QList>
#include <QHash>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QAbstractNetworkCache>

class NetworkHandler;
class QNetworkAccessManager;

/**
* @brief NetworkAccessManager class.
*
* This class handles network requests and receives network replies.
* NetworkAccessManager queues requests when disconnected from network
* and makes requests when connected to network.
*/
class NetworkAccessManager : public QObject
{
    Q_OBJECT

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
    * @brief Returns instance of NetworkAccessManager.
    *
    * Creates instance if not created.
    */
    static NetworkAccessManager *instance();

    /**
    * @brief Returns connection state.
    *
    * @return true if connected, false otherwise
    */
    bool isConnected();

    /**
    * @brief Makes request and return reply.
    *
    * @param request QNetworkRequest
    * @return QNetworkReply
    */
    QNetworkReply *get(const QNetworkRequest &request, bool onlineRequestsOnly = false);

    /**
    * @brief Sets cache.
    *
    * @param cache QAbstractNetworkCache instance
    */
    void setCache(QAbstractNetworkCache *cache);

    /**
    * @brief Returns cache.
    *
    * @return QAbstractNetworkCache
    */
    QAbstractNetworkCache *cache() const;

protected:
    /**
    * @brief Constructor.
    *
    * Instance of this class can only be created by using instance method.
    */
    NetworkAccessManager();

private slots:
    /**
    * @brief Slot for network connected state.
    */
    void connected();

    /**
    * @brief Slot for network disconnected state.
    */
    void disconnected();

    /**
    * @brief Slot for finished download.
    *
    * @param reply reply from network
    */
    void downloadFinished(QNetworkReply *reply);

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * Signal for finished download.
    *
    * @param reply reply from network
    */
    void finished(QNetworkReply *reply);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_connected;                                       ///< Connection flag
    static NetworkAccessManager *m_instance;                ///< Instance of NetworkAccessManager
    NetworkHandler *m_networkHandler;                       ///< Instance of NetworkHandler
    QNetworkAccessManager *m_networkAccessManagerPrivate;   ///< Instance of QNetworkAccessManager
    QList<QNetworkRequest> m_requestQueue;                  ///< Queue for requests
    QHash<QString, QNetworkReply*> m_offlineReplyQueue;     ///< Queue for offline replies
    QHash<QString, QNetworkReply*> m_temporaryReplyQueue;   ///< Queue for temporary replies
};

#endif // NETWORKACCESSMANAGER_H
