/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#include <QDebug>

#include "mapcommon.h"
#include "mapengine.h"
#include "maptile.h"

#include "mapscene.h"

MapScene::MapScene(QObject *parent)
    : QGraphicsScene(parent)
    , m_isRemoveStackedTilesRunning(false)
    , m_tilesSceneRect(QRect(0, 0, 0, 0))
{
    qDebug() << __PRETTY_FUNCTION__;

    setBackgroundBrush(Qt::lightGray);
    setSceneRect(QRect(QPoint(MAP_SCENE_MIN_PIXEL_X, MAP_MIN_PIXEL_Y),
                       QPoint(MAP_SCENE_MAX_PIXEL_X, MAP_MAX_PIXEL_Y)));
}

void MapScene::addTile(int tileZoomLevel, QPoint tileNumber, const QPixmap &image, int viewZoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    // tile might already be in the scene if expired tile was returned from the cache to be
    // temporarily shown while downloading the fresh one.
    QString hashKey = MapEngine::tilePath(tileZoomLevel, tileNumber.x(), tileNumber.y());
    MapTile *oldTile = tileInScene(hashKey);
    if (oldTile)
        removeTile(oldTile);

    MapTile *tile = new MapTile();
    tile->setZoomLevel(tileZoomLevel, viewZoomLevel);
    tile->setTileNumber(tileNumber);
    tile->setPixmap(image);

    m_mapTilesInScene.insert(hashKey, tile);
    addItem(tile);

    qDebug() << __PRETTY_FUNCTION__ << "tiles count:" << m_mapTilesInScene.count();

    enqueueRemoveStackedTiles(tile);
    removeOtherLevelTiles();
}

void MapScene::enqueueRemoveStackedTiles(MapTile *newTile)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_removeStackedTilesList << newTile;
    if (!m_isRemoveStackedTilesRunning) {
        m_isRemoveStackedTilesRunning = true;
        QTimer::singleShot(0, this, SLOT(runNextStackedTilesRemoval()));
    }
}

void MapScene::moveIntersectingItemsHorizontally(QRect from, int distance)
{
    qDebug() << __PRETTY_FUNCTION__;

    QList<QGraphicsItem *> spanItems = items(from, Qt::IntersectsItemBoundingRect);
    foreach (QGraphicsItem *item, spanItems) {
        if (!dynamic_cast<MapTile *>(item))
            item->moveBy(distance, 0);
    }
}

MapTile* MapScene::tileInScene(QString hashKey)
{
    qDebug() << __PRETTY_FUNCTION__;

    return m_mapTilesInScene.value(hashKey, 0);
}

void MapScene::runNextStackedTilesRemoval()
{
    qDebug() << __PRETTY_FUNCTION__;

    if (!m_removeStackedTilesList.isEmpty()) {
        MapTile *tile = m_removeStackedTilesList.takeFirst();
        removeStackedTiles(tile);
    }

    // schedule removal of the next tile if the list is not empty
    if (!m_removeStackedTilesList.isEmpty())
        QTimer::singleShot(0, this, SLOT(runNextStackedTilesRemoval()));
    else
        m_isRemoveStackedTilesRunning = false;
}

void MapScene::removeOtherLevelTiles()
{
    qDebug() << __PRETTY_FUNCTION__;

    for (int x = m_viewTilesGrid.left(); x <= m_viewTilesGrid.right(); x++) {
        for (int y = m_viewTilesGrid.top(); y <= m_viewTilesGrid.bottom(); y++) {
            if (!m_mapTilesInScene.contains(MapEngine::tilePath(m_zoomLevel, x, y)))
                return;
        }
    }

    foreach(MapTile *tile, m_mapTilesInScene) {
        if (tile->zoomLevel() != m_zoomLevel) {
            removeTile(tile);
            qDebug() << __PRETTY_FUNCTION__ << "removed other level tile";
        }
    }
}

void MapScene::removeOutOfViewTiles(QRect tilesGrid, int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    QList<QGraphicsItem *> viewItems = items(m_tilesSceneRect, Qt::IntersectsItemBoundingRect);
    QList<QGraphicsItem *> allItems = items();

    //Remove tiles which are in view from allTiles
    foreach (QGraphicsItem *item, viewItems)
        allItems.removeOne(item);

    // note: add 1 so odd values are rounded up
    int tilesGridWidthHalf = (tilesGrid.width() + 1) / 2;

    // if view is near east limit of the map, then there is duplicate tiles also on the opposite
    // side of the world which are removed from allTiles
    if (tilesGrid.right() > (MapEngine::tileMaxIndex(zoomLevel) - tilesGridWidthHalf + GRID_PADDING)) {
        QRect oppositeRect = m_tilesSceneRect;
        oppositeRect.translate(-MAP_PIXELS_X, 0);
        QList<QGraphicsItem *> oppositeItems = items(oppositeRect, Qt::IntersectsItemBoundingRect);
        foreach (QGraphicsItem *item, oppositeItems)
            allItems.removeOne(item);
    }

    // if view is near west limit of the map, then there is duplicate tiles also on the opposite
    // side of the world which are removed from allTiles
    if (tilesGrid.left() < (tilesGridWidthHalf - GRID_PADDING)) {
        QRect oppositeRect = m_tilesSceneRect;
        oppositeRect.translate(MAP_PIXELS_X, 0);
        QList<QGraphicsItem *> oppositeItems = items(oppositeRect, Qt::IntersectsItemBoundingRect);
        foreach (QGraphicsItem *item, oppositeItems)
            allItems.removeOne(item);
    }

    //Remove tiles out of view
    foreach (QGraphicsItem *item, allItems) {
        MapTile *tile = dynamic_cast<MapTile *>(item);
        if (tile)
            removeTile(tile);
    }
}

void MapScene::removeStackedTiles(MapTile *newTile)
{
    qDebug() << __PRETTY_FUNCTION__;

    QRectF newTileSceneRect = newTile->sceneBoundingRect();

    //Loop all items under new tile
    QList<QGraphicsItem *> collidingItems = newTile->collidingItems(Qt::IntersectsItemBoundingRect);
    foreach (QGraphicsItem *collidingItem, collidingItems) {
        MapTile *collidingTile = dynamic_cast<MapTile *>(collidingItem);
        if (collidingTile) {
            if (newTile->zValue() > collidingTile->zValue()) {
                // remove tile if it is fully obscured by new tile
                QRectF collidingTileSceneRect = collidingTile->sceneBoundingRect();
                if (newTileSceneRect.contains(collidingTileSceneRect))
                    removeTile(collidingTile);
            }
        }
    }
}

void MapScene::removeTile(MapTile *tile)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_mapTilesInScene.remove(MapEngine::tilePath(tile->zoomLevel(),
                                                 tile->tileNumber().x(),
                                                 tile->tileNumber().y()));
    removeItem(tile);
    m_removeStackedTilesList.removeAll(tile);
    delete tile;

    qDebug() << __PRETTY_FUNCTION__ << "tiles count:" << m_mapTilesInScene.count();
}

void MapScene::setSceneVerticalOverlap(int viewHeight, int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__;

    int overlap = viewHeight / 2 * (1 << (MAX_MAP_ZOOM_LEVEL - zoomLevel));

    QRect rect = sceneRect().toRect();
    rect.setTop(MAP_MIN_PIXEL_Y - overlap);
    rect.setBottom(MAP_MAX_PIXEL_Y + overlap);
    setSceneRect(rect);
}

void MapScene::setTilesDrawingLevels(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__ << "zoomLevel:" << zoomLevel;

    QList<QGraphicsItem *> allItems = items();

    for (int i = 0; i < allItems.size(); ++i) {
        MapTile *item = dynamic_cast<MapTile *>(allItems.at(i));
        if (item)
            item->setSceneLevel(zoomLevel);
    }
}

void MapScene::setTilesGrid(QRect grid)
{
    qDebug() << __PRETTY_FUNCTION__ ;

    m_viewTilesGrid = grid;
}

void MapScene::setZoomLevel(int zoomLevel)
{
    qDebug() << __PRETTY_FUNCTION__ ;

    m_zoomLevel = zoomLevel;
}

void MapScene::spanItems(int zoomLevel, QPoint sceneCoordinate, QSize viewSize)
{
    qDebug() << __PRETTY_FUNCTION__;

    // create rects for left and right side
    QRect leftRect = sceneRect().toRect(); // this way we get the horizontal limits of the scene
    leftRect.setTop(MAP_MIN_PIXEL_Y);
    leftRect.setBottom(MAP_MAX_PIXEL_Y);
    QRect rightRect = leftRect;

    // calculate current horizontal area shown on the view
    int viewSceneWidth = (1 << (MAX_MAP_ZOOM_LEVEL - zoomLevel)) * viewSize.width();
    int viewSceneLeft = sceneCoordinate.x() - viewSceneWidth / 2;
    int viewSceneRight = sceneCoordinate.x() + viewSceneWidth / 2;

    // limit rects to include only area which really must be moved
    leftRect.setRight(-1 - (MAP_PIXELS_X - 1 - viewSceneRight));
    rightRect.setLeft(MAP_PIXELS_X + viewSceneLeft);

    Q_ASSERT_X(leftRect.right() < viewSceneLeft, "spanning rect right value", "move rect is in the view area");
    Q_ASSERT_X(rightRect.left() > viewSceneRight, "spanning rect left value", "move rect is in the view area");

    // move all items which intersects the rects
    if (leftRect.left() < leftRect.right())
        moveIntersectingItemsHorizontally(leftRect, MAP_PIXELS_X);
    if (rightRect.left() < rightRect.right())
        moveIntersectingItemsHorizontally(rightRect, -MAP_PIXELS_X);
}

void MapScene::tilesSceneRectUpdated(QRect tilesSceneRect)
{
    qDebug() << __PRETTY_FUNCTION__;

    m_tilesSceneRect = tilesSceneRect;
}
