/*
   Situare - A location system for Facebook
   Copyright (C) 2010  Ixonos Plc. Authors:

       Sami Rämö - sami.ramo@ixonos.com
       Jussi Laitinen - jussi.laitinen@ixonos.com
       Pekka Nissinen - pekka.nissinen@ixonos.com
       Ville Tiensuu - ville.tiensuu@ixonos.com

   Situare is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   version 2 as published by the Free Software Foundation.

   Situare is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Situare; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
   USA.
*/

#ifndef MAPENGINE_H
#define MAPENGINE_H

#include <QtCore>

class QGraphicsScene;

class FriendItemsHandler;
class GPSLocationItem;
class MapFetcher;
class MapScene;
class MapScroller;
class MapTile;
class OwnLocationItem;
class User;

/**
 * @brief Map engine
 *
 * Logic for controlling map functionality. Does also include static methods for
 * converting coordinates.
 *
 * @author Sami Rämö - sami.ramo (at) ixonos.com
 * @author Jussi Laitinen - jussi.laitinen (at) ixonos.com
 * @author Pekka Nissinen - pekka.nissinen (at) ixonos.com
 * @author Ville Tiensuu - ville.tiensuu (at) ixonos.com
 */
class MapEngine : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     *
     * @param parent Parent
     */
    MapEngine(QObject *parent = 0);

    /**
     * @brief Destructor
     * Saves view of the map to settings file
     */
    ~MapEngine();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public:
    /**
     * @brief Coordinates  of the current center point
     *
     * @return Current coordinates (latitude & longitude)
     */
    QPointF centerGeoCoordinate();

    /**
     * @brief Convert latitude and longitude to scene coordinates.
     *
     * @param latLonCoordinate latitude and longitude values
     * @return scene coordinate
     */
    static QPoint convertLatLonToSceneCoordinate(QPointF latLonCoordinate);

    /**
     * @brief converts scene coordinates to latitude and longitude
     *
     * @param zoomLevel current zoom level
     * @param sceneCoordinate that will be converted
     */
    QPointF convertSceneCoordinateToLatLon(int zoomLevel, QPoint sceneCoordinate);

    /**
     * @brief Convert MapScene coordinate to tile x & y numbers.
     *
     * @param zoomLevel ZoomLevel
     * @param sceneCoordinate MapScene coordinate
     * @return QPoint tile x & y numbers
     */
    static QPoint convertSceneCoordinateToTileNumber(int zoomLevel, QPoint sceneCoordinate);

    /**
     * @brief Convert tile x & y numbers to MapScene coordinates
     *
     * @param zoomLevel Zoom level
     * @param tileNumber x & y numbers of the tile
     * @return QPoint MapScene coordinate
     */
    static QPoint convertTileNumberToSceneCoordinate(int zoomLevel, QPoint tileNumber);

    /**
     * @brief Calculate great-circle distance between two geographic coordinates
     *
     * Calculate great-circle distance between two given geographic locations using
     * haversine formula
     *
     * @param firstLocation Coordinates of the first location
     * @param secondLocation Coordinates of the second location
     * @return qreal Distance in kilometers
     */
    qreal greatCircleDistance(QPointF firstLocation, QPointF secondLocation);

    /**
     * @brief MapEngine initializer
     *
     * Set initial location and zoom level for the engine. locationChanged and
     * zoomLevelChanged signals are emitted, so init should be called after
     * those signals are connected to MapView.
     */
    void init();

    /**
     * @brief Return given value translated between min and max
     *
     * If given value is not inside the given range (min <= value <= max), then the allowed range
     * is adder or subtracted until the value does fit in the range.
     *
     * @param value Value to be normalized
     * @param min Minimum allowed value
     * @param max Maximum allowed value
     * @return value which is moved to be inside the given limits
     */
    static int normalize(int value, int min, int max);

    /**
     * @brief Getter for scene
     *
     * @return QGraphicsScene
     */
    QGraphicsScene* scene();

    /**
     * @brief Sets new zoom level
     *
     * @return newZoomLevel value that is set to new zoom level
     */
    void setZoomLevel(const int newZoomLevel);

    /**
     * @brief Return tile path created from tile values.
     *
     * @param zoomLevel tile's zoom level
     * @param x tile's x value
     * @param y tile's y value
     * @return QString tile path
     */
    static QString tilePath(int zoomLevel, int x, int y);

    /**
     * @brief Maximum number of individual tiles per side at given zoom level
     *
     * @param zoomLevel Zoom level
     * @return amount of tiles per side at given zoom level
     */
    static int tilesPerSide(int zoomLevel);

public slots:
    /**
     * @brief Center smoothly to given latitude and longitude coordinates.
     *
     * @param latLonCoordinate Latitude & longitude coordinates for location
     */
    void centerToCoordinates(QPointF latLonCoordinate);

    /**
     * @brief Slot to catch user own location data
     *
     * @param user User info
     */
    void receiveOwnLocation(User *user);

    /**
     * @brief Set auto centering.
     *
     * @param enabled true if enabled, false otherwise
     */
    void setAutoCentering(bool enabled);

    /**
     * @brief Slot for enabling / disabling GPS
     *
     * GPS location item is disabled or enabled based on GPS state
     *
     * @param enabled True is GPS is enabled, otherwise false
     */
    void setGPSEnabled(bool enabled);

    /**
     * @brief Calculate maximum value for tile in this zoom level.
     *
     * @param zoomLevel zoom level
     * @return int tile's maximum value
     */
    static int tileMaxIndex(int zoomLevel);

    /**
     * @brief Slot for view resizing.
     *
     * @param size view size
     */
    void viewResized(const QSize &size);

private:
    /**
     * @brief Calculate grid of tile coordinates from current scene coordinate.
     *
     * Grid size is calculated from view size and scene's current center coordinate.
     *
     * @param sceneCoordinate scene's current center coordinate
     * @return QRect grid of tile coordinates
     */
    QRect calculateTileGrid(QPoint sceneCoordinate);

    /**
     * @brief Request disabling of auto centering if centered too far from the real location.
     *
     * @param sceneCoordinate scene's center coordinate
     */
    void disableAutoCenteringIfRequired(QPoint sceneCoordinate);

    /**
     * @brief Get new tiles.
     *
     * Calculates which tiles has to be fetched. Does emit fetchImage for tiles which
     * aren't already in the scene.
     * @param sceneCoordinate scene's center coordinate
     */
    void getTiles(QPoint sceneCoordinate);

    /**
     * @brief Check if auto centering is enabled
     *
     * @return true if enabled, false otherwise
     */
    bool isAutoCenteringEnabled();

    /**
     * @brief Check if center tile has changed.
     *
     * @param sceneCoordinate scene's center coordinate
     * @return bool true if center tile changed, false otherwise
     */
    bool isCenterTileChanged(QPoint sceneCoordinate);

    /**
     * @brief Calculate scale at the map center of the map in meters/pixel
     *
     * @return qreal Scale of the map in meters/pixel
     */
    qreal sceneResolution();

    /**
     * @brief Set size of tiles grid based on view size
     *
     * @param viewSize Current view size
     */
    void setTilesGridSize(const QSize &viewSize);

    /**
     * @brief Updates the current view rect including margins
     *
     * Calculates tiles rect in scene based on m_viewTilesGrid and
     * calls MapScene::viewRectUpdated()
     */
    void updateViewTilesSceneRect();

    /**
     * @brief This method is ran always when the map is zoomed
     *
     * This method is the right place for all actions which must be done when ever map is zoomed.
     */
    void zoomed();

private slots:
    /**
     * @brief This slot is called after friend items position have been updated
     *
     * Does run MapScene::spanItems()
     */
    void friendsPositionsUpdated();

    /**
     * @brief Slot for GPS position updates
     *
     * GPS location item is updated and map centered to new location (if automatic
     * centering is enabled).
     *
     * @param position New coordinates from GPS
     * @param accuracy Accuracy of the GPS fix
     */
    void gpsPositionUpdate(QPointF position, qreal accuracy);

    /**
     * @brief Slot for received map tile images
     *
     * Does receive map tile images from MapFetcher. Calls MapScene::addTile() for creating and adding
     * the actual MapTile object to the MapScene.
     *
     * Tile is added also to outside the world horizontal limits, if required, for spanning the map.
     *
     * @param zoomLevel Zoom level
     * @param x Tile x index
     * @param y Tile y index
     * @param image Received pixmap
     */
    void mapImageReceived(int zoomLevel, int x, int y, const QPixmap &image);

    /**
    * @brief Called when MapScroller state is changed
    *
    * Does check if the smooth scroll effect was interrupted and should the auto centering
    * feature to be disabled.
    *
    * @param newState New state
    */
    void scrollerStateChanged(QAbstractAnimation::State newState);

    /**
    * @brief Scroll smoothly to given scene position
    *
    * @param scenePosition Target position in the scene
    */
    void scrollToPosition(QPoint scenePosition);

    /**
     * @brief Set center point in the scene
     *
     * Does emit locationChanged signal.
     * @param scenePosition Scene coordinates for new position
     */
    void setCenterPosition(QPoint scenePosition);

    /**
     * @brief Slot for actions after view zoom is finished
     *
     * Does run removeOutOfViewTiles
     */
    void viewZoomFinished();

    /**
     * @brief Slot for zooming in
     */
    void zoomIn();

    /**
     * @brief Slot for zooming out
     */
    void zoomOut();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signals error
    *
    * @param context error context
    * @param error error code
    */
    void error(const int context, const int error);

    /**
     * @brief Signal for image fetching.
     *
     * @param zoomLevel Zoom level
     * @param x Tile x index
     * @param y Tile y index
     */
    void fetchImage(int zoomLevel, int x, int y);

    /**
     * @brief Signal when friend list locations are fetched
     *
     * @param friendsList Friends list data
     */
    void friendsLocationsReady(QList<User *> &friendsList);

    /**
     * @brief Request view centering to new locaiton
     *
     * @param sceneCoordinate New scene coordinates
     */
    void locationChanged(QPoint sceneCoordinate);

    /**
     * @brief Signal is emitted when location item is clicked.
     *
     * @param userIDs list of friends user IDs in the group
     */
    void locationItemClicked(const QList<QString> &userIDs);

    /**
     * @brief Signal to notify map scrolling.
     */
    void mapScrolledManually();

    /**
     * @brief Signal to notify when map is zoomed in to the maxmimum.
     */
    void maxZoomLevelReached();

    /**
     * @brief Signal to notify when map is zoomed out to the minimum.
     */
    void minZoomLevelReached();

    /**
     * @brief Signal to pass the scale of the map to map scale
     */
    void newMapResolution(qreal scale);

    /**
     * @brief Request view changing zoom level
     *
     * @param newZoomLevel New zoom level
     */
    void zoomLevelChanged(int newZoomLevel);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_autoCenteringEnabled;    ///< Auto centering enabled
    bool m_scrollStartedByGps;      ///< Smooth scroll is started by GPS?
    bool m_smoothScrollRunning;     ///< Smooth scroll is running?
    bool m_zoomedIn;                ///< Flag for checking if zoomed in when zoom is finished

    int m_zoomLevel;                ///< Current zoom level

    QPoint m_centerTile;            ///< Current center tile
    QPoint m_lastAutomaticPosition; ///< Last automatically set position in scene coordinate
    QPoint m_sceneCoordinate;       ///< Current center coordinate

    QRect m_viewTilesGrid;          ///< Current grid of tiles in view (includes margin)

    QSize m_tilesGridSize;          ///< Current size of the tiles grid
    QSize m_viewSize;               ///< Current view size

    FriendItemsHandler *m_friendItemsHandler;   ///< Handler for friend and group items
    GPSLocationItem *m_gpsLocationItem;         ///< Item pointing current location from GPS
    MapFetcher *m_mapFetcher;                   ///< Fetcher for map tiles
    MapScene *m_mapScene;                       ///< Scene for map tiles
    MapScroller *m_scroller;                    ///< Kinetic scroller
    OwnLocationItem *m_ownLocation;             ///< Item to show own location
};

#endif // MAPENGINE_H
