 /*
    Situare - A location system for Facebook
    Copyright (C) 2010  Ixonos Plc. Authors:

        Kaj Wallin - kaj.wallin@ixonos.com
        Henri Lampela - henri.lampela@ixonos.com
        Jussi Laitinen - jussi.laitinen@ixonos.com
        Sami Rämö - sami.ramo@ixonos.com

    Situare is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    Situare is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Situare; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
 */


#ifndef ENGINE_H
#define ENGINE_H

#include <QObject>
#include <QTime>
#include <QPointF>

class QMainWindow;

class FacebookAuthentication;
class FacebookCredentials;
class GPSPosition;
class MainWindow;
class MapEngine;
class NetworkAccessManager;
class SituareService;
class User;
class MCE;

class QTimer;

/**
* @brief Engine class for Situare Application
*
* This class handles all the underlaying login of the Situare
* application.
*/
class SituareEngine : public QObject
{
    Q_OBJECT
public:
    /**
    * @brief Constructor
    *
    * @param parent
    */
    SituareEngine(QMainWindow *parent = 0);

    /**
    * @brief Destructor
    */
    ~SituareEngine();

/*******************************************************************************
 * MEMBER FUNCTIONS AND SLOTS
 ******************************************************************************/
public slots:
    /**
    * @brief Slot to intercept error signal from ImageFetcher and SituareService
    *
    * @param context Error context
    * @param error Error message
    */
    void error(const int context, const int error);

    /**
    * @brief Slot to intercept signal when username is fetched from settings
    *
    */
    void fetchUsernameFromSettings();

    /**
    * @brief Slot to intercept signal when Login/Logout action is pressed
    *
    */
    void loginActionPressed();

    /**
    * @brief Slot to intercept signal from successful login
    *
    */
    void loginOk();

    /**
    * @brief Slot to intercept signal when user has cancelled login process
    */
    void loginProcessCancelled();

    /**
    * @brief Changes application state when logged out
    *
    */
    void logout();

    /**
    * @brief Calls reverseGeo from SituareService to translate coordinates to street address
    *
    */
    void requestAddress();

    /**
    * @brief Calls updateLocation from SituareService to send the location update to
    *        Situare server.
    *
    * @param status Status message
    * @param publish Publish on Facebook
    */
    void requestUpdateLocation(const QString &status = QString(), bool publish = false);

    /**
    * @brief Slot to refresh user data
    */
    void refreshUserData();

    /**
    * @brief Slot to intercept signal from successful location update
    *
    */
    void updateWasSuccessful();

    /**
    * @brief Slot to intercept signal when new user data is available.
    *        Splits User and friendsList data and emits them as two different signals.
    *
    * @param user instance of User
    * @param friendsList list of User instances (friends)
    */
    void userDataChanged(User *user, QList<User *> &friendsList);

private:
    /**
    * @brief Read settings and determine whether to use GPS and autocentering.
    * When values does not found on the settings, GPS and autocentering are enabled as a default.
    */
    void initializeGpsAndAutocentering();

    /**
      * @brief Connect signals coming from Facdebook authenticator
      */
    void signalsFromFacebookAuthenticator();

    /**
      * @brief Connect signals coming from GPS
      */
    void signalsFromGPS();

    /**
      * @brief Connect signals coming from MainWindow
      */
    void signalsFromMainWindow();

    /**
      * @brief Connect signals coming from MapEngine
      */
    void signalsFromMapEngine();

    /**
      * @brief Connect signals coming from MapView
      */
    void signalsFromMapView();

    /**
      * @brief Connect signals coming from Situare
      */
    void signalsFromSituareService();

private slots:
    /**
    * @brief Set auto centering feature enabled / disabled
    *
    * @param enabled true if enabled, false otherwise
    */
    void changeAutoCenteringSetting(bool enabled);

    /**
    * @brief Slot for disabling automatic centering when map is scrolled manually
    */
    void disableAutoCentering();

    /**
    * @brief Slot for display state changed.
    *
    * @param enabled true if enabled, false otherwise
    */
    void displayStateChanged(bool enabled);

    /**
    * @brief Slot for auto centering enabling.
    *
    * Calls gps to send last known position
    *
    * @param enabled true if auto centering was enabled, false otherwise
    */
    void enableAutoCentering(bool enabled);

    /**
    * @brief Slot for gps enabling.
    *
    * @param enabled true if gps should be enabled, false otherwise
    */
    void enableGPS(bool enabled);

    /**
    * @brief Enables automatic location update.
    *
    * @param enabled true if enabled, false otherwise
    * @param updateIntervalMsecs update interval in milliseconds
    */
    void enableAutomaticLocationUpdate(bool enabled, int updateIntervalMsecs = 0);

    /**
    * @brief Requests automatic update.
    *
    * Makes automatic location update request if user has moved enough.
    *
    * @param position geo coordinates
    */
    void requestAutomaticUpdateIfMoved(QPointF position);

    /**
     * @brief Sets zoom level to default when first GPS location is received if autocentering
     * is enabled.
     *
     * @param latLonCoordinate own location
     * @param accuracy accuracy of GPS location
     */
    void setFirstStartZoomLevel(QPointF latLonCoordinate, qreal accuracy);

    /**
    * @brief Automatic update interval timer timeout.
    *
    * Requests update location if user has moved.
    */
    void startAutomaticUpdate();

/*******************************************************************************
 * SIGNALS
 ******************************************************************************/
signals:
    /**
    * @brief Signals when automatic location update was enabled.
    *
    * @param enabled true if enabled, false otherwise
    */
    void automaticLocationUpdateEnabled(bool enabled);

    /**
    * @brief Signal to clear locationUpdateDialog's data
    *
    */
    void clearUpdateLocationDialogData();

    /**
    * @brief Signals when new friends data is ready
    *
    * @param friendList List of User instances (friends)
    */
    void friendsLocationsReady(QList<User *> &friendList);

    /**
    * @brief Signals when new user data is ready
    *
    * @param user Instance of User
    */
    void userLocationReady(User *user);

/*******************************************************************************
 * DATA MEMBERS
 ******************************************************************************/
private:
    bool m_autoCenteringEnabled;        ///< Auto centering flag
    bool m_automaticUpdateFirstStart;   ///< Automatic location update first start flag
    bool m_automaticUpdateRequest;      ///< Flag for automatic update request
    bool m_userMoved;                   ///< Flag for user move


    FacebookAuthentication *m_facebookAuthenticator; ///< Instance for facebook authenticator
    GPSPosition *m_gps;                              ///< Instance of the gps position
    MainWindow *m_ui;                                ///< Instance of the MainWindow UI
    MapEngine *m_mapEngine;                          ///< MapEngine
    NetworkAccessManager *m_networkAccessManager;    ///< NetworkAccessManager
    SituareService *m_situareService;  ///< Instance of the situare server communication service
    MCE *m_mce;                        ///< Instance of the MCE

    QTimer *m_automaticUpdateIntervalTimer; ///< Automatic update interval timer
    QPointF m_lastUpdatedGPSPosition;       ///< Last updated GPS position
};

#endif // ENGINE_H
