/*
 * simpler-brightness-applet - simple statusarea applet that allows the user to select the brightness from a range of 1-5
 *
 * A fork of simple-brightness-applet Copyright (c) 2009-2010 Faheem Pervez <trippin1@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *    
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *    
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

//#ifdef HAVE_CONFIG_H
#include <config.h>
//#endif

#include "simpler-brightness-applet.h"

#include <glib/gi18n-lib.h>
#include <string.h>
#include <libosso.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>

#ifdef HILDON_DISABLE_DEPRECATED
#undef HILDON_DISABLE_DEPRECATED
#endif

HD_DEFINE_PLUGIN_MODULE (SimplerBrightnessApplet, simpler_brightness_applet, HD_TYPE_STATUS_MENU_ITEM)

#define SIMPLER_BRIGHTNESS_GCONF_PATH "/system/osso/dsm/display"
#define BRIGHTNESS_KEY SIMPLER_BRIGHTNESS_GCONF_PATH "/display_brightness"

#define SIMPLER_BRIGHTNESS_APPLET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                        SIMPLER_TYPE_BRIGHTNESS_APPLET, SimplerBrightnessAppletPrivate))

struct _SimplerBrightnessAppletPrivate
{
    GConfClient *gconf_client;
    GtkHBox *applet_contents;
    HildonControlbar *brightness_ctrlbar;
    GtkWidget *settings_dialog;
    GtkWidget *dispchkbtn;
    GtkWidget *applet_button;

    gulong brightness_ctrlbar_valchanged_id;
    gulong dispchkbtn_toggled_id;
    guint gconfnotify_id;
    guint display_keepalive_timeout;

    gboolean keepdispon_active;

    osso_context_t *osso_context;
};

static void simpler_brightness_applet_finalize (GObject *object);

/* Callbacks: */
static inline void simpler_brightness_applet_show_fail_msg (void)
{
    hildon_banner_show_information (NULL, NULL, _("Failed to keep backlight on"));
}

static void simpler_brightness_applet_show_status_area_icon (SimplerBrightnessApplet *plugin, gboolean show)
{
    if (show)
    {
        GdkPixbuf *pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (), "general_brightness", 18, 0, NULL);
        if (pixbuf)
        {
            hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), pixbuf);
            g_object_unref (pixbuf);
            return;
        }
    }
    hd_status_plugin_item_set_status_area_icon (HD_STATUS_PLUGIN_ITEM (plugin), NULL);
}

static inline void simpler_brightness_applet_colour_button (GtkWidget *button, gboolean colour)
{
    gtk_widget_set_name (button, colour ? "hildon-reject-button-finger" : "GtkButton-finger");
}

static gboolean simpler_brightness_applet_keep_backlight_alive (SimplerBrightnessApplet *plugin)
{
#if 0
    hildon_banner_show_information (NULL, NULL, G_STRFUNC);
#endif
    osso_display_state_on (plugin->priv->osso_context);
    if (osso_display_blanking_pause (plugin->priv->osso_context) != OSSO_OK)
    {
        simpler_brightness_applet_show_fail_msg ();
        plugin->priv->display_keepalive_timeout = 0;
        if (plugin->priv->dispchkbtn)
            g_signal_handler_block (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
        plugin->priv->keepdispon_active = FALSE;
        if (plugin->priv->dispchkbtn)
        {
            hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), FALSE);
            g_signal_handler_unblock (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
        }
        return FALSE;
    }
    return TRUE;
}

static void simpler_brightness_applet_on_gconf_value_changed (GConfClient *gconf_client G_GNUC_UNUSED, guint cnxn_id G_GNUC_UNUSED, GConfEntry *entry G_GNUC_UNUSED, SimplerBrightnessApplet *plugin)
{
    g_signal_handler_block (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);
    hildon_controlbar_set_value (plugin->priv->brightness_ctrlbar, gconf_client_get_int (plugin->priv->gconf_client, BRIGHTNESS_KEY, NULL));
    g_signal_handler_unblock (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);
}

static void simpler_brightness_applet_on_value_changed (HildonControlbar *brightness_ctrlbar G_GNUC_UNUSED, SimplerBrightnessApplet *plugin)
{
    gconf_client_set_int (plugin->priv->gconf_client, BRIGHTNESS_KEY, hildon_controlbar_get_value (plugin->priv->brightness_ctrlbar), NULL);
}

static void simpler_brightness_applet_on_dispchkbtn_toggled (GtkWidget *button G_GNUC_UNUSED, SimplerBrightnessApplet *plugin)
{
    plugin->priv->keepdispon_active = hildon_check_button_get_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn));
    g_signal_handler_block (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
    gtk_widget_hide(plugin->priv->settings_dialog);

    if (plugin->priv->keepdispon_active)
    {
        simpler_brightness_applet_keep_backlight_alive (plugin);
        plugin->priv->display_keepalive_timeout = g_timeout_add_seconds (50, (GSourceFunc) simpler_brightness_applet_keep_backlight_alive, plugin);
        if (plugin->priv->display_keepalive_timeout == 0)
        {
            plugin->priv->keepdispon_active = FALSE;
            hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), FALSE);
            simpler_brightness_applet_show_fail_msg ();
            goto finish;
        }

        simpler_brightness_applet_colour_button (plugin->priv->applet_button, plugin->priv->keepdispon_active);
        simpler_brightness_applet_show_status_area_icon (plugin, plugin->priv->keepdispon_active);
    }
    else
    {
        if (g_source_remove (plugin->priv->display_keepalive_timeout))
            plugin->priv->display_keepalive_timeout = 0;
        else
            plugin->priv->keepdispon_active = TRUE;

        hildon_check_button_set_active (HILDON_CHECK_BUTTON (plugin->priv->dispchkbtn), plugin->priv->keepdispon_active);
        simpler_brightness_applet_show_status_area_icon (plugin, plugin->priv->keepdispon_active);
        simpler_brightness_applet_colour_button (plugin->priv->applet_button, plugin->priv->keepdispon_active);
    }

finish:
    g_signal_handler_unblock (plugin->priv->dispchkbtn, plugin->priv->dispchkbtn_toggled_id);
    gtk_widget_destroy(plugin->priv->settings_dialog);
    plugin->priv->dispchkbtn = NULL;
}

/* Setup applet and dialogs: */
static void simpler_brightness_applet_on_button_clicked (GtkWidget *widget, SimplerBrightnessApplet *plugin)
{
    gtk_widget_hide (gtk_widget_get_toplevel (widget));

    if (!plugin->priv->keepdispon_active)
    {
        simpler_brightness_applet_keep_backlight_alive (plugin);
        plugin->priv->display_keepalive_timeout = g_timeout_add_seconds (50, (GSourceFunc) simpler_brightness_applet_keep_backlight_alive, plugin);
        if (!plugin->priv->display_keepalive_timeout)
        {
            simpler_brightness_applet_show_fail_msg ();
            return;
        }

        plugin->priv->keepdispon_active = TRUE;
        simpler_brightness_applet_show_status_area_icon (plugin, plugin->priv->keepdispon_active);
        simpler_brightness_applet_colour_button (plugin->priv->applet_button, plugin->priv->keepdispon_active);
    }
    else
    {
        if (g_source_remove (plugin->priv->display_keepalive_timeout))
        {
            plugin->priv->keepdispon_active = FALSE;
            plugin->priv->display_keepalive_timeout = 0;
            simpler_brightness_applet_show_status_area_icon (plugin, plugin->priv->keepdispon_active);
            simpler_brightness_applet_colour_button (plugin->priv->applet_button, plugin->priv->keepdispon_active);
        }
    }
}

static void simpler_brightness_applet_on_tap_and_hold (GtkWidget *widget, SimplerBrightnessApplet *plugin)
{
    gtk_widget_hide (gtk_widget_get_toplevel (widget));

    gtk_widget_hide (plugin->priv->settings_dialog);

    if (plugin->priv->osso_context)
        osso_cp_plugin_execute (plugin->priv->osso_context, "libcpdisplay.so", NULL, TRUE);

    gtk_widget_destroy (plugin->priv->settings_dialog);
    plugin->priv->dispchkbtn = NULL;
}

static void simpler_brightness_applet_setup (SimplerBrightnessApplet *plugin)
{
    plugin->priv->applet_contents = GTK_HBOX (gtk_hbox_new (FALSE, 0));

    plugin->priv->applet_button = hildon_gtk_button_new (HILDON_SIZE_FINGER_HEIGHT);
    gtk_box_pack_start (GTK_BOX (plugin->priv->applet_contents), plugin->priv->applet_button, FALSE, TRUE, 0);
    g_signal_connect (plugin->priv->applet_button, "clicked", G_CALLBACK(simpler_brightness_applet_on_button_clicked), plugin);
    gtk_button_set_image (GTK_BUTTON (plugin->priv->applet_button), gtk_image_new_from_icon_name ("general_brightness", HILDON_ICON_SIZE_THUMB));
    gtk_widget_tap_and_hold_setup (plugin->priv->applet_button, NULL, NULL, GTK_TAP_AND_HOLD_NONE);
    g_signal_connect (plugin->priv->applet_button, "tap-and-hold", G_CALLBACK (simpler_brightness_applet_on_tap_and_hold), plugin);

    plugin->priv->brightness_ctrlbar = HILDON_CONTROLBAR (hildon_controlbar_new ()); /* Yes, I know: I'm very naughty for using a depreciated widget. But let me know, Nokia, when you actually stop using it too. */
    hildon_gtk_widget_set_theme_size(GTK_WIDGET(plugin->priv->brightness_ctrlbar), HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);
    hildon_controlbar_set_range (plugin->priv->brightness_ctrlbar, gconf_client_get_int(plugin->priv->gconf_client, "/system/osso/dsm/display/display_brightness_level_step", NULL), gconf_client_get_int(plugin->priv->gconf_client, "/system/osso/dsm/display/max_display_brightness_levels", NULL));
    hildon_controlbar_set_value (plugin->priv->brightness_ctrlbar, gconf_client_get_int(plugin->priv->gconf_client, BRIGHTNESS_KEY, NULL));
    plugin->priv->brightness_ctrlbar_valchanged_id = g_signal_connect (plugin->priv->brightness_ctrlbar, "value-changed", G_CALLBACK(simpler_brightness_applet_on_value_changed), plugin);
    gtk_box_pack_start (GTK_BOX (plugin->priv->applet_contents), GTK_WIDGET (plugin->priv->brightness_ctrlbar), FALSE, TRUE, 0);

    gconf_client_add_dir (plugin->priv->gconf_client, SIMPLER_BRIGHTNESS_GCONF_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
    plugin->priv->gconfnotify_id = gconf_client_notify_add (plugin->priv->gconf_client, BRIGHTNESS_KEY, (GConfClientNotifyFunc)simpler_brightness_applet_on_gconf_value_changed, plugin, NULL, NULL);
}

/* GObject stuff: */
static void simpler_brightness_applet_class_init (SimplerBrightnessAppletClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->finalize = (GObjectFinalizeFunc) simpler_brightness_applet_finalize;

    g_type_class_add_private (klass, sizeof (SimplerBrightnessAppletPrivate));
}

static void simpler_brightness_applet_class_finalize (SimplerBrightnessAppletClass *klass G_GNUC_UNUSED) {}

static void simpler_brightness_applet_init (SimplerBrightnessApplet *plugin)
{
    plugin->priv = SIMPLER_BRIGHTNESS_APPLET_GET_PRIVATE (plugin);
    if (G_UNLIKELY (!plugin->priv))
        return;
    memset (plugin->priv, 0, sizeof (SimplerBrightnessAppletPrivate));

    bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8"); 

    plugin->priv->keepdispon_active = FALSE;

    plugin->priv->osso_context = osso_initialize (PACKAGE, PACKAGE_VERSION, TRUE, NULL);
    if (G_UNLIKELY (!plugin->priv->osso_context))
        return;

    plugin->priv->gconf_client = gconf_client_get_default ();
    if (G_UNLIKELY (!plugin->priv->gconf_client))
        return;

    simpler_brightness_applet_setup (plugin);
    gtk_container_add (GTK_CONTAINER (plugin), GTK_WIDGET (plugin->priv->applet_contents));
    gtk_widget_show_all (GTK_WIDGET (plugin));
}

static void simpler_brightness_applet_finalize (GObject *object)
{
    SimplerBrightnessApplet *plugin = SIMPLER_BRIGHTNESS_APPLET (object);

    if (plugin->priv->brightness_ctrlbar_valchanged_id != 0)
        g_signal_handler_disconnect (plugin->priv->brightness_ctrlbar, plugin->priv->brightness_ctrlbar_valchanged_id);

    if (plugin->priv->display_keepalive_timeout != 0)
        g_source_remove (plugin->priv->display_keepalive_timeout);

    if (plugin->priv->gconfnotify_id != 0)
    {
        gconf_client_notify_remove (plugin->priv->gconf_client, plugin->priv->gconfnotify_id);
        gconf_client_remove_dir (plugin->priv->gconf_client, SIMPLER_BRIGHTNESS_GCONF_PATH, NULL);
    }

    if (plugin->priv->gconf_client)
    {
        gconf_client_clear_cache (plugin->priv->gconf_client);
        g_object_unref (plugin->priv->gconf_client);
    }

    if (plugin->priv->osso_context)
        osso_deinitialize (plugin->priv->osso_context);

    G_OBJECT_CLASS (simpler_brightness_applet_parent_class)->finalize (object);
}
