#include <QFont>
#include <QPainter>
#include <QPen>
#include <QDebug>

#include "conceptschart.h"
#include "concept.h"

#define LEGEND_CONCEPTS_PER_COL 10
#define LEGEND_ITEM_HEIGHT 30
#define LEGEND_RECT_SIZE 15

#define SMALL_MARGIN 0
#define NORMAL_MARGIN 50
#define EXTRA_MARGIN 100

ConceptsChart::ConceptsChart(QWidget *parent)
        : QWidget(parent)
{
        _padding = 10;
        _data = NULL;
}

void ConceptsChart::setPadding(int padding)
{
        _padding = padding;
        repaint();
}

void ConceptsChart::setSummaryData(SummaryData *data)
{
        _data = data;
}

QSize ConceptsChart::minimumSizeHint() const
{
        return QSize(200, 100);
}

QSize ConceptsChart::sizeHint() const
{
        return QSize(400, 200);
}

void ConceptsChart::paintEvent (QPaintEvent *)
{
        QPainter painter(this);
        int margin;

        painter.save();

        QFont font = painter.font();
        font.setPointSize(10);
        painter.setFont(font);
        painter.setRenderHint(QPainter::Antialiasing);
        painter.fillRect(0, 0, width(), height(), QColor(Qt::black));

        if (_data != NULL && _data->concepts().size() > 0) {
                legendCols = ((_data->concepts().size() - 1) / LEGEND_CONCEPTS_PER_COL) + 1;

                if (legendCols == 1) {
                        margin = EXTRA_MARGIN;
                } else {
                        if (legendCols ==  2) {
                                margin = NORMAL_MARGIN;
                        } else {
                                margin = SMALL_MARGIN;
                        }
                }

                xStart = _padding + margin;
                xEnd = width() - (_padding);
                sceneWidth = xEnd - xStart;

                yStart = height() - _padding;
                yEnd = _padding;
                sceneHeight = yStart - yEnd;

                legendXStart = xStart + sceneHeight + _padding + margin +_padding;
                legendWidth = xEnd - legendXStart;
                legendColWidth = legendWidth / legendCols;
                colorStep = 0xff / _data->concepts().size();

                drawPie(painter);
        }

        painter.restore ();
}

void ConceptsChart::drawPie(QPainter &painter)
{
        QPen pen;
        QBrush brush;
        int startAngle;
        int angle;
        int red;
        int green;
        int blue;

        pen.setColor(Qt::black);
        painter.setPen(pen);
        brush.setStyle(Qt::SolidPattern);

        startAngle = 0;
        red = 0xff;
        green = 0;
        blue = 0;
        brush.setColor(QColor(red, green, blue));
        painter.setBrush(brush);
        for (int i = 0; i < _data->concepts().size(); i++) {
                angle = qRound(360 * (_data->concepts()[i]->price() / _data->totalExpense()));
                painter.drawPie(xStart,
                                yEnd,
                                sceneHeight,
                                sceneHeight,
                                16 * startAngle,
                                16 * angle);
                drawLegend(painter, i);

                startAngle += angle;
                nextColor(red, green, blue);
                brush.setColor(QColor(red, green, blue));
                painter.setBrush(brush);
                painter.setPen(Qt::black);
        }
}

void ConceptsChart::drawLegend(QPainter &painter, int i)
{
        int xPosition;
        int yPosition;
        int row;
        int col;

        row = i % LEGEND_CONCEPTS_PER_COL;
        col = i / LEGEND_CONCEPTS_PER_COL;
        xPosition = legendXStart + (col * legendColWidth);
        yPosition = yEnd + (row * LEGEND_ITEM_HEIGHT);

        painter.drawRect(xPosition,
                         yPosition,
                         LEGEND_RECT_SIZE,
                         LEGEND_RECT_SIZE);
        painter.setPen(Qt::white);

        QString label;

        if (!_data->concepts()[i]->name().isEmpty()) {
                label = QString("%1 (%2)")
                        .arg(_data->concepts()[i]->name())
                        .arg(_data->concepts()[i]->price());
        } else {
                label = QString("%1 (%2)")
                        .arg(tr("No item"))
                        .arg(_data->concepts()[i]->price());
        }

        painter.drawText(xPosition + LEGEND_RECT_SIZE + _padding,
                         yPosition,
                         legendColWidth - (LEGEND_RECT_SIZE + _padding + _padding),
                         LEGEND_ITEM_HEIGHT,
                         Qt::AlignLeft | Qt::AlignTop,
                         label);
}

void ConceptsChart::nextColor(int &red, int &green, int &blue)
{
        red -= colorStep;
        green += colorStep;
        blue += colorStep;
}
