#include "scheduleddetailsdialog.h"
#include <QFormLayout>
#include <QLabel>
#include <QDebug>
#include <QStandardItem>
#include <QMaemo5InformationBox>
#include <QDate>
#include <QHBoxLayout>

ScheduledDetailsDialog::ScheduledDetailsDialog(QStandardItemModel *concepts,
                                               QWidget *parent,
                                               Qt::WindowFlags f)
        : QDialog (parent, f)
{
        monthSelectionAsInt = 0;
        newScheduled = true;
        scheduledId = 0;

        setWindowTitle(tr("New scheduled expense"));

        buildUI(concepts);


        QPushButton *saveButton = new QPushButton(tr("Save"));
        box->addButton(saveButton, QDialogButtonBox::ActionRole);

        connect(saveButton, SIGNAL(clicked()), this, SLOT(checkValues()));
}

ScheduledDetailsDialog::ScheduledDetailsDialog(QStandardItemModel *concepts,
                                               QStandardItemModel *scheduled,
                                               int row,
                                               QWidget *parent,
                                               Qt::WindowFlags f)
        : QDialog (parent, f)
{
        scheduledId = scheduled->item(row, 0)->text().toInt();
        monthSelectionAsInt = scheduled->item(row, 5)->text().toInt();
        newScheduled = false;

        setWindowTitle(tr("Scheduled expense details"));

        buildUI(concepts);

        QPushButton *updateButton = new QPushButton(tr("Save"));
        box->addButton(updateButton, QDialogButtonBox::ActionRole);
        QPushButton *deleteButton = new QPushButton(tr("Delete"));
        box->addButton(deleteButton, QDialogButtonBox::ActionRole);

        /* day */
        daySelector->setCurrentIndex(scheduled->item(row, 1)->text().toInt() - 1);

        /* amount */
        amount->setText(scheduled->item(row, 2)->text());

        /* description */
        description->setText(scheduled->item(row, 4)->text());

        /* concept */
        /* concept may be empty, so we need to check it */
        if (!scheduled->item(row, 3)->text().isEmpty()) {
                QList<QStandardItem*> list = concepts->findItems(scheduled->item(row, 3)->text(),
                                                                 Qt::MatchFixedString,
                                                                 0);
                conceptSelector->setCurrentIndex(list[0]->index().row());
        }

        /* frecuency */
        month->setValueText(Expense::frecuencyAsString(monthSelectionAsInt));

        connect(updateButton, SIGNAL(clicked()), this, SLOT(checkValues()));
        connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));
}

void ScheduledDetailsDialog::buildUI (QStandardItemModel *concepts)
{
        QHBoxLayout *layout = new QHBoxLayout (this);

        QFormLayout *form = new QFormLayout ();
        form->setHorizontalSpacing(50);
        layout->addLayout (form, 1);

        QStandardItemModel *dayModel = new QStandardItemModel (0, 1, this);
        for (int i = 1; i <= 31; i++) {
                QStandardItem *item = new QStandardItem (QString("%1").arg(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                dayModel->appendRow(item);
        }
        day = new QMaemo5ValueButton(tr("Day").toLatin1().data());
        day->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        daySelector = new QMaemo5ListPickSelector();
        daySelector->setModel(dayModel);
        day->setPickSelector(daySelector);
        form->addRow (day);

        month = new QMaemo5ValueButton(tr("Repeat").toLatin1().data());
        month->setValueLayout(QMaemo5ValueButton::ValueUnderText);
        connect (month, SIGNAL(clicked()), this, SLOT(launchMonthDialog()));
        form->addRow (month);

        concept = new QMaemo5ValueButton(tr("Item").toLatin1().data());
        concept->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        conceptSelector = new QMaemo5ListPickSelector();
        conceptSelector->setModel(concepts);
        conceptSelector->setModelColumn(0);
        concept->setPickSelector(conceptSelector);
        form->addRow (concept);

        connect (conceptSelector, SIGNAL(selected(const QString&)), this, SLOT(conceptSelected(const QString&)));

        amount = new QLineEdit();
        form->addRow (new QLabel(tr("Amount")), amount);

        description = new QLineEdit();
        form->addRow (new QLabel(tr("Description")), description);

        box = new QDialogButtonBox (QDialogButtonBox::Cancel, Qt::Vertical);
        connect(box, SIGNAL(rejected()), this, SLOT(reject()));
        layout->addWidget (box, 0, Qt::AlignBottom);
}

ScheduledDetailsDialog::~ScheduledDetailsDialog()
{
}

void ScheduledDetailsDialog::checkValues()
{
        Expense *e;

        int _day = day->valueText().toInt();

        if (_day == 0) {
                QMaemo5InformationBox::information(this, tr("Invalid day"));
                return;
        }

        QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
        int conceptIndex = conceptSelector->currentIndex();
        QString conceptName;
        if (conceptIndex >= 0) {
                QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                conceptName.append(m->item(conceptIndex, 0)->text());
        }

        QString _description(description->text());
        if (conceptIndex < 0 && _description.isEmpty()) {
                QMaemo5InformationBox::information(this, tr("Choose a concept or enter a description"));
                return;
        }

        double _amount(amount->text().toDouble());
        if (_amount <= 0) {
                QMaemo5InformationBox::information(this, tr("Invalid amount"));
                return;
        }

        if (monthSelectionAsInt == 0) {
                QMaemo5InformationBox::information(this, tr("Select at least one month for the expense"));
                return;
        }

        /* check that day-month combinations are ok */
        int i;
        for (i = 0; i < 12; i++) {
                if (monthSelectionAsInt & (1 << i)) {
                        if (!QDate::isValid(QDate::currentDate().year(), i + 1, _day)
                            && !(_day == 29 && i == 1)) {
                                QMaemo5InformationBox::information(this,
                                                                   QString(tr("Day %1 in month %2 is not a valid date"))
                                                                   .arg(_day)
                                                                   .arg(QDate::longMonthName(i + 1)));
                                return;
                        }
                }
        }

        if (conceptIndex < 0) {
                e = new Expense (scheduledId, _day, _amount, "", _description, monthSelectionAsInt);
        } else {
                /* amount and concept */
                e = new Expense (scheduledId, _day, _amount, conceptName, _description, monthSelectionAsInt);
        }

        accept();
        if (newScheduled) {
                emit newScheduledConfirmed(e);
        } else {
                emit updateScheduledConfirmed(e);
        }
}

void ScheduledDetailsDialog::deleteClicked()
{
        emit deleteScheduledConfirmed (new Expense (scheduledId,0,0,"","",0));
        accept();
}

void ScheduledDetailsDialog::conceptSelected(const QString&)
{
        if (amount->text().isEmpty()) {
                QMaemo5ListPickSelector *conceptSelector = dynamic_cast<QMaemo5ListPickSelector*>(concept->pickSelector());
                int conceptIndex = conceptSelector->currentIndex();
                if (conceptIndex >= 0) {
                        QStandardItemModel *m = dynamic_cast<QStandardItemModel*>(conceptSelector->model());
                        amount->setText(m->item(conceptIndex, 1)->text());
                }
        }
}

void ScheduledDetailsDialog::launchMonthDialog()
{
        monthDialog = new QDialog(this);

        monthDialog->setWindowTitle(tr("Select months to perform the expense"));
        monthDialog->setMinimumHeight(400);

        QHBoxLayout *layout = new QHBoxLayout (monthDialog);

        monthModel = new QStandardItemModel (0, 1, monthDialog);
        for (int i=1; i<= 12; i++) {
                QStandardItem *item = new QStandardItem (QDate::longMonthName(i));
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                monthModel->appendRow(item);
        }

        monthList = new QListView();
        monthList->setModel(monthModel);
        monthList->setProperty("FingerScrollable", true);
        monthList->setSelectionMode(QAbstractItemView::MultiSelection);
        monthSelection = monthList->selectionModel();

        int i;
        for (i = 0; i < 12; i++) {
                if (monthSelectionAsInt & (1 << i)) {
                        QModelIndex index = monthModel->index(i, 0);
                        monthSelection->select(index, QItemSelectionModel::Select);
                }
        }

        layout->addWidget(monthList, 1);

        QDialogButtonBox *box = new QDialogButtonBox();
        box->addButton(new QPushButton(tr("Save")), QDialogButtonBox::AcceptRole);
        box->addButton(QDialogButtonBox::Cancel);
        layout->addWidget (box, 0, Qt::AlignBottom);

        connect(box, SIGNAL(accepted()), this, SLOT(saveMonthSelection()));
        connect(box, SIGNAL(rejected()), this, SLOT(closeMonthDialog()));

        monthDialog->exec();
}

void ScheduledDetailsDialog::saveMonthSelection()
{
        QModelIndexList indexes = monthSelection->selectedIndexes();
        QModelIndex index;

        monthSelectionAsInt = 0;

        foreach(index, indexes) {
                monthSelectionAsInt |= 1 << index.row();
        }
        month->setValueText(Expense::frecuencyAsString(monthSelectionAsInt));

        closeMonthDialog();
}

void ScheduledDetailsDialog::closeMonthDialog()
{
        monthDialog->accept();
        delete monthDialog;
}
