#include <hildon/hildon.h>
#include <gucharmap/gucharmap.h>
#include "charmap-dialog.h"
#include "maemo-gucharmap-chartable.h"

typedef enum
{
	CHARMAP_DIALOG_SORT_BY_SCRIPT = 0,
	CHARMAP_DIALOG_SORT_BY_CODE_BLOCK
} CharmapDialogSortMode;

struct _CharmapDialog
{
	GtkDialog __parent_instance__;

	GtkWidget *chap_view;
	GtkWidget *chart;
	GtkWidget *chart_hpa;
	GtkWidget *result;

	/* Workaround for disappearing selection */
	gboolean sel_info_set;
	gint sel_bound, cursor_pos;
};

struct _CharmapDialogClass
{
	GtkDialogClass __parent_class__;
};

enum {
	CHARMAP_DIALOG_CHAR_STRING_PROPERTY = 1
};

static void charmap_dialog_init(CharmapDialog *dlg);
static void charmap_dialog_class_init(CharmapDialogClass *dlg_class);

G_DEFINE_TYPE(CharmapDialog, charmap_dialog, GTK_TYPE_DIALOG);

static void
get_property(GObject *obj, guint property_id, GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
			break;

		case CHARMAP_DIALOG_CHAR_STRING_PROPERTY:
			{
				char *str;

				g_return_if_fail(IS_CHARMAP_DIALOG(obj));

				str = charmap_dialog_get_char_string(CHARMAP_DIALOG(obj));
				g_value_set_string(val,  str);
				g_free(str);
				break;
			}
	}
}

static void
set_property(GObject *obj, guint property_id, const GValue *val, GParamSpec *pspec)
{
	switch (property_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);

		case CHARMAP_DIALOG_CHAR_STRING_PROPERTY:
			g_return_if_fail(IS_CHARMAP_DIALOG(obj));

			charmap_dialog_set_char_string(CHARMAP_DIALOG(obj), g_value_get_string(val));
			break;
	}
}

static void
charmap_dialog_class_init(CharmapDialogClass *dlg_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(dlg_class);

	gobject_class->get_property = get_property;
	gobject_class->set_property = set_property;

	g_object_class_install_property(gobject_class, CHARMAP_DIALOG_CHAR_STRING_PROPERTY,
		g_param_spec_string("char-string", "Character String", "Unicode character as a NULL-terminated string",
			NULL, G_PARAM_READWRITE));
}

static void
charmap_dialog_set_sort_mode(CharmapDialog *dlg, CharmapDialogSortMode sort_mode)
{
	GString *str = g_string_new("");
	GucharmapChaptersModel *model = 
		(CHARMAP_DIALOG_SORT_BY_SCRIPT == sort_mode)
			? gucharmap_script_chapters_model_new ()
			: gucharmap_block_chapters_model_new ();
	gunichar active_char = gucharmap_chartable_get_active_character(GUCHARMAP_CHARTABLE(dlg->chart));

	hildon_touch_selector_set_model(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(dlg->chap_view)), 0, GTK_TREE_MODEL(model));

	g_string_append_unichar(str, active_char);
	charmap_dialog_set_char_string(dlg, str->str);
	g_string_free(str, TRUE);
}

static void
sort_picker_value_changed(HildonPickerButton *btn, CharmapDialog *dlg)
{
	charmap_dialog_set_sort_mode(dlg, hildon_picker_button_get_active(btn));
}

static GtkWidget *
make_sort_picker(CharmapDialog *dlg)
{
	GtkWidget *btn = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	HildonTouchSelector *sel = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());

	hildon_touch_selector_append_text(sel, "Script");
	hildon_touch_selector_append_text(sel, "Unicode Block");
	hildon_touch_selector_set_active(sel, 0, 0);

	gtk_widget_show(btn);
	hildon_button_set_title(HILDON_BUTTON(btn), "Sort Mode");
	hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(btn), sel);
	g_signal_connect(G_OBJECT(btn), "value-changed", (GCallback)sort_picker_value_changed, dlg);

	return btn;
}

static void
chap_changed(HildonPickerButton *btn, CharmapDialog *dlg)
{
	HildonTouchSelector *sel = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(btn));
	GtkTreeModel *tm = hildon_touch_selector_get_model(sel, 0);
	GucharmapChaptersModel *model = GUCHARMAP_CHAPTERS_MODEL(tm);
	GtkTreeIter itr;

	if (hildon_touch_selector_get_selected(sel, 0, &itr))
		gucharmap_chartable_set_codepoint_list(GUCHARMAP_CHARTABLE(dlg->chart),
			gucharmap_chapters_model_get_codepoint_list(model, &itr));
}

static void
do_insert(GtkEditable *editable, char *str)
{
	int pos_cur, pos_sel;

	g_object_get(G_OBJECT(editable), "cursor-position", &pos_cur, "selection-bound", &pos_sel, NULL);
	if (pos_cur != pos_sel) {
		gtk_editable_delete_selection(editable);
		g_object_get(G_OBJECT(editable), "cursor-position", &pos_cur, NULL);
	}
	gtk_editable_insert_text(editable, str, -1, &pos_cur);
	gtk_editable_select_region(editable, pos_cur - g_utf8_strlen(str, -1), pos_cur);
}

static void
chart_notify_active_character(GucharmapChartable *chart, GParamSpec *pspec, CharmapDialog *dlg)
{
	GString *str = g_string_new("");
	gunichar c = gucharmap_chartable_get_active_character(chart);

	g_string_append_unichar(str, c);

	if (dlg->sel_info_set) {
		gtk_editable_select_region(GTK_EDITABLE(dlg->result), MIN(dlg->cursor_pos, dlg->sel_bound), MAX(dlg->cursor_pos, dlg->sel_bound));
		dlg->sel_info_set = FALSE;
	}
	do_insert(GTK_EDITABLE(dlg->result), str->str);

	g_string_free(str, TRUE);
}

static gboolean
backup_sel_info_on_button_press(GtkWidget *hpa, GdkEventButton *event, CharmapDialog *dlg)
{
	g_object_get(dlg->result,
		"selection-bound", &(dlg->sel_bound),
		"cursor-position", &(dlg->cursor_pos),
		NULL);
	dlg->sel_info_set = TRUE;

	return FALSE;
}

static void
result_notify_text(GObject *result, GParamSpec *pspec, GObject *dlg)
{
	g_object_notify(dlg, "char-string");
}

static void
charmap_dialog_init(CharmapDialog *dlg)
{
	GtkListStore *ls;
	GtkWidget *sel;
	GtkWidget *hbox, *sort_picker;
	HildonTouchSelectorColumn *col;
	GtkSizeGroup *sg;

	dlg->sel_info_set = FALSE;

	sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

	hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, NULL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), hbox, FALSE, TRUE, 0);

	sort_picker = make_sort_picker(dlg);
	gtk_box_pack_start(GTK_BOX(hbox), sort_picker, TRUE, TRUE, 0);
	gtk_size_group_add_widget(sg, sort_picker);
	g_signal_connect(G_OBJECT(sort_picker), "button-press-event", (GCallback)backup_sel_info_on_button_press, dlg);

	sel = g_object_new(HILDON_TYPE_TOUCH_SELECTOR, NULL);
	ls = gtk_list_store_new(1, G_TYPE_STRING);
	col = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(sel), GTK_TREE_MODEL(ls), gtk_cell_renderer_text_new(), "text", 1, NULL);
	g_object_set(col, "text-column", 1, NULL);
	g_object_unref(ls);
	dlg->chap_view = g_object_new(HILDON_TYPE_PICKER_BUTTON,
		"visible",        TRUE,
		"arrangement",    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
		"size",           HILDON_SIZE_FINGER_HEIGHT,
		"style",          HILDON_BUTTON_STYLE_PICKER,
		"title",          "Chapter",
		"touch-selector", sel,
		NULL);
	g_signal_connect(G_OBJECT(dlg->chap_view), "value-changed", (GCallback)chap_changed, dlg);
	gtk_box_pack_start(GTK_BOX(hbox), dlg->chap_view, TRUE, TRUE, 0);
	g_signal_connect(G_OBJECT(dlg->chap_view), "button-press-event", (GCallback)backup_sel_info_on_button_press, dlg);
	gtk_size_group_add_widget(sg, dlg->chap_view);

	g_object_unref(sg);

	gtk_box_pack_start(GTK_BOX(hbox),
		g_object_new(GTK_TYPE_LABEL, "visible", TRUE, "label", "Result:", NULL),
		FALSE, TRUE, 0);

	dlg->result = g_object_new(HILDON_TYPE_ENTRY, "visible", TRUE, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), dlg->result, FALSE, TRUE, 0);
	g_signal_connect(G_OBJECT(dlg->result), "notify::text", (GCallback)result_notify_text, dlg);

	dlg->chart = g_object_new(MAEMO_GUCHARMAP_CHARTABLE_TYPE, "visible", TRUE, NULL);
	dlg->chart_hpa = g_object_new(HILDON_TYPE_PANNABLE_AREA,
		"visible",        TRUE,
		"child",          dlg->chart,
		"hovershoot-max", 0,
		"vovershoot-max", 0,
		NULL);
	g_signal_connect(G_OBJECT(dlg->chart_hpa), "button-press-event", (GCallback)backup_sel_info_on_button_press, dlg);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), dlg->chart_hpa, TRUE, TRUE, 0);
	gtk_widget_grab_focus(dlg->chart);

	g_signal_connect(G_OBJECT(dlg->chart), "notify::active-character", (GCallback)chart_notify_active_character, dlg);

	hildon_picker_button_set_active(HILDON_PICKER_BUTTON(sort_picker), CHARMAP_DIALOG_SORT_BY_CODE_BLOCK);
}

char *
charmap_dialog_get_char_string(CharmapDialog *dlg)
{
	return g_strdup(gtk_entry_get_text(GTK_ENTRY(dlg->result)));
}

void
charmap_dialog_set_char_string(CharmapDialog *dlg, const char *str)
{
	HildonTouchSelector *sel = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(dlg->chap_view));
	GucharmapChaptersModel *model = GUCHARMAP_CHAPTERS_MODEL(hildon_touch_selector_get_model(sel, 0));
	GtkTreeIter itr;
	gunichar new_char = g_utf8_get_char_validated(str, -1);
	
	if (gucharmap_chapters_model_character_to_iter(model, new_char, &itr)) {
		hildon_touch_selector_select_iter(sel, 0, &itr, FALSE);
		gucharmap_chartable_set_codepoint_list(GUCHARMAP_CHARTABLE(dlg->chart),
			gucharmap_chapters_model_get_codepoint_list(model, &itr));
		gucharmap_chartable_set_active_character(GUCHARMAP_CHARTABLE(dlg->chart), new_char);
	}

	gtk_entry_set_text(GTK_ENTRY(dlg->result), str);
	gtk_editable_select_region(GTK_EDITABLE(dlg->result), 0, -1);
}
