#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
===============================================================
    Contains deifinition of class Message and states for
state-field in database (table Messages)
===============================================================
"""

__author__ = "Mezhenin Artoym <mezhenin@cs.karelia.ru>"
__version__ = "$Revision: 17 $"
__date__ = "$Date: 2010/02/07 $"
__copyright__ = ""
__license__ = "GPLv2"


from ScriboExc import WarningExc
from Structure import Structure
from SqlDriver import sql_driver, SQL_MESSAGE_FIELDS

"""
===============================================================
    States for type-field in database (table Messages)
===============================================================
"""

DRAFT = 1    # this message is draft
POST = 2     # read 
COMMENT = 3  # comment
INBOX = 4
OUTBOX = 5
POSTER_F = "poster"
TEXT_F =  "text"
READ_F =  "read"
SERVICE_F = "service"
DATE_F = "date"
SORT_DESC_DATE = " ORDER BY date DESC"




class Message(Structure):

    """
        Class for posts and comments. Contain methods to set and get 
    fields from table Messages. All Messages store comments for posts 
    and other comments by making link in field 'parent' to id of post 
    
    @author Mezhenin Artoym <mezhenin@cs.karelia.ru> 
    @version 0.3
    @date 2010/01/12
    """

    def __init__(self, id=None):
        """
        Constuctor for class.
        
        @param id (string) id of message in table. If there is no id given,
                           it will be created.
        @return None
        """
        super(Message, self).__init__("Messages", id)
        if not id:
            self.type = DRAFT
            self.unread_cmnts = 0
            self.read = False


    """
    ===============================================================
        Getter methods
    ===============================================================
    """

    def get_title(self):
        """
        Get field from Messages table.

        @return (string) title, None
        """
        return self._get_field("title")


    def get_text(self):
        """
        Get field from Messages table.

        @return (string) text, None
        """
        return self._get_field("text")


    def get_date(self):
        """
        Get field from Messages table.

        @return (string) date, None
        """
        return self._get_field("date")


    def get_poster(self):
        """
        Get field from Messages table. Create Account of user.

        @return (Account) poster, None
        """

        from Account import Account
        res = self._get_field("poster")
        if res:
            return Account(res)

        return None

    def get_journal(self):
        """
        Get field from Messages table. Create Account of user.

        @return (Account) journal, None
        """

        from Account import Account
        res = self._get_field("journal")
        if res:
            return Account(res)

        return None


    def get_path_to_ava(self):
        """
        Get field from Messages table.

        @return (string) path to avatar, None
        """
        return self._get_field("path_to_ava")


    def get_unread_cmnts(self):
        """
        Get field from Messages table.

        @return (int) number of unread comments (for posts)
        @exception WorningExc field is not integer (or other reason)
        """
        return int(self._get_field("unread_cmnts"))



    def get_parent(self):
        """
        Get field from Messages table.

        @return (Message) parent for this message, None
        """
        res = self._get_field("parent")
        if res:
            return Message(int(res))

        return None


    def get_read(self):
        """
        Get field from Messages table. 

        @return (bool) True if message is read
        """

        return bool(self._get_field("read"))


    def get_type(self):
        """
        Get field from Messages table. 

        @return (int) type of message (post, comment, draft...)
        """

        return int(self._get_field("type"))


    def get_tags(self):
        """
        Get field from Messages table.

        @return (string) tags to msg, None
        """
        return self._get_field("tags")


    def get_url(self):
        """
        Get field from Messages table.

        @return (string) url of message, None
        """
        return self._get_field("url")


    def get_custom_id(self):
        """
        Get field from Messages table.

        @return (string) custom_id, None
        """
        return self._get_field("custom_id")


    def get_comments(self):
        """
        Get all comments to this message. Note that this comments may have their
        own comments
        
        @return (list) list of Messages with comments to this message
        """
        query = 'SELECT id FROM Messages WHERE parent=? ORDER BY custom_id'
        id_list = sql_driver.select_mcmd(query, (self.id,))

        comments = []
        for i in id_list:
            comments.append(Message(i[0]))

        return comments

    """
    ===============================================================
        Setter methods
    ===============================================================
    """

    def set_title(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) new title
        @return None
        """
        self._set_field("title", value)


    def set_text(self, value):
        """
        Change field value in Messages table.

        @param value (string) new text
        @return None
        """
        self._set_field("text", value)

    def set_date(self, value):
        """
        Change field value in Messages table.

        @param value (string) new date
        @return None
        """
        self._set_field("date", value)

    def set_poster(self, value):
        """
        Change field value in Messages table.

        @param value (Account) new poster
        @return None
        """

        from Account import Account

        if isinstance(value, Account):
            self._set_field("poster", value.id)
        elif value == None:
            self._set_field("poster", None)
        else:
            raise WarningExc("Message setter",
                             "value is not Account or None")

    def set_journal(self, value):
        """
        Change field value in Messages table.

        @param value (Account) new journal
        @return None
        """

        from Account import Account

        if isinstance(value, Account):
            self._set_field("journal", value.id)
        elif value == None:
            self._set_field("journal", None)
        else:
            raise WarningExc("Message setter",
                             "value is not Account or None")


    def set_read(self, value):
        """
        Change field value in Messages table.

        @param value (bool) state message (read or unread) 
        @return None, WarningExc if value is not bool
        """
        if type(value) != bool:
            raise WarningExc("Message setter",
                             "value is not bool")

        if COMMENT == self.type and value != self.read:
            post = self.parent
            while post.parent:
                post = post.parent

            if self.read:
                post.unread_cmnts += 1
            else:
                post.unread_cmnts -= 1

        self._set_field("read", value)


    def set_type(self, value):
        """
        Change field value in Messages table.

        @param value (int) new type for message(post, comment, draft ...) 
        @return None, WarningExc if value is not integer
        """
        if type(value) == int:
            self._set_field("type", value)
        else:
            raise WarningExc("Message setter",
                             "value is not int")


    def set_path_to_ava(self, value):
        """
        Change field value in Messages table.

        @param value (string) new path to avatar
        @return None
        """
        self._set_field("path_to_ava", value)


    def set_unread_cmnts(self, value):
        """
        Change field value in Messages table.

        @param value (int) new number of unread comments
        @return None
        """

        if type(value) == int:
            self._set_field("unread_cmnts", value)
        else:
            raise WarningExc("Message setter",
                             "value is not int")


    def set_parent(self, value):
        """
        Change field value in Messages table.

        @param value (Message) parent message 
        @return None
        """
        if type(value) == Message:
            self._set_field("parent", value._id)
        elif value == None:
            self._set_field("parent", None)
        else:
            raise WarningExc("Message setter",
                             "value is not Message or None")


    def set_tags(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) new tags (separated by commas)
        @return None
        """
        self._set_field("tags", value)


    def set_url(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) new message url
        @return None
        """
        self._set_field("url", value)


    def set_custom_id(self, value):
        """
        Change field value in Messages table.
        
        @param value (string) custom_id
        @return None
        """
        self._set_field("custom_id", value)




    """
    ===============================================================
        Init getter/setter  methods
    ===============================================================
    """

    ## property
    title = property(get_title, set_title)
    ## property
    text = property(get_text, set_text)
    ## property
    date = property(get_date, set_date)
    ## property
    poster = property(get_poster, set_poster)
    ## property
    journal = property(get_journal, set_journal)
    ## property
    path_to_ava = property(get_path_to_ava, set_path_to_ava)
    ## property
    unread_cmnts = property(get_unread_cmnts, set_unread_cmnts)
    ## property
    parent = property(get_parent, set_parent)
    ## property
    comments = property(get_comments)
    ## property
    read = property(get_read, set_read)
    ## property
    type = property(get_type, set_type)
    ## property
    tags = property(get_tags, set_tags)
    ## property
    url = property(get_url, set_url)
    ## property
    custom_id = property(get_custom_id, set_custom_id)


    """
    ===============================================================
        Other methods
    ===============================================================
    """

    def del_msg_tree(self):
        """
        Depth-first search with deleting messages. (This message will be root.)
        
        @return None
        """
        query = 'SELECT id FROM Messages WHERE parent=?'
        id_list = sql_driver.select_mcmd(query, (self._id,))

        for i in id_list:
            Message(i[0]).del_msg_tree()

        self._del_message()


    def add_comment(self, comment, post=None):
        """
        Add comment to this message.
        
        @param comment (Message) comment that we want to add 
        @param post (Message) post to this message (to self) 
        @return None
        @exception WarningExc post is None, but self is not a 'post' 
        """
        if not post:
            if POST == self.type:
                post = self
            else:
                raise WarningExc("Message.add_comment",
                                 "post is unknown")

        comment.parent = self
        comment.read = False
        comment.type = COMMENT
        # incremen number of unread comments for post
        post.unread_cmnts += 1


    def _del_message(self):
        """
        Delete record from DB
        
        @return None
        """
        # when we deleting unread comment we need to dec unread_cmnts for post
        self.read = False
        self._del_struct()


    @staticmethod
    def find(fetch_one=True, custom_id=None, where=None):
        req = 'SELECT id FROM Messages WHERE '
        args = ()

        if custom_id:
            req = req + 'custom_id=?'
            args = (custom_id,)
        elif where:
            req = req + where
        else:
            raise WarningExc("Message find",
                             "no option set")

        if fetch_one:
            try:
                res = sql_driver.select_cmd(req, args)
                return Message(res)
            except WarningExc:
                return None
        else:
            id_list = sql_driver.select_mcmd(req, args)
            msg_list = []
            for i in id_list:
                msg_list.append(Message(i[0]))

            return msg_list


    def copy(self):
        """
        Make full copy of this message.
        
        @return (Message) new message with exact values
        """
        return Message(self._copy_fields(SQL_MESSAGE_FIELDS))


    def is_editable(self):
        """
        Datermines that we can edit this post. We have posts in our journal(we
        can edit), post in friends journal(can't edit) and posts in groups(some
        of them we can edit because we post them)
        
        @return (bool) True if we can edit this message, False in other way
        """

        if not self.poster or \
           self.poster.passwd == None or \
           len(self.poster.passwd) == 0:
            return False

        return True
 
    @staticmethod
    def filter_msg(field, first_value = None, second_value = None, already_found = None, poster = None):
        list_post_filters = []
        cur_list = []
        if poster:
           whr = 'journal='+str(poster)+' and '
        else:
           whr = ''

        if already_found:
            whr +='('
            for i in xrange(0,len(already_found)-1):
                whr += 'id='+ str(already_found[i].id) + ' or '
            whr += 'id='+ str(already_found[-1].id) + ') and '


        if field == POSTER_F:
            if first_value:
                whr += 'poster='+str(first_value)+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)
        elif field == TEXT_F:
            if first_value:
                first_value = unicode(first_value).lstrip().rstrip().replace(" ","%")
                whr += '(text like \'%'+first_value+'%\' or tags like \'%'+\
                      first_value+'%\' or title like \'%'+first_value+'%\')'+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)
        elif field == READ_F:
            if first_value:
                whr += 'read='+str(first_value)+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)          
        elif field == SERVICE_F:
             if first_value:
                subwhr = '(SELECT id FROM Accounts WHERE service_id='+first_value+' and prof ='+ str(1)+')'
                whr += 'poster='+ subwhr+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list) 
        elif field == DATE_F:
            if first_value and second_value:
                whr += '(date >=\''+str(first_value)+'\' and date <=\''+str(second_value)+'\')'+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)                   
            elif first_value:
                whr += 'date >=\''+str(first_value)+'\''+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)       
            elif second_value:
                whr += 'date <=\''+str(second_value)+'\''+SORT_DESC_DATE
                cur_list = Message.find(fetch_one = False, where = whr)
                if cur_list:
                    list_post_filters.extend(cur_list)  
        return list_post_filters     
        
    @staticmethod
    def sort_msg(value, sort, already_found, poster = None):
        list_sort_msg = []
        whr = ""
    
        if already_found:
            whr +='('
            for i in xrange(0,len(already_found)-1):
                whr += 'id='+ str(already_found[i].id) + ' or '
            whr += 'id='+ str(already_found[-1].id) + ')'
        
        if poster:
            if already_found:
                whr += ' and '
            whr += '('
            for i in xrange(0,len(poster)-1):
                whr += 'journal='+str(poster[i])+' or '
            whr += 'journal='+str(poster[-1])+')'

        whr +=' ORDER BY '+value+" "+ sort
        print whr
        list_sort_msg = Message.find(fetch_one = False, where = whr)

        return list_sort_msg


        
    """
    ===============================================================
        Comments support
    ===============================================================
    """

    def is_deleted(self):
        """
        Check comment for deleting
        @return True if comment is marked for deletind or False otherwise
        """
        
        if COMMENT <> self.type:
            return False
            #raise TypeError("Illegal message type", "Message is not comment")
          
        sm=self.misc
        return sm['deleted']
            

    def has_undeleted_childes(self):
        """
        Look for undeleted childes
        
        @return True if comment have undeleted childes or False otherwise
        """
        if COMMENT <> self.type:
            raise TypeError("Illegal message type", "Message is not comment")
          
        comm = self.get_comments()
        for i in comm:
            if not i.is_deleted():
                return True
            if i.has_undeleted_childes():
                return True
        return False
        
        
    def del_comment_local(self):
        """
        Delete comment sub-tree localy
        
        @return None
        """
        if COMMENT <> self.type:
            raise TypeError("Illegal message type", "Message is not comment")
          
        tmp = self.misc
        tmp['deleted'] = True
        self.misc = tmp
        if self.has_undeleted_childes():
            self.text = '(Deleted Comment)'
            self.title = ''
        else:
            par = self.parent
            if not par:
                return
            self.del_msg_tree()
            if par.is_deleted():
                par.del_comment_local()
    
    def clean_list_marked(self,  mark_type, newlist):  
        if self.misc[mark_type] and (self.custom_id not in newlist):
            self.__mark_childs(mark_type,False)
            i = self.misc[mark_type+"_next"]
            tmp = self.misc
            del tmp[mark_type+"_next"]
            self.misc = tmp
            if i==None:
                return None
            return Message(i).clean_list_marked(mark_type,newlist)
        else:
            if self.misc[mark_type+"_next"]<>None:
                tmp = self.misc
                tmp[mark_type+"_next"] = Message(tmp[mark_type+"_next"]).clean_list_marked(mark_type,newlist)
                self.misc = tmp
            return self.id
        
        
        
    def have_frozen_parent(self):
        return self._have_marked_parent('frozen',True)
        
    def have_screened_parent(self):
        return self._have_marked_parent('screened',True)     
        
    def _have_marked_parent(self,mark,val):
        if COMMENT <> self.type:
            raise TypeError("Illegal message type", "Message is not comment")
          
        if self.parent.type == POST:
            return False
        
        if self.parent.misc[mark]==val:
            return True
            
        

    
    
    def froze_childs(self):
        self.__mark_childs('frozen',True)
        
    def unfroze_childs(self):
        if self.have_frozen_parent():
            raise TypeError("Illegal message type", "Message have frozen comments")
        self.__mark_childs('frozen',False)
    
    def screen_childs(self):
        self.__mark_childs('screened',True)
        
    def unscreen_childs(self):
        if self.have_screened_parent():
            raise TypeError("Illegal message type", "Message have screened comments")
        self.__mark_childs('screened',False)    
            
      
    def __mark_childs(self,field,flag):
        if COMMENT <> self.type:
            raise TypeError("Illegal message type", "Message is not comment")
          
        tmp = self.misc
        tmp[field] = flag
        self.misc = tmp 
        
        comm = self.get_comments()
        for i in comm:
            i.__mark_childs(field,flag)

        
    