/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "calendar-search-settings.h"
#include <libintl.h>
#include <locale.h>
#include <CMulticalendar.h>
#include <vector>
#include <hildon/hildon.h>
#include <gconf/gconf-client.h>	

#include "time-frame-selection.h"

static GtkWidget*
_create_calendar_colour_icon_image(CalendarColour colour, GtkIconSize icon_size)
{
  switch(colour)
  {
  case COLOUR_DARKBLUE:
    return gtk_image_new_from_icon_name("calendar_colors_darkblue", icon_size);
  case COLOUR_DARKGREEN:
    return gtk_image_new_from_icon_name("calendar_colors_darkgreen", icon_size);
  case COLOUR_DARKRED:
    return gtk_image_new_from_icon_name("calendar_colors_darkred", icon_size);
  case COLOUR_ORANGE:
    return gtk_image_new_from_icon_name("calendar_colors_orange", icon_size);
  case COLOUR_VIOLET:
    return gtk_image_new_from_icon_name("calendar_colors_violet", icon_size);
  case COLOUR_YELLOW:
    return gtk_image_new_from_icon_name("calendar_colors_yellow", icon_size);
  case COLOUR_BLUE:
    return gtk_image_new_from_icon_name("calendar_colors_blue", icon_size);
  case COLOUR_RED:
    return gtk_image_new_from_icon_name("calendar_colors_red", icon_size);
  case COLOUR_GREEN:
    return gtk_image_new_from_icon_name("calendar_colors_green", icon_size);
  default:
    return gtk_image_new_from_icon_name("calendar_colors_white", icon_size);
  }
}


static gint
_load_num_calendars(calendar_search_settings_t* calendar_search_settings)
{
  if(calendar_search_settings->num_calendars == 0)
    return 0;
  vector<CCalendar*> calendar_list = CMulticalendar::MCInstance()->getListCalFromMc();
  gint count = 0;

  for(int i = 0;i<calendar_list.size();++i)
  {
    for(int j=0;j<calendar_search_settings->num_calendars;++j)
    {
      if(calendar_list[i]->getCalendarId() == calendar_search_settings->calendar_ids[j])
      {
	count++;
	continue;
      }
    }
  }
  CMulticalendar::MCInstance()->releaseListCalendars(calendar_list);
  return count;
}

static GtkWidget**
_create_list_of_calendar_toggle_buttons(const vector<CCalendar*>& calendar_list, 
					GtkWidget* container, 
					calendar_search_settings_t* calendar_search_settings)
{
  gint cal_count = calendar_list.size();
  GtkWidget** cal_toggle_buttons = g_new0(GtkWidget*, cal_count);
  for(int i = 0;i<cal_count;++i)
  {
    cal_toggle_buttons[i] = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);

    gtk_button_set_label(GTK_BUTTON(cal_toggle_buttons[i]), 
			 dgettext("calendar", calendar_list[i]->getCalendarName().c_str()));

    GtkWidget* image = _create_calendar_colour_icon_image(calendar_list[i]->getCalendarColor(),
							  HILDON_ICON_SIZE_FINGER);

    gtk_button_set_image(GTK_BUTTON(cal_toggle_buttons[i]), image);
    gtk_button_set_alignment(GTK_BUTTON(cal_toggle_buttons[i]),0 ,0);
    gtk_box_pack_start(GTK_BOX(container), cal_toggle_buttons[i], TRUE, TRUE, 0);

    for(int j=0;j<calendar_search_settings->num_calendars;++j)
    {
      if(calendar_list[i]->getCalendarId() == calendar_search_settings->calendar_ids[j])
      {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cal_toggle_buttons[i]), TRUE);
	continue;
      }
    }
  }
  return cal_toggle_buttons;
}

static void
_store_calendar_search_settings(calendar_search_settings_t* calendar_search_settings)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  g_return_if_fail(GCONF_IS_CLIENT(client));
  gconf_client_set_bool(client, 
			GCONF_CALENDAR_SETTINGS_SEARCH_EVENTS, 
			calendar_search_settings->search_events, 
			NULL);

  gconf_client_set_bool(client, 
			GCONF_CALENDAR_SETTINGS_SEARCH_TASKS, 
			calendar_search_settings->search_tasks,
			NULL);
  gconf_client_set_bool(client, 
			GCONF_CALENDAR_SETTINGS_SEARCH_NOTES, 
			calendar_search_settings->search_notes,
			NULL);

  gconf_client_set_bool(client, 
			GCONF_CALENDAR_SETTINGS_SEARCH_BIRTHDAYS, 
			calendar_search_settings->search_birthdays,
			NULL);

  gconf_client_set_int(client, GCONF_CALENDAR_SETTINGS_NUM_CALENDARS, 
		       calendar_search_settings->num_calendars,
		       NULL);

  gconf_client_set_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_TYPE, 
		       calendar_search_settings->search_type, 
		       NULL);
  
  gconf_client_set_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_START_TIME, 
		       calendar_search_settings->start_time, 
		       NULL);
  
  gconf_client_set_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_END_TIME, 
		       calendar_search_settings->end_time, 
		       NULL);

  if(calendar_search_settings->num_calendars!=0)
  {
    GSList* cal_list = NULL;
    for(int i = 0;i<calendar_search_settings->num_calendars;++i)
    {
      cal_list = g_slist_append(cal_list, GINT_TO_POINTER(calendar_search_settings->calendar_ids[i]));
    }
    gconf_client_set_list(client,
			  GCONF_CALENDAR_SETTINGS_CALENDAR_IDS,
			  GCONF_VALUE_INT,
			  cal_list,
			  NULL);
    g_slist_free(cal_list);
  }
  g_object_unref(client);  
}

void
show_calendar_id_selection_dlg(GtkButton* button, gpointer user_data)
{
  calendar_search_settings_t* calendar_search_settings = (calendar_search_settings_t*)user_data;
  GtkWidget* dialog;
  dialog = gtk_dialog_new_with_buttons("Calendar Search",
				       NULL,
				       GTK_DIALOG_NO_SEPARATOR,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), -1, 350);

  vector<CCalendar*> calendar_list = CMulticalendar::MCInstance()->getListCalFromMc();
  GtkWidget* cal_button_box = gtk_vbox_new(TRUE, 3);
  GtkWidget** cal_toggle_buttons = _create_list_of_calendar_toggle_buttons(calendar_list, 
									   cal_button_box, 
									   calendar_search_settings);

  GtkWidget* pan = hildon_pannable_area_new();
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pan), cal_button_box);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), pan, TRUE, TRUE, 0);
  gtk_widget_show_all(dialog);
  int result = gtk_dialog_run(GTK_DIALOG(dialog));
  if(result == GTK_RESPONSE_ACCEPT)
  {
    int active_count = 0;
    int cal_count = calendar_list.size();
    for(int i = 0;i<cal_count;++i)
    {
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cal_toggle_buttons[i])))
      {
	active_count++;
      }
    }
    g_free(calendar_search_settings->calendar_ids);
    calendar_search_settings->calendar_ids = g_new0(gint, active_count);
    calendar_search_settings->num_calendars = 0;
    for(int i = 0;i<cal_count;++i)
    {
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cal_toggle_buttons[i])))
      {
	calendar_search_settings->calendar_ids[calendar_search_settings->num_calendars] = 
	  calendar_list[i]->getCalendarId();
	calendar_search_settings->num_calendars++;
      }
    }
    gchar* num_cal_text = g_strdup_printf("(%d)", _load_num_calendars(calendar_search_settings));
    hildon_button_set_value(HILDON_BUTTON(button),
			    num_cal_text);
    g_free(num_cal_text);
  }
  g_free(cal_toggle_buttons);
  gtk_widget_destroy(dialog);
  CMulticalendar::MCInstance()->releaseListCalendars(calendar_list);
}

void
show_calendar_search_settings_dlg(GtkButton* button, gpointer user_data)
{
  calendar_search_settings_t* calendar_search_settings = (calendar_search_settings_t*)user_data;
  GtkWidget* dialog;
  dialog = gtk_dialog_new_with_buttons("Calendar Search",
				       NULL,
				       GTK_DIALOG_NO_SEPARATOR,
				       dgettext("hildon-libs", "wdgt_bd_save"),
				       GTK_RESPONSE_ACCEPT,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_OK,
				       NULL);
  GtkWidget* search_events_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* search_tasks_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* search_notes_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* search_birthdays_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* cal_id_picker_button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(search_events_button),
			       calendar_search_settings->search_events);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(search_tasks_button),
			       calendar_search_settings->search_tasks);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(search_notes_button),
			       calendar_search_settings->search_notes);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(search_birthdays_button),
			       calendar_search_settings->search_birthdays);
  GtkWidget* enable_time_frame_search_button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
								 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title(HILDON_BUTTON(enable_time_frame_search_button),
			  dgettext("scout", "Interval"));
  search_time_settings_t* search_time_settings = g_new0(search_time_settings_t,1);

  search_time_settings->search_type = calendar_search_settings->search_type;
  search_time_settings->start_time = calendar_search_settings->start_time;
  search_time_settings->end_time = calendar_search_settings->end_time;

  gchar* time_frame_value = time_frame_value_str(search_time_settings);
  hildon_button_set_value(HILDON_BUTTON(enable_time_frame_search_button),
			  time_frame_value);
  g_free(time_frame_value);

  g_signal_connect(enable_time_frame_search_button, "clicked", G_CALLBACK(show_time_selection_dialog), search_time_settings);
		   
  gchar* num_cal_text = g_strdup_printf("(%d)", _load_num_calendars(calendar_search_settings));
  hildon_button_set_title(HILDON_BUTTON(cal_id_picker_button),
			  "Calendars");
  
  hildon_button_set_value(HILDON_BUTTON(cal_id_picker_button),
			  num_cal_text);
  g_free(num_cal_text);
  g_signal_connect(cal_id_picker_button, "clicked", G_CALLBACK(show_calendar_id_selection_dlg), calendar_search_settings);
  gtk_button_set_image(GTK_BUTTON(search_events_button),
		       gtk_image_new_from_icon_name("general_calendar",
						    GtkIconSize(HILDON_ICON_SIZE_FINGER)));
  
  gtk_button_set_image(GTK_BUTTON(search_tasks_button),
		       gtk_image_new_from_icon_name("calendar_todo",
						    GtkIconSize(HILDON_ICON_SIZE_FINGER)));
  gtk_button_set_image(GTK_BUTTON(search_notes_button),
		       gtk_image_new_from_icon_name("general_notes",
						    GtkIconSize(HILDON_ICON_SIZE_FINGER)));
  
  gtk_button_set_image(GTK_BUTTON(search_birthdays_button),
		       gtk_image_new_from_icon_name("calendar_birthday",
						    GtkIconSize(HILDON_ICON_SIZE_FINGER)));
  GtkWidget* cal_type_box = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(cal_type_box), search_events_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cal_type_box), search_tasks_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cal_type_box), search_notes_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(cal_type_box), search_birthdays_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), cal_type_box, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), cal_id_picker_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), enable_time_frame_search_button, TRUE, TRUE, 0);
  gtk_widget_show_all(dialog);
  int result = gtk_dialog_run(GTK_DIALOG(dialog));
  if(result == GTK_RESPONSE_ACCEPT ||
     result == GTK_RESPONSE_OK)
  {
    calendar_search_settings->search_events = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(search_events_button));
    calendar_search_settings->search_tasks = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(search_tasks_button));
    calendar_search_settings->search_notes = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(search_notes_button));
    calendar_search_settings->search_birthdays = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(search_birthdays_button));
    calendar_search_settings->search_type = search_time_settings->search_type;
    calendar_search_settings->start_time = search_time_settings->start_time;
    calendar_search_settings->end_time = search_time_settings->end_time;
    if(result == GTK_RESPONSE_ACCEPT)
    {
      _store_calendar_search_settings(calendar_search_settings);
    }
  }
  g_free(search_time_settings);
  gtk_widget_destroy(dialog);
}

void
calendar_search_settings_read(GConfClient* client, calendar_search_settings_t* calendar_search_settings)
{
  if(gconf_client_dir_exists(client, GCONF_CALENDAR_SEARCH_SETTINGS, NULL))
  {
    calendar_search_settings->search_events = 
      gconf_client_get_bool(client, GCONF_CALENDAR_SETTINGS_SEARCH_EVENTS, NULL);
    calendar_search_settings->search_tasks = 
      gconf_client_get_bool(client, GCONF_CALENDAR_SETTINGS_SEARCH_TASKS, NULL);
    calendar_search_settings->search_notes = 
      gconf_client_get_bool(client, GCONF_CALENDAR_SETTINGS_SEARCH_NOTES, NULL);
    calendar_search_settings->search_birthdays = 
      gconf_client_get_bool(client, GCONF_CALENDAR_SETTINGS_SEARCH_BIRTHDAYS, NULL);
    calendar_search_settings->num_calendars = 
      gconf_client_get_int(client, GCONF_CALENDAR_SETTINGS_NUM_CALENDARS, NULL);

    calendar_search_settings->search_type = 
      gconf_client_get_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_TYPE, NULL);
    calendar_search_settings->start_time = 
      gconf_client_get_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_START_TIME, NULL);
    calendar_search_settings->end_time = 
      gconf_client_get_int(client, GCONF_CALENDAR_SETTINGS_SEARCH_END_TIME, NULL);

    GSList* cal_list = gconf_client_get_list(client, GCONF_CALENDAR_SETTINGS_CALENDAR_IDS, GCONF_VALUE_INT, NULL);
    if(cal_list && g_slist_length(cal_list) == calendar_search_settings->num_calendars)
    {
      calendar_search_settings->calendar_ids = g_new0(gint, calendar_search_settings->num_calendars);
      for(int i = 0;i<calendar_search_settings->num_calendars;++i)
      {
	calendar_search_settings->calendar_ids[i] = (gint)g_slist_nth(cal_list, i)->data;
      }
      g_slist_free(cal_list);
    }
  }
  else
  {
    vector<CCalendar*> calendar_list = CMulticalendar::MCInstance()->getListCalFromMc();
    calendar_search_settings->num_calendars = calendar_list.size();
    calendar_search_settings->calendar_ids = g_new0(gint, calendar_search_settings->num_calendars);
    for(int i = 0;i<calendar_search_settings->num_calendars;++i)
    {
      calendar_search_settings->calendar_ids[i] = 
	calendar_list[i]->getCalendarId();
    }
    CMulticalendar::MCInstance()->releaseListCalendars(calendar_list);
    calendar_search_settings->search_events = TRUE;
    calendar_search_settings->search_tasks = TRUE;
    calendar_search_settings->search_notes = TRUE;
    calendar_search_settings->search_birthdays = TRUE;
    calendar_search_settings->search_type = SEARCH_ALL;
    calendar_search_settings->start_time = 0;
    calendar_search_settings->end_time = 0;
  }
}
