/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/


#include <cstdlib>
#include <iostream>

#include <Lum/Base/String.h>

#include <Lum/OS/Main.h>

#include "Heap.h"
#include "PlayerGoal.h"
#include "PlayerGreedy.h"
#include "PlayerJudge.h"
#include "PlayerNeuron.h"
#include "PlayerRandom.h"
#include "Result.h"

//#define DEBUG_OUTPUT 1
#define GAMES 50000
//#define GAMES 10000
//#define GAMES 5000
//#define GAMES 1000
//#define GAMES 1

class Main : public Lum::OS::MainIllumination
{
public:
  bool Body()
  {
    std::cout << "Starting..." << std::endl;

    srand(time(NULL));

    Heap                 *heap=new Heap();
    Table                *table=new Table();
    std::vector<Player*> players;
    size_t               globalPoints[2];
    size_t               globalGames[2];

    players.push_back(new NeuronPlayer());
    players.push_back(new JudgePlayer());

    globalPoints[0]=0;
    globalPoints[1]=0;

    globalGames[0]=0;
    globalGames[1]=0;

    assert(players.size()==2 || players.size()==4);

    for (size_t game=1; game<=GAMES; game++) {
      std::vector<Card>   initialCards;
      size_t              lastExchange=0;
      std::vector<Result> result;

      if (globalPoints[0]==0 && globalPoints[1]==0) {
        std::cout << "Game " << game << std::endl;
      }
      else {
        std::cout << "Game " << game << " " << (globalPoints[0]*100)/(globalPoints[0]+globalPoints[1]) << "% " << (globalPoints[1]*100)/(globalPoints[0]+globalPoints[1]) << "%" << " " << globalGames[0] << " " << globalGames[1] << std::endl;
      }

      result.resize(2);
      heap->InitializeAndMix();

      for (size_t player=0; player<players.size(); player++) {
        players[player]->Initialize();
      }

      for (size_t player=0; player<players.size(); player++) {
        for (size_t c=1; c<=3; c++) {
          players[player]->PutCard(heap->GetCard());
        }
      }

      for (size_t h=1; h<=4; h++) {
        initialCards.push_back(heap->GetCard());
      }

#if defined(DEBUG_OUTPUT)
      for (size_t i=0; i<initialCards.size(); i++) {
        std::cout << "Placing " << Lum::Base::WStringToString(initialCards[i].GetName()) << " on table." << std::endl;
      }
#endif

      table->Initialize(initialCards);

      size_t round=1;
      size_t finished=false;

      while (!finished) {
#if defined(DEBUG_OUTPUT)
        std::cout << "--- Round " << round << std::endl;
#endif
        for (size_t player=0; player<players.size(); player++) {

#if defined(DEBUG_OUTPUT)
          std::cout << "Cards in hand of player " << player+1 << ":"<< std::endl;
          for (size_t i=0; i<players[player]->hand.size(); i++) {
            std::cout << "* "<< Lum::Base::WStringToString(players[player]->hand[i].GetName()) << std::endl;
          }
          std::cout << "Cards on the table:" << std::endl;
          for (size_t i=0; i<table->table.size(); i++) {
            std::cout << "* "<< Lum::Base::WStringToString(table->table[i].GetName()) << std::endl;
          }
#endif
          std::vector<Card> collected;
          SolutionSet       solution=players[player]->PlayCard(*table,round);

          players[player]->hand.erase(players[player]->hand.begin()+solution.yoursPos);

#if defined(DEBUG_OUTPUT)
          std::cout << "Player " << player+1 << " plays " << Lum::Base::WStringToString(solution.yours.GetName()) << std::endl;
#endif

          collected=table->PutCard(solution.yours,solution.solution);

          if (collected.size()>0) {
#if defined(DEBUG_OUTPUT)
            std::cout << "..and collected the following cards: " << std::endl;
            for (size_t card=0; card<collected.size(); card++) {
              std::cout << "* "<< Lum::Base::WStringToString(collected[card].GetName()) << std::endl;
            }
#endif

#if defined(DEBUG_OUTPUT)
            if (table->IsEmpty()) {
              std::cout << "=>>> SCOPA!" << std::endl;
            }
#endif
            result[player%2].AddToCollected(collected,table->IsEmpty());
            lastExchange=player;
          }

          for (size_t i=0; i<players.size(); i++) {
            players[i]->CardsPlayed(collected,
                                    (i%2)==(player%2),
                                    round);
          }
        }

        // Check if finished

        if (heap->IsEmpty()) {
          finished=true;

          for (size_t player=0; player<players.size(); player++) {
            if (players[player]->HasCardsInHand()) {
              finished=false;
              break;
            }
          }
        }

        if (!finished) {
          for (size_t i=0; i<players.size(); i++) {
            players[i]->RoundFinished(round,
                                      result[i%2].GetTemporaryPoints(),
                                      result[(i+1)%2].GetTemporaryPoints());
          }
        }

        // New cards

        if (round>1 && round%3==0 && !heap->IsEmpty()) {
          for (size_t player=0; player<players.size(); player++) {
            for (size_t c=1; c<=3; c++) {
              players[player]->PutCard(heap->GetCard());
            }
          }
        }

        round++;
      }

      if (!table->IsEmpty()) {
#if defined(DEBUG_OUTPUT)
        std::cout << "--- Final" << std::endl;
        std::cout << "Player " << lastExchange+1 << " gets the rest of the cards on the table:" << std::endl;

        for (size_t i=0; i<table->table.size(); i++) {
          std::cout << "* "<< Lum::Base::WStringToString(table->table[i].GetName()) << std::endl;
        }
#endif
        result[lastExchange%2].AddToCollected(table->table,false);
      }

      // Now calculate final points

      size_t points[2];

      // Scopas
      points[0]=result[0].GetScopas();
      points[1]=result[1].GetScopas();

      // Who has more cards?
      if (result[0].GetCards()>result[1].GetCards()) {
        points[0]++;
      }
      else if (result[1].GetCards()>result[0].GetCards()) {
        points[1]++;
      }

      // Who has more denares?
      if (result[0].GetDenares()>result[1].GetDenares()) {
        points[0]++;
      }
      else if (result[1].GetDenares()>result[0].GetDenares()) {
        points[1]++;
      }

      // Who has denare 7?
      if (result[0].HasDenare7()) {
        points[0]++;
      }
      else {
        points[1]++;
      }

      // Primeria
      size_t pp[2];

      pp[0]=0;
      pp[1]=0;
      for (size_t c=Card::spade; c<=Card::denare; c++) {
        pp[0]+=result[0].GetHighestPointsOfColor((Card::Color)c);
        pp[1]+=result[1].GetHighestPointsOfColor((Card::Color)c);
      }

#if defined(DEBUG_OUTPUT)
      std::cout << std::endl;
      std::cout << "Now calculating points:..." << std::endl;

      std::cout << "Scopas:   " << result[0].GetScopas() << " " << result[1].GetScopas() << std::endl;
      std::cout << "Cards:    " << result[0].GetCards() << " " << result[1].GetCards() << std::endl;
      std::cout << "Denares:  " << result[0].GetDenares() << " " << result[1].GetDenares() << std::endl;
      std::cout << "Denare7:  " << result[0].HasDenare7() << " " << result[1].HasDenare7() << std::endl;
      std::cout << "Primiera:  " << pp[0] << " " << pp[1] << std::endl;
#endif

      if (pp[0]>pp[1]) {
        points[0]++;
      }
      else if (pp[1]>pp[0]) {
        points[1]++;
      }

#if defined(DEBUG_OUTPUT)
      std::cout << std::endl;
      for (size_t i=0; i<=1; i++) {
        std::cout << "Group " << i+1 << " has " << points[i] << " points." << std::endl;
      }
#endif

      for (size_t i=0; i<players.size(); i++) {
        players[i]->RoundFinished(18,
                                  points[i%2],
                                  points[(i+1)%2]);
      }

      if (game%2==1) {
        globalPoints[0]+=points[0];
        globalPoints[1]+=points[1];

        if (points[0]>points[1]) {
          globalGames[0]++;
        }
        else if (points[1]>points[0]) {
          globalGames[1]++;
        }
      }
      else {
        globalPoints[1]+=points[0];
        globalPoints[0]+=points[1];

        if (points[0]>points[1]) {
          globalGames[1]++;
        }
        else if (points[1]>points[0]) {
          globalGames[0]++;
        }
      }

      Player *p=players[0];
      for (size_t i=0; i<players.size()-1; i++) {
        players[i]=players[i+1];
      }
      players[players.size()-1]=p;
    }

    std::cout << globalPoints[0] << " " << globalPoints[1] << std::endl;

    std::cout << (globalPoints[0]*100)/(globalPoints[0]+globalPoints[1]) << "% " << (globalPoints[1]*100)/(globalPoints[0]+globalPoints[1]) << "%" << std::endl;

    delete heap;

    for (size_t i=0; i<players.size()-1; i++) {
      players[i]->PrintStatistics();
    }

    std::cout << "done." << std::endl;

    return 0;
  }
};

LUM_MAIN(Main,L"ScopaTest")
