/*
  RulerJinni - A simple virtual ruler
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <cmath>
#include <limits>

#include <Lum/Base/String.h>

#include <iostream>

#include "Configuration.h"

GameArea::GameArea()
 : font(Lum::OS::display->GetFont()),
   rulerOffset(0),
   scaleOffset(0),
   dragActive(false),
   rulerHeight(font->height+2*font->height/3),
   backgroundColor(1,1,1,Lum::OS::Display::whiteColor),
   primaryColor(0,0,0,Lum::OS::Display::blackColor),
   secondaryColor(0.7,0.7,0.7,Lum::OS::Display::blackColor)
 {
  SetBackground(new Lum::OS::PlainFill(backgroundColor));
  SetCanFocus(true);
  RequestFocus();

   Observe(factor);
   Observe(unit);
}

GameArea::~GameArea()
{
}

/*
bool GameArea::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<GameArea*>(model);

  Lum::Control::SetModel(this->model);

  return this->model.Valid();
}*/

void GameArea::CalcSize()
{
  minWidth=100;
  minHeight=100;

  width=minWidth;
  height=minHeight;

  Lum::Control::CalcSize();
}

void GameArea::Layout()
{
  rulerOffset=std::max(rulerOffset,0);
  rulerOffset=std::min(rulerOffset,(int)(height-rulerHeight));

  if (!factor->IsNull()) {
    scaleOffset=std::max(scaleOffset,-(int)(width/(10*factor->Get())));
    scaleOffset=std::min(scaleOffset,(int)(width/(10*factor->Get())));
  }

  Lum::Control::Layout();
}

void GameArea::Draw(int x, int y, size_t w, size_t h)
{
  Lum::Control::Draw(x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  if (factor->IsNull()) {
    return;
  }

  double factor=::factor->Get();

  draw->PushForeground(primaryColor);

  draw->DrawLine(this->x,this->y,this->x,this->y+height-1);
  draw->DrawLine(this->x,this->y,this->x+width-1,this->y);

  // Draw unit lines (x.0)
  size_t c=0;
  int    pos=this->x+lround(c*factor);
  while (pos<this->x+(int)width) {
    draw->DrawLine(pos,this->y,pos,this->y+height-1);
    c+=10;

    pos=this->x+lround(c*factor);
  }

  if (!dragActive) {
    draw->PushForeground(secondaryColor);
    // Draw half unit lines (x.5)
    if (round(15*factor)-
        round(5*factor)>=3) {
      draw->PushDash("\003\003",2,Lum::OS::DrawInfo::fDash);
      c=5;
      pos=this->x+lround(c*factor);
      while (pos<this->x+(int)width) {
        if (c % 10!=0) {
          draw->DrawLine(pos,this->y,pos,this->y+height-1);
        }

        c+=10; // We walk only through the uneven ones to avoid
               // Colliding with 10
        pos=this->x+lround(c*factor);
      }
      draw->PopDash();
    }

    // Draw tenth of unit lines (x.1)
    if (round(2*factor)-
        round(1*factor)>=3) {
      draw->PushDash("\002\002",2,Lum::OS::DrawInfo::fDash);
      c=1;
      pos=this->x+lround(c*factor);
      while (pos<this->x+(int)width) {
        if (c % 10!=0 && c % 5!=0) {
          draw->DrawLine(pos,this->y,pos,this->y+height-1);
        }

        c++;
        pos=this->x+lround(c*factor);
      }
      draw->PopDash();
    }
    draw->PopForeground();
  }

  draw->PopForeground();


  draw->PushForeground(Lum::OS::Display::whiteColor);
  draw->FillRectangle(this->x,
                      this->y+rulerOffset,
                      this->width,
                      font->height+2*font->height/3);
  draw->PopForeground();

  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->DrawRectangle(this->x,
                      this->y+rulerOffset,
                      this->width,
                      font->height+2*font->height/3);
  draw->PopForeground();

  // Draw unit labels (x.0)
  draw->PushForeground(Lum::OS::Display::blackColor);
  draw->PushFont(font);
  draw->PushClip(x,y,w,h);

  int lastPos=std::numeric_limits<int>::min();
  c=0;
  pos=this->x+lround(c*factor);
  while (pos<this->x+(int)width) {
    Lum::OS::FontExtent extent;
    std::wstring        text=Lum::Base::NumberToWString((int)c/10+scaleOffset);

    font->StringExtent(text,extent);

    if (pos-(int)extent.width/2>lastPos) {
      draw->DrawString(pos-extent.width/2,
                       this->y+rulerOffset+(rulerHeight-font->height)/2+font->ascent,
                       text);

      lastPos=pos-extent.width/2+extent.width;
    }

    c+=10;
    pos=this->x+lround(c*factor);
  }

  draw->PopClip();
  draw->PopFont();
  draw->PopForeground();

}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  /* It makes no sense to get the focus if we are currently not visible */
  if (!visible) {
    return false;
  }

  /*
    Pressing left mouse button within the bounds of the ruler initiates drag
  */

  if (event.type==Lum::OS::MouseEvent::down &&
      PointIsIn(event) &&
      event.button==Lum::OS::MouseEvent::button1 &&
      event.y>=this->y+rulerOffset &&
      event.y<this->y+rulerOffset+(int)rulerHeight) {
    dragActive=true;
    dragStartX=event.x;
    dragStartY=event.y;
    dragStartRulerOffset=rulerOffset;
    dragStartScaleOffset=scaleOffset;
    return true;
  }
  else if (event.IsGrabEnd() ||
           (event.type==Lum::OS::MouseEvent::move && event.IsGrabed())) {
    double factor=::factor->Get();
    int    oldRulerOffset=rulerOffset;
    int    oldScaleOffset=scaleOffset;

    rulerOffset=dragStartRulerOffset+(event.y-dragStartY);
    rulerOffset=std::max(rulerOffset,0);
    rulerOffset=std::min(rulerOffset,(int)(height-rulerHeight));

    scaleOffset=dragStartScaleOffset+(dragStartX-event.x)/(10*factor);
    scaleOffset=std::max(scaleOffset,-(int)(width/(10*factor)));
    scaleOffset=std::min(scaleOffset,(int)(width/(10*factor)));

    if (event.IsGrabEnd()) {
      dragActive=false;
    }

    if (oldRulerOffset!=rulerOffset || oldScaleOffset!=scaleOffset || event.IsGrabEnd()) {
      Redraw();
    }

    return true;
  }

  return false;
}

bool GameArea::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{
  return false;
}

void GameArea::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==factor || model==unit) {
    Redraw();
  }

  Control::Resync(model,msg);
}

