/*************************************************************************}
{ equalizer.cpp - gstreamer equalizer wrapper                             }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 21 May 2012                                              }
{*************************************************************************/

#include "equalizer.h"

QString Equalizer::moduleName = EQ_COMMON_NAME;
int Equalizer::bandsCount = EQ_COMMON_BANDS_COUNT;
int Equalizer::bandFrequencies[EQ_COMMON_BANDS_COUNT] =
        {29, 59, 119, 237, 474, 947, 1889, 3770, 7523, 15011};
QString Equalizer::bandNames[EQ_COMMON_BANDS_COUNT] =
        {"band0", "band1", "band2", "band3", "band4", "band5", "band6", "band7", "band8", "band9"};

Equalizer::Equalizer(QObject *parent) : QObject(parent)
{
    effect = createEffect();
    if(effect)
        init();
}

Equalizer::~Equalizer()
{
    if(effect)
        delete effect;
}

void Equalizer::setBandLevel(int bandIndex, double newLevel)
{
    effect->setParameterValue(bandFxParams[bandIndex], newLevel);
    double level = getRealBandLevel(bandIndex);
    bandLevels[bandIndex] = level;
    emit onBandLevelChange(bandIndex, level);
}

void Equalizer::setDefaultBandLevel(int bandIndex)
{
    return setBandLevel(bandIndex, bandDefaultLevels[bandIndex]);
}

void Equalizer::setDefaultLevels()
{
    for(int a=0; a<bandsCount; a++)
        setDefaultBandLevel(a);
}

double Equalizer::getRealBandLevel(int bandIndex)
{
    return effect->parameterValue(bandFxParams[bandIndex]).toDouble();
}

double Equalizer::updateLevel(int bandIndex)
{
    double level = getRealBandLevel(bandIndex);
    bandLevels[bandIndex] = level;
    emit onBandLevelChange(bandIndex, level);
    return level;
}

void Equalizer::updateLevels()
{
    for(int a=0; a<bandsCount; a++)
        updateLevel(a);
}

void Equalizer::copyLevels(Equalizer *eq)
{
    int n = qMin(eq->getBandsCount(), bandsCount);
    for(int a=0; a<n; a++)
        setBandLevel(a, eq->getBandLevel(a));
}

void Equalizer::saveToPreset(const QString &presetName)
{
    saveArrayToPreset(&bandLevels[0], presetName);
}

bool Equalizer::loadFromPreset(const QString &presetName)
{
    double levels[bandsCount];
    if(!loadArrayFromPreset(&levels[0], presetName))
        return false;
    for(int a=0; a<bandsCount; a++)
        setBandLevel(a, levels[a]);
    return true;
}

bool Equalizer::renamePreset(const QString &oldPresetName, const QString &newPresetName)
{
    if(!copyPreset(oldPresetName, newPresetName))
        return false;
    deletePreset(oldPresetName);
    return true;
}

bool Equalizer::copyPreset(const QString &srcPresetName, const QString &destPresetName)
{
    if(srcPresetName == destPresetName)
        return false;
    double levels[bandsCount];
    if(!loadArrayFromPreset(&levels[0], srcPresetName))
        return false;
    saveArrayToPreset(&levels[0], destPresetName);
    return true;
}

void Equalizer::deletePreset(const QString &presetName)
{
    ini->remove(getPresetIniKey(presetName));
}

QStringList Equalizer::getPresetsList()
{
    ini->beginGroup(EQ_INI_PRESETS_GROUP);
    QStringList presets = ini->childKeys();
    ini->endGroup();
    return presets;
}

QStringList Equalizer::getValidPresetsList()
{
    QStringList validPresets;
    QStringList presets = getPresetsList();
    double levels[bandsCount];
    foreach(QString presetName, presets)
    {
        if(loadArrayFromPreset(&levels[0], presetName))
            validPresets.append(presetName);
    }
    return validPresets;
}

bool Equalizer::isPresetExists(const QString &presetName)
{
    ini->beginGroup(EQ_INI_PRESETS_GROUP);
    bool exists = (ini->childKeys().indexOf(presetName) >= 0);
    ini->endGroup();
    return exists;
}

bool Equalizer::hasValidPreset(const QString &presetName)
{
    if(!isPresetExists(presetName))
        return false;
    double levels[bandsCount];
    return loadArrayFromPreset(&levels[0], presetName);
}

Phonon::Effect *Equalizer::createEffect()
{
    foreach(Phonon::EffectDescription desc, Phonon::BackendCapabilities::availableAudioEffects())
    {
        if(desc.name() == EQ_COMMON_NAME)
        {
            if(!desc.isValid())
                return NULL;

            return new Phonon::Effect(desc, this);
        }
    }
    return NULL;
}

void Equalizer::init()
{
    Phonon::EffectParameter param;
    for(int a=0; a<bandsCount; a++)
    {
        param = getParamByName(bandNames[a]);
        bandUpperBounds[a] = param.maximumValue().toDouble();
        bandLowerBounds[a] = param.minimumValue().toDouble();
        bandDefaultLevels[a] = param.defaultValue().toDouble();
        bandFxParams[a] = param;
    }
    updateLevels();
}

Phonon::EffectParameter Equalizer::getParamByName(const QString &paramName)
{
    foreach(Phonon::EffectParameter param, effect->parameters())
        if(param.name() == paramName)
            return param;
    return Phonon::EffectParameter();
}

QString Equalizer::getPresetIniKey(const QString &presetName)
{
    return QString(EQ_INI_PRESETS_GROUP)+"/"+presetName;
}

void Equalizer::saveArrayToPreset(const double *levels, const QString &presetName)
{
    QStringList lines;
    QString num;
    QString line;
    for(int a=0; a<bandsCount; a++)
    {
        num.setNum(levels[a], 'f');
        lines.append(num);
    }
    line = lines.join(";");
    ini->setValue(getPresetIniKey(presetName), line);
}

bool Equalizer::loadArrayFromPreset(double *levels, const QString &presetName)
{
    QString line = ini->value(getPresetIniKey(presetName), "").toString();
    if(line.isEmpty())
        return false;
    QStringList lines = line.split(";", QString::SkipEmptyParts);
    if(lines.size() != bandsCount)
        return false;

    double level;
    bool ok;
    for(int a=0; a<bandsCount; a++)
    {
        level = lines.at(a).trimmed().toDouble(&ok);
        if(!ok)
            return false;
        levels[a] = level;
    }
    return true;
}
