/*
 * Copyright (C) 2010 Ruslan Nigmatullin <euroelessar@ya.ru>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _QUASAR_VERSION_H_
#define _QUASAR_VERSION_H_

#include "quasar_export.h"

/// @brief Quasar version as string at compile time.
#define LIBQUASAR_VERSION_STRING "${CMAKE_LIBQUASAR_VERSION_STRING}"

/// @brief The major Quasar version number at compile time
#define LIBQUASAR_VERSION_MAJOR ${CMAKE_LIBQUASAR_VERSION_MAJOR}

/// @brief The minor Quasar version number at compile time
#define LIBQUASAR_VERSION_MINOR ${CMAKE_LIBQUASAR_VERSION_MINOR}

/// @brief The Quasar release version number at compile time
#define LIBQUASAR_VERSION_RELEASE ${CMAKE_LIBQUASAR_VERSION_PATCH}

/**
 * \brief Create a unique number from the major, minor and release number of a %Quasar version
 *
 * This function can be used for preprocessing. For version information at runtime
 * use the version methods in the Quasar namespace.
 */
#define LIBQUASAR_MAKE_VERSION( a,b,c ) (((a) << 16) | ((b) << 8) | (c))

/**
 * \brief %Quasar Version as a unique number at compile time
 *
 * This macro calculates the %Quasar version into a number. It is mainly used
 * through LIBQUASAR_IS_VERSION in preprocessing. For version information at runtime
 * use the version methods in the Quasar namespace.
 */
#define LIBQUASAR_VERSION \
    LIBQUASAR_MAKE_VERSION(LIBQUASAR_VERSION_MAJOR,LIBQUASAR_VERSION_MINOR,LIBQUASAR_VERSION_RELEASE)

/**
 * \brief Check if the %Quasar version matches a certain version or is higher
 *
 * This macro is typically used to compile conditionally a part of code:
 * \code
 * #if LIBQUASAR_IS_VERSION(0,2)
 * // Code for Quasar 0.2
 * #else
 * // Code for Quasar 0.1
 * #endif
 * \endcode
 *
 * For version information at runtime
 * use the version methods in the Quasar namespace.
 */
#define LIBQUASAR_IS_VERSION(a,b,c) ( LIBQUASAR_VERSION >= LIBQUASAR_MAKE_VERSION(a,b,c) )


namespace Quasar {
    /**
     * @brief Returns the major number of Quasar's version, e.g.
     * 1 for %Quasar 1.0.2.
     * @return the major version number at runtime.
     */
    QUASAR_EXPORT unsigned int versionMajor();

    /**
     * @brief Returns the minor number of Quasar's version, e.g.
     * 0 for %Quasar 1.0.2.
     * @return the minor version number at runtime.
     */
    QUASAR_EXPORT unsigned int versionMinor();

    /**
     * @brief Returns the release of Quasar's version, e.g.
     * 2 for %Quasar 1.0.2.
     * @return the release number at runtime.
     */
    QUASAR_EXPORT unsigned int versionRelease();

    /**
     * @brief Returns the %Quasar version as string, e.g. "1.0.2".
     *
     * On contrary to the macro LIBQUASAR_VERSION_STRING this function returns
     * the version number of Quasar at runtime.
     * @return the %Quasar version. You can keep the string forever
     */
    QUASAR_EXPORT const char* versionString();
}

#endif
