/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-orientation: Utility for QBW Orientation Management and Operations
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <hildon/hildon.h>
#include <hildon/hildon-gtk.h>
#include <libosso.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-orientation.h"

#ifdef DOCUMENTATION_REFERENCE_HERE
//#define MCE_ORIENTATION_UNKNOWN   "unknown"   Device rotation name for unknown
//#define MCE_ORIENTATION_PORTRAIT  "portrait"  Device rotation name for portrait orientation
//#define MCE_ORIENTATION_LANDSCAPE "landscape" Device rotation name for landscape orientation
//#define MCE_ORIENTATION_ON_STAND  "on_stand"  Device rotation name for on stand
//#define MCE_ORIENTATION_OFF_STAND "off_stand" Device rotation name for off stand */
//#define MCE_ORIENTATION_FACE_UP   "face_up"   Device rotation name for facing up */
//#define MCE_ORIENTATION_FACE_DOWN "face_down" Device rotation name for facing down */
#endif

#define MCE_MATCH_RULE "type='signal',interface='" MCE_SIGNAL_IF "',member='" MCE_DEVICE_ORIENTATION_SIG "'"

extern gchar *qbwExecReason[];

gchar *qbwOrientationAction[] = {
	"QBW_ENABLE_ORIENTATION_MONITOR",
	"QBW_READ_ORIENTATION_XYZ",
	"QBW_DISABLE_ORIENTATION_MONITOR"
};

/*******************************************************************************/
void
set_portrait(GtkWidget *self, char const *prop, guint32 value){
  qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s prop,val=[%s,%d]",self, G_STRFUNC, prop, value);
  gdk_property_change(gtk_widget_get_toplevel(self)->window,
                      gdk_atom_intern_static_string (prop),
                      gdk_x11_xatom_to_atom(XA_CARDINAL), 32,
                      GDK_PROP_MODE_REPLACE, (gpointer)&value, 1);
}
/*******************************************************************************/
void
init_portrait(GtkWidget *win){
  qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s",win, G_STRFUNC);
  gtk_widget_realize(win);
  set_portrait(win, "_HILDON_PORTRAIT_MODE_SUPPORT", 1);
  set_portrait(win, "_HILDON_PORTRAIT_MODE_REQUEST", 1);
}
/*******************************************************************************/
void
init_landscape(GtkWidget *win){
  qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s",win, G_STRFUNC);
  gtk_widget_realize(win);
  set_portrait(win, "_HILDON_PORTRAIT_MODE_SUPPORT", 1);
  set_portrait(win, "_HILDON_PORTRAIT_MODE_REQUEST", 0);
}
/*******************************************************************************/

DBusHandlerResult queen_beecon_orientation_filter_function(DBusConnection *connection, DBusMessage *message, QueenBeecon *self)
{
	QueenBeeconClass *klass = QUEEN_BEECON_GET_CLASS (self);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s dc=%p msg=%p updOnOrientation=%d",self, G_STRFUNC, connection, message, self->priv->updOnOrientation);
	if (!self->priv->updOnOrientation) return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    DBusError error;
    dbus_error_init(&error);

    if (dbus_message_is_signal(message, MCE_SIGNAL_IF, MCE_DEVICE_ORIENTATION_SIG)) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) MCE_DEVICE_ORIENTATION_SIG", self);
		const gchar *orientationMode=NULL;
		const gchar *orientationStand=NULL;
		const gchar *orientationFace=NULL;
		if (klass->orientationMode!=NULL) {g_free(klass->orientationMode); klass->orientationMode=NULL;}
		if (klass->orientationStand!=NULL) {g_free(klass->orientationStand); klass->orientationStand=NULL;}
		if (klass->orientationFace!=NULL) {g_free(klass->orientationFace); klass->orientationFace=NULL;}
	    if (dbus_message_get_args(message, &error,	DBUS_TYPE_STRING, &orientationMode,
													DBUS_TYPE_STRING, &orientationStand,
													DBUS_TYPE_STRING, &orientationFace,
													DBUS_TYPE_INT32, &klass->orientationX,
													DBUS_TYPE_INT32, &klass->orientationY,
													DBUS_TYPE_INT32, &klass->orientationZ,
													DBUS_TYPE_INVALID)) {
			klass->orientationMode = g_strdup(orientationMode);
			klass->orientationStand = g_strdup(orientationStand);
			klass->orientationFace = g_strdup(orientationFace);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) ORIENTATION Mode[%s] Stand[%s] Face[%s] x,y,z(%d,%d,%d)", self, orientationMode, orientationStand, orientationFace, klass->orientationX, klass->orientationY, klass->orientationZ);

#ifdef NOT_WORKING
			if (!g_strcmp0(klass->orientationMode, "landscape"))
               init_landscape(GTK_WIDGET(self));
            else if (!g_strcmp0(klass->orientationMode, "portrait"))
               init_portrait(GTK_WIDGET(self));
#endif //NOT_WORKING
			queen_beecon_update_content (self, qbwExecReason[QBW_ORIENTATION_MONITOR]);
	    } else {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) DBUS dbus_message_get_args Error [%s]",self, error.message);
			dbus_error_free (&error);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static const char *accel_filename = "/sys/class/i2c-adapter/i2c-3/3-001d/coord";
//static const char *accel_filename = "/tmp/xxx";

void liqaccel_read(int *ax,int *ay,int *az)
{
	FILE *fd;
	int rs;
	fd = fopen(accel_filename, "r");
	if (fd == NULL) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Can't read Accelerometer device!", NULL);
		*ax=0,*ay=0,*az=0;
		return;
	}
	rs=fscanf((FILE*) fd,"%i %i %i",ax,ay,az);
	fclose(fd);
	if (rs != 3) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Can't read Accelerometer information from device!", NULL);
		*ax=0,*ay=0,*az=0;
		return;
	}
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Orientation x,y,z(%d,%d,%d)", NULL, *ax, *ay, *az);
}

gboolean queen_beecon_orientation_manager(QueenBeecon *self, QbwOrientationAction action)
{
	QueenBeeconClass *klass = QUEEN_BEECON_GET_CLASS (self);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s action=[%d,%s] orientationONOFF=[%d]",self, G_STRFUNC, action, qbwOrientationAction[action], self->priv->orientationONOFF);
	DBusConnection *sys_conn;
    DBusMessage *message, *reply;
    DBusError error;
    dbus_error_init(&error);

    switch (action) {
	case QBW_ENABLE_ORIENTATION_MONITOR:
		if (self->priv->orientationONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_ENABLE_ORIENTATION_MONITOR", self);
		sys_conn = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	    if (dbus_error_is_set (&error)){
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) DBUS dbus_bus_get Error [%s]",self, error.message);
	        dbus_error_free (&error);
	        return FALSE; //FALSE on error
	    }
	    dbus_message_unref(message);
	    /* Add the callback, which should be called, once the device is rotated */
	    dbus_bus_add_match(sys_conn, MCE_MATCH_RULE, NULL);
	    dbus_connection_add_filter(sys_conn, (DBusHandleMessageFunction)queen_beecon_orientation_filter_function, self, NULL);
	    /* Get the current orientation */
	    message = dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ);
	    reply = dbus_connection_send_with_reply_and_block(sys_conn, message,-1, &error);
	    if (dbus_error_is_set (&error)){
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) DBUS dbus_connection_send_with_reply_and_block Error [%s]",self, error.message);
	        dbus_error_free (&error);
		    dbus_connection_unref (sys_conn);
	        return FALSE; //FALSE on error
	    }
	    dbus_message_unref(message);
		const gchar *orientationMode=NULL;
		const gchar *orientationStand=NULL;
		const gchar *orientationFace=NULL;
	    if (dbus_message_get_args(reply, &error,DBUS_TYPE_STRING, &orientationMode,
												DBUS_TYPE_STRING, &orientationStand,
												DBUS_TYPE_STRING, &orientationFace,
												DBUS_TYPE_INT32, &klass->orientationX,
												DBUS_TYPE_INT32, &klass->orientationY,
												DBUS_TYPE_INT32, &klass->orientationZ,
												DBUS_TYPE_INVALID)) {
			if (klass->orientationMode!=NULL) {g_free(klass->orientationMode); klass->orientationMode=NULL;}
			if (klass->orientationStand!=NULL) {g_free(klass->orientationStand); klass->orientationStand=NULL;}
			if (klass->orientationFace!=NULL) {g_free(klass->orientationFace); klass->orientationFace=NULL;}
			klass->orientationMode = g_strdup(orientationMode);
			klass->orientationStand = g_strdup(orientationStand);
			klass->orientationFace = g_strdup(orientationFace);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) ORIENTATION Mode[%s] Stand[%s] Face[%s] x,y,z(%d,%d,%d)", self, orientationMode, orientationStand, orientationFace, klass->orientationX, klass->orientationY, klass->orientationZ);
	    } else {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) DBUS dbus_message_get_args Error [%s]",self, error.message);
			dbus_error_free (&error);
		    dbus_message_unref(reply);
			dbus_connection_unref (sys_conn);
			return FALSE; //FALSE on error
		}
	    dbus_message_unref(reply);
	    dbus_connection_unref (sys_conn);
	    //hildon_gtk_window_set_portrait_flags (GTK_WINDOW (self), HILDON_PORTRAIT_MODE_SUPPORT);
		self->priv->orientationONOFF=TRUE;
		break;
	case QBW_READ_ORIENTATION_XYZ:
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_READ_ORIENTATION_XYZ", self);
		liqaccel_read(&klass->orientationX, &klass->orientationY, &klass->orientationZ);
		break;
	case QBW_DISABLE_ORIENTATION_MONITOR:
		if (!self->priv->orientationONOFF) break;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW_DISABLE_ORIENTATION_MONITOR", self);
		sys_conn = dbus_bus_get (DBUS_BUS_SYSTEM, &self->priv->dbus_mon_error);
	    if (dbus_error_is_set (&error)){
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) DBUS dbus_bus_get Error [%s]",self, error.message);
	        dbus_error_free (&error);
	        return FALSE; //FALSE on error
	    }
	    dbus_bus_remove_match(sys_conn, MCE_MATCH_RULE, NULL);
	    dbus_connection_remove_filter(sys_conn, (DBusHandleMessageFunction)queen_beecon_orientation_filter_function, self);
	    message = dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ);
	    dbus_message_set_no_reply(message, TRUE);
	    dbus_connection_send(sys_conn, message, NULL);
		dbus_message_unref(message);
	    dbus_connection_unref (sys_conn);
	    //hildon_gtk_window_set_portrait_flags (GTK_WINDOW (self), ~HILDON_PORTRAIT_MODE_SUPPORT);
	    self->priv->orientationONOFF=FALSE;
		break;
	default:
		return FALSE; // Unhandled command
	}
	return TRUE;
}
