#include <QApplication>
#include <QDesktopWidget>
#include <QGridLayout>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QMessageBox>
#include <QPair>
#include <QList>
#include <QSettings>
#include <QPushButton>
#include <QIcon>
#include <QtNetwork/QTcpServer>
#include <QtNetwork/QTcpSocket>
#include <QtNetwork/QHostAddress>
#include <QtNetwork/QNetworkInterface>

#include "centralwidget.h"
#include "fieldview.h"
#include "infopanel.h"
#include "modepanel.h"
#include "scorepanel.h"
#include "serverdialog.h"
#include "clientdialog.h"
#include "protocol.h"
#include "utils.h"
#include "reversidefs.h"
#include "optionsdialog.h"

#include "pixmaps/undo.xpm"

CentralWidget::CentralWidget(QWidget *parent, ReversiGame &reversiGame):
    QWidget(parent),
    gameServer_(0),
    gameClient_(0),
    gameConnection_(0),
    isNetworkGame_(false),
    reversiGame_(reversiGame)
{
    //readSettings();

    layout_ = new QGridLayout(this);
    fieldView_ = new FieldView(reversiGame_, this);
    fieldView_->setAnimationSpeed(settings_.animationSpeed());
    layout_->addWidget(fieldView_, 0, 0);

    infoLayout_ = new QVBoxLayout();
    infoLayout_->addStretch(1);
    QHBoxLayout *hb = new QHBoxLayout;
    infoPanel_ = new InfoPanel(reversiGame_, this);
    hb->addStretch(1);
    hb->addWidget(infoPanel_);
    hb->addStretch(1);
    btUndo_ = new QPushButton(QIcon(undo), "", this);
    hb->addWidget(btUndo_);
    infoLayout_->addItem(hb);

    modePanel_ = new ModePanel(this, settings_.mode(), settings_.AILevel());
    infoLayout_->addWidget(modePanel_);

    scorePanel_ = new ScorePanel(this);
    infoLayout_->addWidget(scorePanel_);
    infoLayout_->addStretch(1);

    layout_->addItem(infoLayout_, 0, 1);

    showInfoPanels();

    connect(&reversiGame_, SIGNAL(stepUnavailable()), SLOT(stepUnavailable()));
    connect(&reversiGame_, SIGNAL(gameOver()), SLOT(slotGameOver()));
    connect(&reversiGame_, SIGNAL(enableUndo(bool)), SLOT(slotEnableUndo(bool)));

    connect(btUndo_, SIGNAL(clicked()), &reversiGame_, SLOT(slotUndo()));

    //network stuff
    gameServer_ = new QTcpServer(this);
    gameClient_ = new QTcpSocket(this);
    connect(gameClient_, SIGNAL(readyRead()), this, SLOT(slotReadNetworkData()));
    connect(gameClient_, SIGNAL(error(QAbstractSocket::SocketError)), this, SLOT(displayP2PError(QAbstractSocket::SocketError)));
}//ctor

CentralWidget::~CentralWidget()
{
    //saveSettings();
    if (settings_.saveMode())
    {
        QPair<int, int> mode = modePanel_->mode();
        settings_.setMode(mode.first);
        settings_.setAILevel(mode.second);
    }
}

void CentralWidget::slotOrientationChanged()
{
    showInfoPanels();
}

void CentralWidget::showInfoPanels()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height())
    {
        layout_->removeItem(infoLayout_);
        layout_->addItem(infoLayout_, 0, 1);
    }
    else
    {
        layout_->removeItem(infoLayout_);
        layout_->addItem(infoLayout_, 1, 0);
    }
}//showInfoPanels

void
CentralWidget::newGame()
{
    static int ai_levels[] = {2,5,6,MAX_AI_LEVEL};
    QPair<int, int> mode = modePanel_->mode();
    gameMode_ = mode.first;
    strength_ = ai_levels[mode.second];
    if (mode.first == 3)
    {
        if (!newGameP2PServer())
            return;
    }
    else if (mode.first == 4)
    {
        if (!newGameP2PClient())
            return;
    }
    isNetworkGame_ = mode.first > 2;
    reversiGame_.init(gameMode_, strength_);

    if (mode.first == 1 || mode.first == 2)
    {
        unsigned w, l, d;
        score_.score(mode.second, &w, &l, &d);
        scorePanel_->setScore(w, l, d);
        scorePanel_->show();
        btUndo_->show();
    }
    else
    {
        scorePanel_->hide();
        btUndo_->hide();
        if (mode.first == 0)
            btUndo_->show();
        else
            btUndo_->hide();
    }
    btUndo_->setEnabled(false);
}//newGame

bool CentralWidget::newGameP2PServer()
{
    if (!gameServer_->isListening())
        if (!gameServer_->listen())
        {
            showMessage(this, tr("Error!!\nUnable to start the game server: %1.").arg(gameServer_->errorString()), MSG_ERROR);
            return false;
        }

    ServerDialog serverDialog(gameServer_, &gameConnection_, isNetworkGame_, this);
    int res = serverDialog.exec();
    if (res == QDialog::Rejected)
        return false;
    connect(gameConnection_, SIGNAL(readyRead()), this, SLOT(slotReadNetworkData()));
    connect(gameConnection_, SIGNAL(error(QAbstractSocket::SocketError)), this, SLOT(displayP2PError(QAbstractSocket::SocketError)));
    writeP2PData(gameConnection_, QString(PROTOCOL_VERSION));
    //TODO: wait answer from client
    P2PState_ = WAIT_CLIENT_VERSION;
    return true;
}

bool CentralWidget::newGameP2PClient()
{
    ClientDialog clientDialog(gameClient_, this);
    int res = clientDialog.exec();
    if (res == QDialog::Rejected)
        return false;
    P2PState_ = WAIT_SERVER_VERSION;
    return true;
}

void
CentralWidget::slotReadNetworkData()
{
    QString data;
    if (gameMode_ == 4)
        data = readP2PData(gameClient_);
    else
        data = readP2PData(gameConnection_);
    if (!data.isEmpty())
    {
        //qDebug("st %d", P2PState_);
        //qDebug() << data;
        if (P2PState_ == WAIT_CLIENT_VERSION && data.at(0) == 'v')
        {
            writeP2PData(gameConnection_, QString("newgame"));
            P2PState_ = RUNNING;
            reversiGame_.init(gameMode_, strength_);
        }
        else if (P2PState_ == WAIT_SERVER_VERSION && data.at(0) == 'v')
        {
            writeP2PData(gameClient_, QString(PROTOCOL_VERSION));
            P2PState_ = WAIT_SERVER_NEW_GAME;
        }
        else if (P2PState_ == WAIT_SERVER_NEW_GAME && data.at(0) == 'n')
        {
            P2PState_ = RUNNING;
            reversiGame_.init(gameMode_, strength_);
        }
        else if ((P2PState_ == CLIENT_PAUSED || P2PState_ == SERVER_PAUSED) && data.at(0) == 'u')
        {
            P2PState_ = RUNNING;
        }
        else if (data.at(0) == 'x')
        {
            QChar x, y;
            x = data.at(1);
            y = data.at(3);
            reversiGame_.makeStep(x.toAscii() - '0', y.toAscii() - '0');
        }
    }
}//slotReadNetworkData

void
CentralWidget::slotPlayerStep(int x, int y)
{
    if (P2PState_ != RUNNING)
    {
        switch (P2PState_)
        {
            case WAIT_CLIENT_VERSION:
                showMessage(this, tr("Warning!\nWait answer from client."), MSG_WARNING);
                break;
            case CLIENT_PAUSED:
                showMessage(this, tr("Warning!\nClient is paused."), MSG_WARNING);
                break;
            case SERVER_PAUSED:
                showMessage(this, tr("Warning!\nServer is paused."), MSG_WARNING);
                break;
            case WAIT_SERVER_VERSION:
            case WAIT_SERVER_NEW_GAME:
                showMessage(this, tr("Warning!\nWait answer from server."), MSG_WARNING);
                break;
        }
        return;
    }//(P2PState_ != RUNNING)

    if (gameMode_ == 3 && reversiGame_.currentPlayer() == White)
    {
        showMessage(this, tr("Warning!\nIt's White player turn now."), MSG_WARNING);
        return;
    }
    if (gameMode_ == 4 && reversiGame_.currentPlayer() == Black)
    {
        showMessage(this, tr("Warning!\nIt's Black player turn now."), MSG_WARNING);
        return;
    }

    QString data("x%1y%2");
    reversiGame_.makeStep(x, y);
    if (gameMode_ == 3)
        writeP2PData(gameConnection_, data.arg(x).arg(y));
    else if (gameMode_ == 4)
        writeP2PData(gameClient_, data.arg(x).arg(y));
}//slotPlayerStep

void CentralWidget::stepUnavailable()
{
    static const char *players[] = {QT_TR_NOOP("White"), QT_TR_NOOP("Black")};
    QString info;
    info = QString(tr("%1 can't move. %2 turn.")).arg(tr(players[reversiGame_.currentPlayer()])).arg(tr(players[reversiGame_.opponentPlayer()]));

    //set pause while message will be shown
    if (isNetworkGame_)
    {
        if (gameMode_ == 4)
        {
            P2PState_ = SERVER_PAUSED;
        }
        else if (gameMode_ == 3)
        {
            P2PState_ = CLIENT_PAUSED;
        }
    }//if (isNetworkGame_)

    showMessage(parentWidget(), info, MSG_WARNING);

    //set perr about unpaused
    if (isNetworkGame_)
    {
        if (gameMode_ == 4)
        {
            writeP2PData(gameClient_, QString("unpaused"));
        }
        else if (gameMode_ == 3)
        {
            writeP2PData(gameConnection_, QString("unpaused"));
        }
    }//if (isNetworkGame_)
}//stepUnavailable

void CentralWidget::displayP2PError(QAbstractSocket::SocketError socketError)
{
    switch (socketError) {
        case QAbstractSocket::HostNotFoundError:
            showMessage(parentWidget(), tr("Warning!\nThe host was not found."), MSG_WARNING);
            break;
        case QAbstractSocket::ConnectionRefusedError:
            showMessage(parentWidget(), tr("Warning!\nThe connection was refused by the peer."), MSG_WARNING);
            break;
        default:
            QString errmsg(tr("The following error occurred: %1."));
            if (gameMode_ == 3)
                showMessage(parentWidget(), errmsg.arg(gameConnection_->errorString()));
            else
                showMessage(parentWidget(), errmsg.arg(gameClient_->errorString()));
    }
}//displayP2PError

void CentralWidget::slotOptions()
{
    OptionsDialog optDlg(&settings_, parentWidget());
    if (optDlg.exec() == QDialog::Accepted)
        fieldView_->setAnimationSpeed(settings_.animationSpeed());
}

void CentralWidget::slotGameOver()
{
    QString info;
    unsigned white_score = reversiGame_.score(White);
    unsigned black_score = reversiGame_.score(Black);
    info = QString(tr("Game over!\n White player %1; Black player %2.")).arg(white_score).arg(black_score);
    showMessage(parentWidget(), info);

    QPair<int, int> mode = modePanel_->mode();
    if (mode.first == 1 || mode.first == 2) //human vs comp or comp vs human
    {
        Score::score_type s_type = Score::DRAW;
        if (black_score != white_score)
        {
            if (mode.first == 1)
                s_type = (black_score > white_score) ? Score::WIN : Score::LOSS;
            else
                s_type = (black_score < white_score) ? Score::WIN : Score::LOSS;
        }
        score_.incScore(mode.second, s_type);
        unsigned w, l, d;
        score_.score(mode.second, &w, &l, &d);
        scorePanel_->setScore(w, l, d);
    }//if (mode.first == 1 || mode.first == 2)
    if (settings_.startNewGame())
        newGame();
}

void CentralWidget::slotResetScore()
{
    score_.reset();
    scorePanel_->setScore(0, 0, 0);
}

void CentralWidget::slotEnableUndo(bool isEnabled)
{
    btUndo_->setEnabled(isEnabled);
}
