/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include <qabstractkineticscroller.h>
#include <private/qabstractkineticscroller_p.h>

#include <QGraphicsView>
#include <QScrollBar>
#include <QAbstractItemView>
#include <QCoreApplication>
#include <QMouseEvent>

#include <QtDebug>

QT_BEGIN_NAMESPACE

// #define KINETIC_SCROLLER_DEBUG

#ifdef KINETIC_SCROLLER_DEBUG
#  define qKSDebug  qDebug
#else
#  define qKSDebug  while (false) qDebug
#endif

extern bool qt_sendSpontaneousEvent(QObject *receiver, QEvent *event);

/*! \class QAbstractKineticScroller
    \brief The QAbstractKineticScroller class allows kinetic scrolling any widget.

    The kinetic scroller installs an event filter (by calling
    attachToWidget()) on a given widget (an QAbstractScrollArea in most
    cases) to handle mouse presses and moves on the widget.  They will be
    interpreted as scroll actions depending on the current state().

    Even though this kinetic scroller has a huge number of settings, we
    recommend to leave them all at their default values.  In case you really
    want to change them you can try out the kineticscroller example in the
    maemo5 examples directory.

    \sa QWidget
*/


/*!
    Constructs a new kinetic scroller.
*/
QAbstractKineticScroller::QAbstractKineticScroller()
    : QObject(*new QAbstractKineticScrollerPrivate())
{
    Q_D(QAbstractKineticScroller);
    d->init();
}

/*! \internal
*/
QAbstractKineticScroller::QAbstractKineticScroller(QAbstractKineticScrollerPrivate &dd)
    : QObject(dd)
{
    Q_D(QAbstractKineticScroller);
    d->init();
}

/*!
    Destroys the scroller.
*/
QAbstractKineticScroller::~QAbstractKineticScroller()
{
}

/*!
    Returns the widget the kinetic scroller is attached to.

    \sa setWidget()
*/
QWidget *QAbstractKineticScroller::widget() const
{
    Q_D(const QAbstractKineticScroller);
    return d->widget;
}

/*!
    Attaches a new widget to this kinetic scroller, replacing an existing
    one.  The kinetic scroller object will be re-parented to the widget \a w.

    This function will call attachToWidget() and removeFromWidget()
    respectively to actually set up the event filters.

    \sa attachToWidget(), removeFromWidget()
*/
void QAbstractKineticScroller::setWidget(QWidget *w)
{
    Q_D(QAbstractKineticScroller);
    if (d->widget)
        removeFromWidget();

    d->widget = w;
    d->state = Inactive;
    if (d->idleTimerId)
        killTimer(d->idleTimerId);
    d->idleTimerId = 0;
    if (d->scrollTimerId)
        killTimer(d->scrollTimerId);
    d->scrollTimerId = 0;
    d->velocity = d->oldVelocity = QPointF(0, 0);
    d->overShootDist = QPoint(0, 0);
    d->overShooting = 0;
    d->ignoreEvents = false;

    setParent(w);
    if (d->widget)
        attachToWidget();
}

/*!
    Returns the current state of the kinetic scroller.
*/
QAbstractKineticScroller::State QAbstractKineticScroller::state() const
{
    Q_D(const QAbstractKineticScroller);
    return d->state;
}

/*!
    This filter function changes, blocks or creates mouse events \a e for QObject \a o
    depending on the detected kinetic scrolling state.
    It is mostly an internal implementation.
*/
bool QAbstractKineticScroller::eventFilter(QObject *o, QEvent *e)
{
    Q_D(QAbstractKineticScroller);
    qKSDebug("MP: eventFilter ignore? %d", d->ignoreEvents);

    bool res = false;

    if (!d->ignoreEvents) {
        switch (e->type()) {
        case QEvent::MouseButtonPress:
            res = d->handleMousePress(static_cast<QMouseEvent *>(e));
            if (d->widget) { // re-install event filter so that we get the priority for the mouse release
                removeFromWidget();
                attachToWidget();
            }
            break;
        case QEvent::MouseMove:
            res = d->handleMouseMove(static_cast<QMouseEvent *>(e));
            break;
        case QEvent::MouseButtonRelease:
            res = d->handleMouseRelease(static_cast<QMouseEvent *>(e));
            break;
        case QEvent::ChildAdded:
        case QEvent::ChildRemoved:
            d->handleChildEvent(static_cast<QChildEvent *>(e));
            break;
        default:
            break;
        }
    }
    return res ? true : QObject::eventFilter(o, e);
}

/*! \internal
    The timer event will be triggered by the scroll and idle timer.
*/
void QAbstractKineticScroller::timerEvent(QTimerEvent *te)
{
    Q_D(QAbstractKineticScroller);

    if (te->timerId() == d->idleTimerId)
        d->handleIdleTimer();
    else if (te->timerId() == d->scrollTimerId)
        d->handleScrollTimer();
}

QAbstractKineticScrollerPrivate::QAbstractKineticScrollerPrivate()
    : enabled(true), mode(QAbstractKineticScroller::AutoMode), state(QAbstractKineticScroller::Inactive),
    lastType(QEvent::User), moved(false), lastIn(true),
    firstDrag(true), centerOnChildFocusPending(false), lowFrictionMode(false),
    scrollTo(-1, -1), bounceSteps(3), maxOverShoot(150, 150), vmaxOverShoot(130),
    overShootDist(0, 0), overShooting(0),
    minVelocity(10), maxVelocity(3500), fastVelocityFactor(0.01), deceleration(0.85),
    scrollsPerSecond(20), panningThreshold(25), directionErrorMargin(10), force(50),
    dragInertia(0.85), scrollTime(1000),
    childWidget(0),
    idleTimerId(0), scrollTimerId(0), widget(0), ignoreEvents(false)
{ }

QAbstractKineticScrollerPrivate::~QAbstractKineticScrollerPrivate()
{ }

void QAbstractKineticScrollerPrivate::init()
{ }

void QAbstractKineticScrollerPrivate::changeState(QAbstractKineticScroller::State newState)
{
    Q_Q(QAbstractKineticScroller);

    if (newState != state) {
        QAbstractKineticScroller::State oldState = state;
        state = newState;
        emit q->stateChanged(newState, oldState);
    }
}

void QAbstractKineticScrollerPrivate::sendEvent(QWidget *w, QEvent *e)
{
    ignoreEvents = true;
    qt_sendSpontaneousEvent(w, e);
    ignoreEvents = false;
}

void QAbstractKineticScrollerPrivate::handleChildEvent(QChildEvent *e)
{
    if (e->child()->isWidgetType())
        static_cast<QWidget *>(e->child())->setAttribute(Qt::WA_TransparentForMouseEvents, e->added());
}

void QAbstractKineticScrollerPrivate::handleScrollTimer()
{
    Q_Q(QAbstractKineticScroller);

    qKSDebug("handleScrollTimer: %d/%d", motion.x(), motion.y());

    if (!motion.isNull())
        setPositionHelper(q->position() - overShootDist - motion);
    q->killTimer(scrollTimerId);
    scrollTimerId = 0;
}

void QAbstractKineticScrollerPrivate::handleIdleTimer()
{
    Q_Q(QAbstractKineticScroller);

    if (mode == QAbstractKineticScroller::PushMode && overShootDist.isNull()) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
        changeState(QAbstractKineticScroller::Inactive);
        return;
    }
    qKSDebug() << "idle timer - velocity: " << velocity << " overShoot: " << overShootDist;

    setPositionHelper(q->position() - overShootDist - velocity.toPoint());

    if (state == QAbstractKineticScroller::AutoScrolling) {
        // Decelerate gradually when pointer is raised
        if (overShootDist.isNull()) {
            overShooting = 0;

            // in case we move to a specific point do not decelerate when arriving
            if (scrollTo.x() != -1 || scrollTo.y() != -1) {

                // -- check if target was reached
                QPoint pos = q->position();
                QPointF  dist = QPointF(scrollTo - pos);

                qKSDebug() << "handleIdleTimer dist:" << dist << " scrollTo:" << scrollTo;

                // -- break if target was reached
                if ((velocity.x() < 0.0 && dist.x() <= 0.0) ||
                    (velocity.x() > 0.0 && dist.x() >= 0.0) )
                    velocity.setX(0.0);

                if ((velocity.y() < 0.0 && dist.y() <= 0.0) ||
                    (velocity.y() > 0.0 && dist.y() >= 0.0) )
                    velocity.setY(0.0);

                // -- break if we reached the borders
                /*
                QRect range = q->positionRange();
                if ((range.left() > pos.x()-velocity.x() && velocity.x() > 0.0) ||
                    (range.right() < pos.x()-velocity.x() && velocity.x() < 0.0))
                    velocity.setX(0.0);

                if ((range.top() > pos.y()-velocity.y() && velocity.y() > 0.0) ||
                    (range.bottom() < pos.y()-velocity.y() && velocity.y() < 0.0))
                    velocity.setY(0.0);
                */

                if (velocity.x() == 0.0 && velocity.y() == 0.0) {
                    q->killTimer(idleTimerId);
                    idleTimerId = 0;
                    changeState(QAbstractKineticScroller::Inactive);
                    return;
                }

                // -- don't get too slow if target was not yet reached
                if (qAbs(velocity.x()) >= qreal(1.5))
                    velocity.rx() *= deceleration;
                if (qAbs(velocity.y()) >= qreal(1.5))
                    velocity.ry() *= deceleration;

            } else {
                if (!lowFrictionMode || (qAbs(velocity.x()) < qreal(0.8) * maxVelocity))
                    velocity.rx() *= deceleration;
                if (!lowFrictionMode || (qAbs(velocity.y()) < qreal(0.8) * maxVelocity))
                    velocity.ry() *= deceleration;

                if ((qAbs(velocity.x()) < qreal(1.0)) && (qAbs(velocity.y()) < qreal(1.0))) {
                    velocity = QPointF(0, 0);
                    q->killTimer(idleTimerId);
                    idleTimerId = 0;
                    changeState(QAbstractKineticScroller::Inactive);
                }
            }
        } else { // overShootDist != 0
            overShooting++;
            scrollTo = QPoint(-1, -1);

            /* When the overshoot has started we continue for
             * PROP_BOUNCE_STEPS more steps into the overshoot before we
             * reverse direction. The deceleration factor is calculated
             * based on the percentage distance from the first item with
             * each iteration, therefore always returning us to the
             * top/bottom most element
             */
            if (overShooting < bounceSteps) {
                velocity.setX( overShootDist.x() / maxOverShoot.x() * velocity.x() );
                velocity.setY( overShootDist.y() / maxOverShoot.y() * velocity.y() );
            } else {
                velocity.setX( -overShootDist.x() * 0.8 );
                velocity.setY( -overShootDist.y() * 0.8 );

                // ensure a minimum speed when scrolling back or else we might never return
                if (velocity.x() > -1.5 && velocity.x() < 0.0)
                    velocity.setX(-1.5);
                if (velocity.x() <  1.5 && velocity.x() > 0.0)
                    velocity.setX( 1.5);
                if (velocity.y() > -1.5 && velocity.y() < 0.0)
                    velocity.setY(-1.5);
                if (velocity.y() <  1.5 && velocity.y() > 0.0)
                    velocity.setY( 1.5);
            }

            velocity.setX( qBound((qreal)-vmaxOverShoot, velocity.x(), (qreal)vmaxOverShoot));
            velocity.setY( qBound((qreal)-vmaxOverShoot, velocity.y(), (qreal)vmaxOverShoot));
        } // overshoot
    } else if (mode == QAbstractKineticScroller::AutoMode) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
        changeState(QAbstractKineticScroller::Inactive);
    }
}


static QWidget *mouseTransparentChildAtGlobalPos(QWidget *parent, const QPoint &gp)
{
    foreach (QWidget *w, parent->findChildren<QWidget *>()) {
        if (w && !w->isWindow() && w->isVisible() && (w->rect().contains(w->mapFromGlobal(gp)))) {
            if (QWidget *t = mouseTransparentChildAtGlobalPos(w, gp))
                return t;
            else
                return w;
        }
    }
    return 0;
}

bool QAbstractKineticScrollerPrivate::handleMousePress(QMouseEvent *e)
{
    Q_Q(QAbstractKineticScroller);
    qKSDebug("MP: start");
    if (e->button() != Qt::LeftButton || !enabled || !widget->isEnabled())
        return false;

    QRect posRange = q->positionRange();
    bool canScrollX = (posRange.width() > 1);
    bool canScrollY = (posRange.height() > 1);
    scrollTo = QPoint(-1, -1);

    pos = e->globalPos();
    ipos = pos;

    childWidget = 0;

    // don't allow a click if we're still moving fast
    if ((qAbs(velocity.x()) <= (maxVelocity * fastVelocityFactor)) &&
        (qAbs(velocity.y()) <= (maxVelocity * fastVelocityFactor))) {
        if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(widget))
            childWidget = area->viewport();
        else
            childWidget = widget;

        if (QWidget *w = mouseTransparentChildAtGlobalPos(childWidget, e->globalPos()))
            childWidget = w;
    }

    if (canScrollX || canScrollY) {
        lastTime.start();
        lastPressTime.start();
        lastType = e->type();

        // stop scrolling on mouse press (so you can flick, then hold to stop)
        oldVelocity = velocity;
        velocity = QPointF(0, 0);

        if (idleTimerId) {
            q->killTimer(idleTimerId);
            idleTimerId = 0;
            changeState(QAbstractKineticScroller::Inactive);
        }

        changeState(QAbstractKineticScroller::MousePressed);
    }
    if (childWidget) {
        lastIn = true;

        if (childWidget) {
            QEvent ee(QEvent::Enter);
            sendEvent(childWidget, &ee);

            QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                           e->globalPos(), e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);
        }
    }
    qKSDebug("MP: end");
    return true;
}

bool QAbstractKineticScrollerPrivate::handleMouseMove(QMouseEvent *e)
{
    qKSDebug() << "MM: pos: " << e->globalPos() << " - time: " << QTime::currentTime().msec();
    if (!(e->buttons() & Qt::LeftButton) || !enabled || !widget->isEnabled())
        return false;
    if (moved && !lastTime.elapsed())
        return true;
    if (lastType == QEvent::MouseButtonPress)
        firstDrag = true;

    QPoint delta = e->globalPos() - pos;
    qKSDebug() << "Delta: " << delta << " Moved: " << moved;
    if (childWidget)
        qKSDebug() << "transparent?: " << childWidget->testAttribute(Qt::WA_TransparentForMouseEvents);

    if (!moved) {
        checkMove(e, delta);
        qKSDebug() << "After Check: " << delta << " Moved: " << moved;
    }
    if (moved) {
        if (childWidget) {
            // send leave and mouse up
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);

            QMouseEvent me(e->type(), QPoint(-INT_MAX, -INT_MAX),
                           e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);

            childWidget = 0;
        }
        changeState(QAbstractKineticScroller::Pushing);

        handleMove(e, delta);
        qKSDebug() << "After Handle: " << delta << " Moved: " << moved;
    } else if (childWidget) {
        // send leave event to child according to lastIn and set lastIn
        if (childWidget) {
            bool in = childWidget->geometry().contains(childWidget->mapFromGlobal(e->globalPos()));

            if (in != lastIn) {
                QEvent ele(in ? QEvent::Enter : QEvent::Leave);
                sendEvent(childWidget, &ele);
                lastIn = in;
            }
        }
    }
    lastTime.restart();
    lastType = e->type();

    if (childWidget) {
        // send mouse move event to child
        QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                e->globalPos(), e->button(), e->buttons(), e->modifiers());
        sendEvent(childWidget, &me);
    }
    qKSDebug("MM: end");
    return true;
}

bool QAbstractKineticScrollerPrivate::handleMouseRelease(QMouseEvent *e)
{
    qKSDebug() << "MR: pos: " << e->globalPos() << " - time: " << QTime::currentTime().msec();
    Q_Q(QAbstractKineticScroller);

    if (e->button() != Qt::LeftButton || !enabled || !widget->isEnabled())
        return false;

    // if last event was a motion-notify we have to check the
    // movement and launch the animation
    if (lastType == QEvent::MouseMove) {
        if (moved) {
            // move all the way to the last position now
            if (scrollTimerId) {
                q->killTimer(scrollTimerId);
                scrollTimerId = 0;
                setPositionHelper(q->position() - overShootDist - motion);
                motion = QPoint(0, 0);
            }
        }
    }
    // If overshoot has been initiated with a finger down,
    // on release set max speed
    if (overShootDist.x()) {
        overShooting = bounceSteps; // Hack to stop a bounce in the finger down case
        velocity.setX(overShootDist.x() * qreal(0.9));

    }
    if (overShootDist.y()) {
        overShooting = bounceSteps; // Hack to stop a bounce in the finger down case
        velocity.setY(overShootDist.y() * qreal(0.9));
    }

    bool forceFast = true;

    // if widget was moving fast in the panning, increase speed even more
    if ((lastPressTime.elapsed() < FastClick) &&
        ((qAbs(oldVelocity.x()) > minVelocity) ||
         (qAbs(oldVelocity.y()) > minVelocity)) &&
        ((qAbs(oldVelocity.x()) > MinimumAccelerationThreshold) ||
         (qAbs(oldVelocity.y()) > MinimumAccelerationThreshold))) {

        qKSDebug() << "FAST CLICK - using oldVelocity " << oldVelocity;
        int signX = 0, signY = 0;
        if (velocity.x())
            signX = (velocity.x() > 0) == (oldVelocity.x() > 0) ? 1 : -1;
        if (velocity.y())
            signY = (velocity.y() > 0) == (oldVelocity.y() > 0) ? 1 : -1;

        // calculate the acceleration velocity
        QRect range = q->positionRange();
        accelerationVelocity  = QPoint(0, 0);
        if (widget->width())
            accelerationVelocity.setX( qMin( (int)q->maximumVelocity(), range.width() / widget->width() * AccelFactor));
        if (widget->height())
            accelerationVelocity.setY( qMin( (int)q->maximumVelocity(), range.height() / widget->height() * AccelFactor));

        velocity.setX(signX * (oldVelocity.x() + (oldVelocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x())));
        velocity.setY(signY * (oldVelocity.y() + (oldVelocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y())));
        forceFast = false;
    }

    if ((qAbs(velocity.x()) >= minVelocity) ||
        (qAbs(velocity.y()) >= minVelocity)) {

        qKSDebug() << "over min velocity: " << velocity;
        // we have to move because we are in overshooting position
        if (!moved) {
            // (opt) emit panningStarted()
        }
        // (must) enable scrollbars

        if (forceFast) {
            if ((qAbs(velocity.x()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.x() > MaximumVelocityThreshold)) {
                velocity.setX(velocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x());
            }
            if ((qAbs(velocity.y()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.y() > MaximumVelocityThreshold)) {
                velocity.setY(velocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y());
            }
            qKSDebug() << "Force fast is on - velocity: " << velocity;

        }
        changeState(QAbstractKineticScroller::AutoScrolling);

    } else {
        if (centerOnChildFocusPending) {
            centerOnChildFocus();
        }

        if (moved) {
            // (opt) emit panningFinished()
        }
        changeState(QAbstractKineticScroller::Inactive);
    }

    // -- create the idle timer if we are auto scrolling or overshooting.
    if (!idleTimerId
            && ((qAbs(velocity.x()) >= minVelocity)
                || (qAbs(velocity.y()) >= minVelocity)
                || overShootDist.x()
                || overShootDist.y()) ) {
        idleTimerId = q->startTimer(1000 / scrollsPerSecond);
    }

    centerOnChildFocusPending = false;
    lastTime.restart();
    lastType = e->type();

    if (!childWidget) {
        bool wasMoved = moved;
        moved = false;
        return wasMoved; // when we did move we use the mouse release ourself

    } else {
        if (!childWidget->rect().contains(childWidget->mapFromGlobal(e->globalPos()))) {
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);
        }

        QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                e->button(), e->buttons(), e->modifiers());
        sendEvent(childWidget, &me);

        childWidget = 0;
        moved = false;
        return true;
    }
}

void QAbstractKineticScrollerPrivate::checkMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QAbstractKineticScroller);

    if (firstDrag && !moved && ((qAbs(delta.x()) > panningThreshold) || (qAbs(delta.y()) > panningThreshold))) {
        moved = true;
        delta = QPoint(0, 0);

        if (firstDrag) {
            int deltaXtoY = qAbs(ipos.x() - me->globalPos().x()) - qAbs(ipos.y() - me->globalPos().y());

            qKSDebug() << "First Drag with delta " << delta << ", greater than " << panningThreshold << " -- deltaXtoY: " << deltaXtoY;

            QRect posRange = q->positionRange();
            bool canScrollX = (posRange.width() > 1);
            bool canScrollY = (posRange.height() > 1);

            if (deltaXtoY < 0) {
                if (!canScrollY && (!canScrollX || (-deltaXtoY >= directionErrorMargin)))
                    moved = false;
            } else {
                if (!canScrollX && (!canScrollY || (deltaXtoY >= directionErrorMargin)))
                    moved = false;
            }
        }
        firstDrag = false;

        if (moved && (mode == QAbstractKineticScroller::AccelerationMode)) {

            if (!idleTimerId) {
                changeState(QAbstractKineticScroller::AutoScrolling);
                idleTimerId = q->startTimer(1000 / scrollsPerSecond);
            }
        }
    }
}

void QAbstractKineticScrollerPrivate::handleMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QAbstractKineticScroller);

    switch (mode) {
    case QAbstractKineticScroller::PushMode:
        // Scroll by the amount of pixels the cursor has moved
        // since the last motion event.
        scrollUpdate(delta);
        pos = me->globalPos();
        break;

    case QAbstractKineticScroller::AccelerationMode:
        // Set acceleration relative to the initial click
        // epos = me->globalPos(); //TODO: what the heck is epos?
        velocity.setX(qreal(delta.x() < 0 ? -1 : 1) * ((qreal(qAbs(delta.x())) / qreal(widget->width()) * (maxVelocity - minVelocity)) + minVelocity));
        velocity.setY(qreal(delta.y() < 0 ? -1 : 1) * ((qreal(qAbs(delta.y())) / qreal(widget->height()) * (maxVelocity - minVelocity)) + minVelocity));
        break;

    case QAbstractKineticScroller::AutoMode:
        QPointF newVelocity = calculateVelocity(me->globalPos() - pos, lastTime.elapsed());
        QRect posRange = q->positionRange();

        if (!posRange.width()) {
            delta.setX(0);
            newVelocity.setX(0);
        }
        if (!posRange.height()) {
            delta.setY(0);
            newVelocity.setY(0);
        }
        velocity = newVelocity;

        scrollUpdate(delta);

        if (posRange.width())
            pos.setX(me->globalPos().x());
        if (posRange.height())
            pos.setY(me->globalPos().y());
        break;
    }
}

QPointF QAbstractKineticScrollerPrivate::calculateVelocity(const QPointF &dPixel, int dTime)
{
    qKSDebug() << "calculateVelocity(dP = " << dPixel << ", dT = " << dTime << ") -- velocity: " << velocity;

    QPointF newv = velocity;

    // fast than 15 pix / ms seems bogus (that's a screen height in 1/30 second)
    if ((dPixel / qreal(dTime)).manhattanLength() < 25) {
        QPointF rawv = dPixel / qreal(dTime) * qreal(force);
        newv = newv * (qreal(1) - dragInertia) + rawv * dragInertia;
    }

    qKSDebug() << " --> " << newv << " (before clamping)";

    newv.setX(dPixel.x() ? qBound(-maxVelocity, newv.x(), maxVelocity) : velocity.x());
    newv.setY(dPixel.y() ? qBound(-maxVelocity, newv.y(), maxVelocity) : velocity.y());
    return newv;
}

void QAbstractKineticScrollerPrivate::scrollUpdate(const QPoint &delta)
{
    Q_Q(QAbstractKineticScroller);

    if (scrollTimerId) {
        motion += delta;
    } else {
        // we do not delay the first event but the next ones
        setPositionHelper(q->position() - overShootDist - delta);
        motion = QPoint(0, 0);
        scrollTimerId = q->startTimer(1000 / MotionEventsPerSecond);
    }
}


void QAbstractKineticScrollerPrivate::centerOnChildFocus()
{
    //TODO:
}

/*!
    Enables or disables this kinetic scroller.
*/
void QAbstractKineticScroller::setEnabled(bool enable)
{
    Q_D(QAbstractKineticScroller);
    d->enabled = enable;
}

/*!
    Returns wether this kinetic scroller is enabled or not.
*/
bool QAbstractKineticScroller::isEnabled() const
{
    Q_D(const QAbstractKineticScroller);
    return d->enabled;
}


/*!
    Returns the scrolling mode.

    \sa setMode(), Mode
*/
QAbstractKineticScroller::Mode QAbstractKineticScroller::mode() const
{
    Q_D(const QAbstractKineticScroller);
    return d->mode;
}

/*!
    Set the scrolling mode to \a mode

    \sa mode(), Mode
*/
void QAbstractKineticScroller::setMode(Mode mode)
{
    Q_D(QAbstractKineticScroller);
    d->mode = mode;
}


/*!
    Returns the value of the low-friction-mode.
    Low friction means that scrolling will not be slowed down by the decelerationFactor.

    \sa decelerationFactor(), setLowFrictionEnabled()
*/
bool QAbstractKineticScroller::isLowFrictionEnabled() const
{
    Q_D(const QAbstractKineticScroller);
    return d->lowFrictionMode;
}

/*!
    Enables or disabled the low-friction-mode.

    \sa decelerationFactor(), isLowFrictionEnabled()
*/
void QAbstractKineticScroller::setLowFrictionEnabled(bool b)
{
    Q_D(QAbstractKineticScroller);
    d->lowFrictionMode = b;
}


/*!
    Returns the value of the dragInertia.
    Percentage of the calculated speed in each moment we are are going to use to calculate
    the launch speed.

    \sa setDragInertia()
*/
qreal QAbstractKineticScroller::dragInertia() const
{
    Q_D(const QAbstractKineticScroller);
    return d->dragInertia;
}

/*!
    Sets the value of the drag intertia to \a inertia

    \sa dragInertia()
*/
void QAbstractKineticScroller::setDragInertia(qreal inertia)
{
    Q_D(QAbstractKineticScroller);
    d->dragInertia = inertia;
}


/*!
    Returns the directionErrorMargin.
    This value influences whether a scroll is realized as such if the mouse is dragged
    diagonal.

    \sa setDirectionErrorMargin()
*/
int QAbstractKineticScroller::directionErrorMargin() const
{
    Q_D(const QAbstractKineticScroller);
    return d->directionErrorMargin;
}

/*!
    Sets the directionErrorMargin to \a errorMargin.

    \sa directionErrorMargin()
*/
void QAbstractKineticScroller::setDirectionErrorMargin(int errorMargin)
{
    Q_D(QAbstractKineticScroller);
    d->directionErrorMargin = errorMargin;
}


/*!
    Returns the panningThreshold value.
    The amount in pixels the mouse must move until scrolling is started.

    \sa setPanningThreshold()
*/
int QAbstractKineticScroller::panningThreshold() const
{
    Q_D(const QAbstractKineticScroller);
    return d->panningThreshold;
}

/*!
    Sets the panning threshold to \a threshold

    \sa panningThreshold()
*/
void QAbstractKineticScroller::setPanningThreshold(int threshold)
{
    Q_D(QAbstractKineticScroller);
    d->panningThreshold = threshold;
}


/*!
    Returns the deceleration factor which is the percentage of scrolling velocity
    remaining after every scroll step.

    \sa setDecelerationFactor()
*/
qreal QAbstractKineticScroller::decelerationFactor() const
{
    Q_D(const QAbstractKineticScroller);
    return d->deceleration;
}

/*!
    Sets the deceleration factor to \a f.

    \sa decelerationFactor()
*/
void QAbstractKineticScroller::setDecelerationFactor(qreal f)
{
    Q_D(QAbstractKineticScroller);
    d->deceleration = f;
}


/*!
    Returns the fast velocity factor which is the factor that
    determines if the scrolling speed is too fast for a mouse click.

    If the current velocity is greater then maximumVelocity()*fastVelocityFactor()
    then the mouse click will just stop the scrolling.
    If the velocity is less then a click will stop the scrolling and transmitted to
    the scroll area.

    \sa setFastVelocityFactor()
*/
qreal QAbstractKineticScroller::fastVelocityFactor() const
{
    Q_D(const QAbstractKineticScroller);
    return d->fastVelocityFactor;
}

/*!
    Sets the fast velocity factor to \a f.

    \sa fastVelocityFactor()
*/
void QAbstractKineticScroller::setFastVelocityFactor(qreal f)
{
    Q_D(QAbstractKineticScroller);
    d->fastVelocityFactor = f;
}


/*!
    Returns the minimumVelocity.
    The minimum velocity is the slowest speed in the acceleration mode.

    \sa setMinimumVelocity()
*/
qreal QAbstractKineticScroller::minimumVelocity() const
{
    Q_D(const QAbstractKineticScroller);
    return d->minVelocity;
}

/*!
    Sets the miminum velocity to \a v.

    \sa minimumVelocity()
*/
void QAbstractKineticScroller::setMinimumVelocity(qreal v)
{
    Q_D(QAbstractKineticScroller);
    d->minVelocity = v;
}


/*!
    Returns the maximumVelocity.
    The maximum velocity is the slowest speed in the acceleration mode.

    \sa setMaximumVelocity()
*/
qreal QAbstractKineticScroller::maximumVelocity() const
{
    Q_D(const QAbstractKineticScroller);
    return d->maxVelocity;
}

/*!
    Sets the maximum velocity to \a v.

    \sa maximumVelocity()
*/
void QAbstractKineticScroller::setMaximumVelocity(qreal v)
{
    Q_D(QAbstractKineticScroller);
    d->maxVelocity = v;
}


/*!
    Start scrolling to the position \a pos.
    The area will scroll in scrollTime milliseconds to the given position.
    The default for the private scrollTime is 1000 milliseconds.

    The speed will be calculated so that the scrolling reaches the given positon after
    this time. However the speed a the end position is not guaranteed to be zero.

    If a position outside the positionRange is given the area will overshoot and not end
    up at the invalid position.

    \sa positionRange()
*/
void QAbstractKineticScroller::scrollTo(const QPoint &pos)
{
    Q_D(QAbstractKineticScroller);

    if ((pos == position()) ||
        (d->state == QAbstractKineticScroller::MousePressed) ||
        (d->state == QAbstractKineticScroller::Pushing)) {
        return;
    }

    // --- calculate the initial velocity to get to the point
    // -- calc the number of steps we have:
    int steps = d->scrollTime * d->scrollsPerSecond / 1000;

    // -- calc the distance we will move with a starting velocity of 1.0
    float dist = 0.0;
    float curVel = 1.0;
    for (int i = 0; i<steps; i++) {
        dist += curVel;
        curVel *= d->deceleration;
    }

    // --- start the scrolling
    d->scrollTo = pos;
    d->velocity = - QPointF(pos-position()) / dist;
    d->mode = QAbstractKineticScroller::AutoMode;

    qKSDebug() << "QAbstractKineticScroller::scrollTo new pos:" << pos << " velocity:"<<d->velocity;

    if (!d->idleTimerId) {
        d->changeState(QAbstractKineticScroller::AutoScrolling);
        d->idleTimerId = startTimer(1000 / d->scrollsPerSecond);
    }
}

/*!
    Starts scrolling the widget so that the point \a pos is visible inside the
    viewport with margins specified in pixels by xmargin and ymargin.  If
    the specified point cannot be reached, the contents are scrolled to the
    nearest valid position.  The default value for both margins is 50
    pixels.

    The actual scrolling is done by calling scrollTo()
*/
void QAbstractKineticScroller::ensureVisible(const QPoint &pos, int xmargin, int ymargin)
{
    QSize visible = viewportSize();
    QPoint currentPos = position();

    qKSDebug() << "QAbstractKineticScroller::ensureVisible(" << pos << ", " << xmargin << ", " << ymargin << ") - position: " << position();

    QRect posRect(pos.x() - xmargin, pos.y() - ymargin, 2 * xmargin, 2 * ymargin);
    QRect visibleRect(currentPos, visible);

    if (visibleRect.contains(posRect))
        return;

    QPoint newPos = currentPos;
    if (posRect.top() < visibleRect.top())
        newPos.setY(posRect.top());
    else if (posRect.bottom() > visibleRect.bottom())
        newPos.setY(posRect.bottom() - visible.height());
    if (posRect.left() < visibleRect.left())
        newPos.setX(posRect.left());
    else if (posRect.right() > visibleRect.right())
        newPos.setY(posRect.right() - visible.width());

    scrollTo(newPos);
}

/*
    Decomposes the position into a scroll and an overShoot part.
    Also keeps track of the current over-shooting value in overShootDist.
*/
void QAbstractKineticScrollerPrivate::setPositionHelper(const QPoint &pos)
{
    Q_Q(QAbstractKineticScroller);
    qKSDebug() << "setPosition to " << pos;

    QRect range = q->positionRange();

    QPoint clampedPos;
    clampedPos.setX(qBound(range.left(), pos.x(), range.right()));
    clampedPos.setY(qBound(range.top(), pos.y(), range.bottom()));

    int overShootX = (range.width() <= 1) ? 0 : clampedPos.x() - pos.x();
    int overShootY = (range.height() <= 1) ? 0 : clampedPos.y() - pos.y();

    QPoint oldOverShootDist = overShootDist;
    overShootDist.setX(qBound(-maxOverShoot.x(), overShootX, maxOverShoot.x()));
    overShootDist.setY(qBound(-maxOverShoot.y(), overShootY, maxOverShoot.y()));

    qKSDebug() << "setPosition to " << clampedPos << " overshoot: " << overShootDist << " (was: " << oldOverShootDist << ")";
    q->setPosition(clampedPos, overShootDist - oldOverShootDist);
}

/*!
    \enum QAbstractKineticScroller::Mode
    \since 4.6

    This enum contains the different modes for the QMaemo5KineticScroller.

    \value AutoMode The mode will allow pushing and AccelerationMode.

    \value PushMode The area will be scrolled as long as the user drags it around with pressed mouse button.

    \value AccelerationMode The area will continue scrolling after the user releases the mouse button.
*/


/*! \fn QRect QAbstractKineticScroller::positionRange() const

    Returns the rect with the lowest and highest valid position values.

    \sa scrollTo()
*/

/*! \fn QSize QAbstractKineticScroller::viewportSize() const

    Returns the size of the currently visible position values.
    In case of a QAbstractScrollArea this is equivalent to the viewport()
    size.

    \sa scrollTo()
*/

/*! \fn QPoint QAbstractKineticScroller::position() const

    \brief Returns the current position of the scrolling.
    Note that overshooting is not considered a real scrolling so the position might be (0,0)
    even if the user is currently dragging the widget over the "normal" positionRange.

    \sa positionRange()
*/


/*! \fn void QAbstractKineticScroller::setPosition(const QPoint &pos, const QPoint &overShootDelta)

    \brief set the scroll position of the widget.

    This function sets the scroll position of the widget to \a pos. This
    parameter will always be in the valid range returned by positionRange().
    In case over-shooting is required, the \a overShootDelta parameter will
    give the direction and pixel distance to over-shoot - relative to the
    value given in last setPositon() call.

    \sa positionRange()
*/

QT_END_NAMESPACE
