#include "spotifysession.h"
#include "callbacks.h"

/**
 *  spotifysession.cpp. Spotify session class. Handles the session functions.
 */
Spotify::Session* Spotify::Session::callingSession;

Spotify::Session::Session( QObject* parent ):
    QObject( parent ),
    m_strCachePath( "" ),
    m_strSettingsPath( "" ),
    m_currentTrack( 0 ),
    m_playlistcontainer( 0 ),
    m_currentUser( 0 ),
    m_userReqLogout ( false ),
    m_offlineMode( false ),
    m_offlineModeForced ( false ),
    m_connRules ( (sp_connection_rules)(SP_CONNECTION_RULE_ALLOW_SYNC_OVER_WIFI| SP_CONNECTION_RULE_NETWORK | SP_CONNECTION_RULE_NETWORK_IF_ROAMING) ),
    m_connState( SP_CONNECTION_STATE_UNDEFINED ),
    m_allPlaylistsLoaded( false ),
    m_pAudioThread ( 0 ),
    m_preBufActive ( true ),
    m_playbackPos( 0 ),
    m_bufferedAmount( 0 ),
    m_audioFragmentBuffer ( QByteArray () ),
    m_framesConsumed ( 0 ),
    m_pStarredPlaylist ( 0 )
{
    //Register meta types
    qRegisterMetaType<Spotify::Error>("Spotify::Error");
    qRegisterMetaType< QList<Spotify::Track*> >("QList<Spotify::Track*>");
    qRegisterMetaType<sp_track**>("sp_track**");
    qRegisterMetaType<qint32*>("qint32*");

    // C callbacks <-> signal/slot hack:
    Spotify::Session::callingSession = this;

    // Set up signal / slot connections:

    connect( this, SIGNAL(onNotifyMainThread()),
             this, SLOT(OnProcessThreadEvents()),Qt::QueuedConnection);

    connect(this, SIGNAL(rootListContainerLoaded(sp_playlistcontainer*)),
            this, SLOT(OnRootListContainerLoaded(sp_playlistcontainer*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onPlaylistAdded(sp_playlist*,qint32)),
            this, SLOT(OnPlayListAdded(sp_playlist*,qint32)),Qt::QueuedConnection);
    connect(this, SIGNAL(onPlaylistRemoved(sp_playlist*,qint32)),
            this, SLOT(OnPlayListRemoved(sp_playlist*,qint32)),Qt::QueuedConnection);
    connect(this, SIGNAL(onImageLoaded(sp_image*,void*)),
            this, SLOT(OnImageLoaded(sp_image*, void*)),Qt::QueuedConnection);
    connect(this, SIGNAL(metadataUpdated()),
                    this, SLOT(updateMetadata()),Qt::QueuedConnection);
    connect(this, SIGNAL(onMusicDelivery(int, const sp_audioformat*,QByteArray)),
                    this, SLOT(OnMusicDelivery(int, const sp_audioformat*,QByteArray)),Qt::QueuedConnection);
    connect(this, SIGNAL(onSearchComplete(sp_search*)),
                    this, SLOT(OnSearchComplete(sp_search*)),Qt::QueuedConnection);

    connect(this, SIGNAL(onAlbumBrowserReady(sp_albumbrowse*,qint32*)),
                    this, SLOT(OnAlbumBrowserReady(sp_albumbrowse*,qint32*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onArtistBrowseComplete(sp_artistbrowse*)),
                    this, SLOT(OnArtistBrowseComplete(sp_artistbrowse*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onOfflineStatusUpdated()),
                    this, SLOT(OnOfflineStatusUpdated()),Qt::QueuedConnection);
}

void Spotify::Session::setCachePath(const QString& strCachePath)
{//set the path for the libspotify cache data (directory will be created automatically if not exists upon session initialisation)
    m_strCachePath = strCachePath;
}

void Spotify::Session::setSettingsPath(const QString& strSettingsPath)
{//set the path for the libspotify settings data
    m_strSettingsPath = strSettingsPath;
}

QString Spotify::Session::getRememberedUser()
{
    //Returns remembered user (in case logged in via sp_session_relogin)
    char userID[200]; //allocate buffer to hold userID

    if(m_session)
    {
        qint32 ret = sp_session_remembered_user(m_session,userID,sizeof(userID));
        if(ret!=-1)
        {
            return QString::fromUtf8((const char*)userID,sizeof(userID));
        }
        else
        {
            qDebug() << "No stored credentials..";
            return QString();
        }
    }
    else
        qDebug() << "No valid session!";

    return QString();
}

QString Spotify::Session::getCurrentUser()
{
    //Returns the currently username for the currently logged-in user
    if(m_currentUser)
    {
        return m_currentUser->getDisplayName();
    }
    else
        return QString();
}

bool Spotify::Session::getOfflineTimeLeft(qint32& days, qint32& hours, qint32& minutes)
{
    //Returns time until the offline keystore expires and user is required to go online (days, hours, and minutes
    //left as ref. params). Returns true on success; false on error.

    qint32 secsLeft;

    if(m_session)
    {
        secsLeft = sp_offline_time_left(m_session);
        if(secsLeft>0)
        {
            days = secsLeft / 86400;
            hours = (secsLeft / 3600) - days * 24;
            minutes = (secsLeft / 60) - (days * 1440) - (hours * 60);
            return true;
        }
    }

    return false;
}

Spotify::Playlist* Spotify::Session::getStarredPlaylist()
{
    //Returns (pointer) to the playlist containing user's starred tracks (or NULL in case error has occured in allocating the playlist).
    return m_pStarredPlaylist;
}

void Spotify::Session::init()
{
    //Init the current Spotify session.
    //Note: this method NEEDS to be initially called to set up the session.

    QByteArray cachePath = m_strCachePath.toLatin1();
    QByteArray settingsPath = m_strSettingsPath.toLatin1();

    //make sure that necessary settings / cache directories exist

    QDir cacheDir(m_strCachePath);
    QDir settingsDir(m_strSettingsPath);

    if(!cacheDir.exists())
    {
        if(!cacheDir.mkpath(m_strCachePath))
        {
            qDebug() << "Failed to create application cache directory " << m_strCachePath << ", exiting..";
            qApp->quit();
        }
    }

    if(!settingsDir.exists())
    {
        if(!settingsDir.mkpath(m_strSettingsPath))
        {
            qDebug() << "Failed to create application settings directory " << m_strSettingsPath << ", exiting..";
            qApp->quit();
        }
    }

    // Set config parameters
    m_config.api_version = SPOTIFY_API_VERSION;
    m_config.cache_location = cachePath.data();
    m_config.settings_location = settingsPath.data();
    m_config.application_key = CredManager::getSpotifyAPIKey();
    m_config.application_key_size = CredManager::getSpotifyAPIKeySize();
    m_config.device_id = "N900";
    m_config.user_agent = "QSpot";
    m_config.tracefile = NULL;
    m_config.callbacks = &g_callbacks;

    // Initiate  / allocate new session
    m_error = sp_session_create( &m_config, &m_session );
    if( m_error.isError() )
    {
        qDebug() << "FATAL ERROR: Could not initiate session. Error reason: " << m_error.toString();
        qApp->quit();
    }

    m_pConnMan = new ConnectionManager(this); //allocate connection manager instance
    connect(m_pConnMan,SIGNAL(onlineStateChanged(bool)),SLOT(OnOnlineStateChanged(bool)));

    setNetworkState(); //sets connection type / rules

    m_ProcessEvents = false; //event processing not initiated
    m_pProcEvtTimer = new QTimer(this);
    connect(m_pProcEvtTimer,SIGNAL(timeout()),this,SLOT(OnProcessThreadEvents()));
}

bool Spotify::Session::isOfflineMode()
{
    //returns true in case logged in and offline mode is active ; false in other cases (e.g., connected against Spotify access point)
  return m_offlineMode;
}

void Spotify::Session::setNetworkState()
{
    //sets connection type / rule
    sp_connection_type connType;

    if(!m_pConnMan->currOnlineState()) //no network connectivity, set (force) offline mode
    {
       setConnectionType(SP_CONNECTION_TYPE_NONE); //no connection (offline)
       setOfflineMode(true,false);
    }
    else
    { //we have online connectivity
      //check / set type of online interface (e.g., WLAN / 3G)
       QNetworkConfiguration::BearerType type = m_pConnMan->bearerType();
       if(type==QNetworkConfiguration::BearerWLAN) //wifi connection type
           connType = SP_CONNECTION_TYPE_WIFI;
       else if(type==QNetworkConfiguration::Bearer2G||type==QNetworkConfiguration::BearerCDMA2000||type==QNetworkConfiguration::BearerWCDMA||
               type==QNetworkConfiguration::BearerHSPA)
       { //mobile connection type
           if(m_pConnMan->roamingSupported())
               connType = SP_CONNECTION_TYPE_MOBILE_ROAMING; //current bearer supports roaming
           else
               connType = SP_CONNECTION_TYPE_MOBILE; //mobile, no roaming
       }
       else
       {
           //unknown bearer type
           connType = SP_CONNECTION_TYPE_UNKNOWN;
       }

       setConnectionType(connType); //set current connection type
       setOfflineMode(false,false);
    }
}

void Spotify::Session::setPreferredBitrate(const sp_bitrate& bitrate)
{
    //set preferred audio streaming bitrate
     sp_session_preferred_bitrate(m_session, bitrate);
}

void Spotify::Session::setPreferredSyncBitrate(const sp_bitrate& bitrate, bool resyncTracks)
{
    //set preferred offline synchronisation bitrate (resync existing tracks in case resyncTracks is true).
    sp_session_preferred_offline_bitrate(m_session,bitrate,resyncTracks);
}

void Spotify::Session::initAudio()
{
    //Init the audio / playback thread (pulseaudio).
    //Note: Must be called before calling any Spotify session player functions.

    m_pAudioThread = new PlaybackThread(this);

    m_pAudioThread->setBufferingPaused(false);
    m_pAudioThread->setPlaybackPaused(false);

    connect(m_pAudioThread, SIGNAL(resumeBuffering()),
                     this, SLOT(OnResumeBuffering()),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playbackPositionUpdated(qint64)),
                     this, SLOT(OnPlaybackPositionUpdated(qint64)),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playbackFinished()),
                     this, SLOT(OnPlaybackFinished()),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playing(bool)),
                     this, SLOT(OnPlayingState(bool)),Qt::QueuedConnection);

    m_pAudioThread->start(); //start audio playback thread
}

void Spotify::Session::setMinBufferSize(qint32 secs)
{
    //set the (minimum) size of the internal FIFO used to hold audio data
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setMinBufferSize(secs);
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::setPreBufferingActive(bool active)
{
    //set whether track pre-buffering is active (i.e., buffer next track data in advance)

    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_preBufActive = active;
        m_pAudioThread->setPreBufActive(active);
        m_pAudioThread->getMutex()->unlock();
    }
}

bool Spotify::Session::getPreBufferingActive()
{
    return m_preBufActive;
}

void Spotify::Session::setPlaybackPos(qint64 pos)
{
    m_playbackPos = pos;
}

qint64 Spotify::Session::getPlaybackPos()
{
    return m_playbackPos;
}

void Spotify::Session::setBufferingPos(qint32 pos)
{
    m_bufferedAmount = pos;
}

qint32 Spotify::Session::getBufferingPos()
{
    return m_bufferedAmount;
}


void Spotify::Session::setPlaying(bool playing)
{
    m_buffering = playing;
}

void Spotify::Session::setNextTrackStartPos()
{
    //set the next track start position in the audio FIFO
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setNextTrackStartPos();
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::setNewTrackStarted(bool started)
{
    //indicate start of new track audio buffer data (in playback thread FIFO)
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setNewTrackStarted(started);
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::setConnectionRule(sp_connection_rules rule, bool state)
{ // Set the connection rule specified to given state (on / off)

    qint32 activeRules = (qint32)m_connRules;

    if (state)
        activeRules |= rule;
    else
        activeRules &= ~rule;

    if(activeRules!=(qint32)m_connRules)
    {
        m_connRules = (sp_connection_rules)activeRules;
        sp_session_set_connection_rules(m_session,m_connRules); //set conection rules (if changed)
    }
}

 void Spotify::Session::setConnectionType(sp_connection_type type)
 { //Set the current type of connection (e.g., wifi / 3G etc.)
     sp_session_set_connection_type(m_session, type);
 }

 void Spotify::Session::setOfflineMode(bool offline, bool userRequested)
 {
     //Toggle between offline / online playback mode (according to offline param). In case state change is requested by user,
     //userRequested parameter is set to true; otherwise false (requested by system i.e., due to changed online connectivity state).

     qDebug() << "SetOfflineMode: " << offline << "," << userRequested;

     bool bStatus = true;

     if(m_offlineMode!=offline)
     {
         if(m_offlineModeForced && !offline && !userRequested)
             return;

         if(!offline)
         { //online mode requested; establish network connection in case we are not already connected
             bStatus = m_pConnMan->attemptConnect();
         }
         else
         { //activate offline mode
             if(m_currentTrack)
             {
                 if(!m_currentTrack->isOffline())
                 {
                     stop(); //stop playback / buffering of current track in case it is not available offline
                     qApp->removePostedEvents(this);
                 }
             }

         }

         if(bStatus)
         {
             m_offlineMode = offline; //set new offline mode state
             if(m_offlineMode && userRequested)
                 m_offlineModeForced = true;
             else
                 m_offlineModeForced = false;

             //allow / disallow network traffic depending on new online state
             setConnectionRule(SP_CONNECTION_RULE_NETWORK, m_offlineMode);
             setConnectionRule(SP_CONNECTION_RULE_NETWORK, !m_offlineMode);
             emit offlineModeStateChanged(m_offlineMode); //notify about state change
         }
     }
 }

void Spotify::Session::OnPlayingState(bool state)
{
    //emit current playback state (playing / paused)
    emit playing(state);
}

void Spotify::Session::OnProcessThreadEvents()
{
    qint32 next_timeout = -1;
    
    do
    {
        if(m_session)
        {
            sp_session_process_events(m_session,&next_timeout);

            updateConnectionState(); //update connection state (in the case it has changed)

            if (m_offlineMode && m_connState == SP_CONNECTION_STATE_LOGGED_IN)
            {
              setConnectionRule(SP_CONNECTION_RULE_NETWORK, true);
              setConnectionRule(SP_CONNECTION_RULE_NETWORK, false);
            }
        }
    }
    while(!next_timeout);

    if(m_ProcessEvents)
    {
        //QTimer::singleShot(next_timeout,this,SLOT(OnProcessThreadEvents()));
        m_pProcEvtTimer->start(next_timeout);
    }
}

void Spotify::Session::login( QString username, QString password, bool storeCredentials )
{
    // Request to login to Spotify with supplied credentials (remember login credentials in case storeCredentials set)
    sp_session_login( m_session, username.toUtf8(), password.toUtf8(), storeCredentials);
}

Spotify::Error Spotify::Session::relogin()
{
    //Rre-logins remembered user to Spotify.
    //Returns SP_ERROR_OK on success; otherwise error code is set (i.e., no remembered credentials).

    m_buffering = false; //no buffering as we are not logged in
    Spotify::Error err = sp_session_relogin(m_session);
    return err;
}

void Spotify::Session::forgetCurrentUser()
{
    //forget the currently logged user (in case user's credentials have been stored by libspotify)
    sp_session_forget_me( m_session ); //clear any stored credentials
}

void Spotify::Session::logout()
{
    //Logs out the current user from the Spotify service.
    if(m_currentUser)
    {
        delete m_currentUser;
        m_currentUser = NULL;
    }
    m_userReqLogout = true; //indicate that user has request logout (signal will be emitted upon receiving loggedout callback)
    sp_session_logout( m_session );
}

Spotify::Error Spotify::Session::load( Spotify::Track* track )
{
    Spotify::Error error;
    sp_track* sptrack = track->getTrack();
    if( sptrack )
    {
        error = sp_session_player_load( m_session, sptrack );
    }
    return error;
}

void Spotify::Session::OnPlaybackFinished()
{
    emit playbackFinished();
}

void Spotify::Session::OnPlaybackPositionUpdated(qint64 duration)
{ //the current playback position has been updated (signaled by audio thread)
    m_playbackPos += duration;
    emit playbackPositionUpdated(m_playbackPos);
}

void Spotify::Session::resetPlaybackMembers()
{
    //public method for resetting the playback related instance members (and removes any pending events from the playback thread)

    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setPlaybackPaused(true);
        m_pAudioThread->flushPAbuffer();
        m_pAudioThread->clearFIFO();
        m_pAudioThread->setFIFOSize(0);
        m_pAudioThread->setBufferingPaused(false);
        m_pAudioThread->setPlaybackPaused(false);
        m_pAudioThread->setEndOfTrack(false);
        qApp->removePostedEvents(this); //remove any pending events for this instance (from e.g., the playback thread)
        m_pAudioThread->getMutex()->unlock();
        m_audioFragmentBuffer = QByteArray();
        m_framesConsumed = 0;
    }
}


Spotify::Error Spotify::Session::play( Spotify::Track* track, bool preBuffer )
{
    m_buffering = false;
    Spotify::Error error;


    // Disregard tracks that are not available:
    if( !track->isAvailable() )
    {
        qDebug() << "ERROR: Asked to play back a unavailable track..";
        return Spotify::Error( SP_ERROR_TRACK_NOT_PLAYABLE );
    }


    // Disregard unloaded of unavailable tracks:
    if( !track->isLoaded() )
    {
        qDebug() << "ERROR: Asked to play back an unloaded track..";
        return Spotify::Error( SP_ERROR_TRACK_NOT_PLAYABLE );
    }

    // Load track:
    error = load( track );
    if( error.isError() )
    {
        qDebug() << "ERROR: Could not load track: " << error.toString();
        return error;
    }

    m_currentTrack = track;//assign current track

    if(!preBuffer) //pre-buffering not active (i.e., audio FIFO should be cleared)
    {
        if(m_pAudioThread)
        {
            m_pAudioThread->getMutex()->lock();
            m_pAudioThread->clearFIFO();
            m_pAudioThread->setFIFOSize(0);
            m_pAudioThread->setBufferingPaused(false);
            m_pAudioThread->setPlaybackPaused(false);
            m_pAudioThread->setEndOfTrack(false);
            m_pAudioThread->setNewTrackStarted(false);
            m_pAudioThread->setNextTrackStartPos(-1);
            m_pAudioThread->getMutex()->unlock();
        }

        m_playbackPos = 0;
        m_bufferedAmount = 0;
    }

    // Play track:
    sp_session_player_play( m_session, true );
    m_buffering = true;

    return Spotify::Error( SP_ERROR_OK );
}

void Spotify::Session::seek( qint32 pos )
{
    //Seek to specified position (pos) within the currenly loaded track. Pos is specified in msecs.

    m_playbackPos = pos;
    m_bufferedAmount = pos;
    m_pAudioThread->getMutex()->lock();
    m_pAudioThread->clearFIFO();
    m_pAudioThread->flushPAbuffer();
    m_pAudioThread->setFIFOSize(0);
    qApp->removePostedEvents(this); //remove any pending events for this instance (from e.g., the playback thread)
    m_pAudioThread->getMutex()->unlock();

    sp_session_player_seek( m_session, pos ); //seek to acutal requested position
 }

void Spotify::Session::stop()
{
    //stop buffering and playback
    if(m_currentTrack)
    {
        pauseResumePlayback();
        if( m_buffering )
        {
            sp_session_player_play( m_session, false );
            m_buffering = false;
        }
        sp_session_player_unload( m_session ); //un-load current track
        m_currentTrack = NULL;
    }
}

void Spotify::Session::pauseResumePlayback()
{
    //Pause / resume playback from Spotify (e.g., signal to playback thread to stop / continue writing data to PulseAudio)
    m_pAudioThread->getMutex()->lock();
    if(m_pAudioThread->getPlaybackPaused())
       m_pAudioThread->setPlaybackPaused(false);
    else
        m_pAudioThread->setPlaybackPaused(true);
    m_pAudioThread->getMutex()->unlock();
}

void Spotify::Session::pauseResumeBuffering()
{
    //Pause / resume  music delivery (i.e., buffering) from  Spotify

    //playing (buffering) music data
    if( m_buffering )
    { //pause buffering
        sp_session_player_play( m_session, false );
        m_buffering = false;
    }
    // buffering currently paused
    else if( !m_buffering)
    {
        if(m_currentTrack)
        {
            if(m_currentTrack->getTrack())
            {
                if(sp_track_is_loaded( m_currentTrack->getTrack()))
                {
                    //resume buffering
                    sp_session_player_play( m_session, true );
                    m_buffering = true;
                }
            }
        }
    }
}

Spotify::Error Spotify::Session::search( const QString& query, const int trackOffset, const int trackCount,
                                         const int albumOffset, const int albumCount, const int artistOffset,
                                         const int artistCount, void* userdata )
{
    Spotify::Error error;
    //sp_search_type type;

    //if(suggestedSearch)
      //  type=SP_SEARCH_SUGGEST;
    //else
      //  type=SP_SEARCH_STANDARD;

    sp_search* search = sp_search_create( m_session, query.toUtf8().data(), trackOffset, trackCount,
                                          albumOffset, albumCount, artistOffset, artistCount, &searchCompleteCallback, userdata );

    Spotify::Search* s = new Spotify::Search(m_session, search, this );
    m_searchHash.insert( search, s );
    connect( s, SIGNAL(destroyed(QObject*)),
             this, SLOT(searchDeletedSlot(QObject*)));
    return error;
}

bool Spotify::Session::loadImage(const byte* imageID, Spotify::ImageContainer* pContainer)
{
    //request to load image (identified by imageID) from the Spotify backend.

    Spotify::Error error;
    if( imageID && pContainer )
    {
        sp_image* image = sp_image_create( m_session, imageID );
        if(image)
            sp_image_add_load_callback( image, &Spotify::Session::imageLoadedCallback, (void*)pContainer );
        return true;
    }

    return false;
}

void Spotify::Session::updateConnectionState()
{
    //update the current session status

    if(m_connState !=  sp_session_connectionstate(m_session))
    { //connection state has changed
        m_connState = sp_session_connectionstate(m_session);
        qDebug() << "Connection state: " << m_connState;
    }
}

bool Spotify::Session::addPlaylist( QString name )
{
    if(!m_playlistcontainer) return false;
    sp_playlist* p = sp_playlistcontainer_add_new_playlist( m_playlistcontainer, name.toUtf8().data() );
    return p!=0;
}

bool Spotify::Session::addPlaylist( Spotify::Link link )
{
    if(!m_playlistcontainer) return false;
    if( link.type() == Spotify::Link::PLAYLIST )
    {
        sp_link* l = link.getLink();
        if( l )
        {
            sp_playlist* p = sp_playlistcontainer_add_playlist( m_playlistcontainer, l );
            return p!=0;
        }
    }
    return false;
}

bool Spotify::Session::removePlaylist( const int index )
{
    if( !m_playlistcontainer) return false;
    Spotify::Error error = sp_playlistcontainer_remove_playlist( m_playlistcontainer, index );
    return !error.isError();
}

bool Spotify::Session::movePlaylist( const int index, int position )
{
    //if( !m_playlistcontainer) return false;
    //if( position > index ) position+=1; // libspotify wants from n to n+2 rather than n to n+1.....
    //Spotify::Error error = sp_playlistcontainer_move_playlist( m_playlistcontainer, index, position );
    return false;
}

void Spotify::Session::updateMetadata()
{
    qDebug() << "metadata updated..";

    bool loadedFlag = true; //false in case some playlist not loaded

    if(!m_playlistcontainer)
        m_playlistcontainer = sp_session_playlistcontainer(m_session); //get playlist root container

    if(m_playlistList.size()==0)
    { //initially allocate all playlist containers (and attempt to load associated metadata)
        if(m_playlistcontainer)
        {
            //add initially empty playlist container instances
            for(qint32 pos = 0; pos < sp_playlistcontainer_num_playlists( m_playlistcontainer ); pos++)
            {
                Spotify::Playlist* playlist = new Spotify::Playlist(m_session, PLAYLIST_TYPE_USER, sp_playlistcontainer_playlist(m_playlistcontainer, pos), this);
                m_playlistList.insert(pos, playlist); //insert container instance into list
                m_playlistHash.insert(playlist->getPlaylist(),playlist);
            }
        }
    }

    //attempt to update playlist metadata
    if(!m_allPlaylistsLoaded) //check current loaded state for playlists
    {
        for( qint32 i = 0; i < m_playlistList.count(); ++i )
        {
            Spotify::Playlist* playlist = m_playlistList.at(i);
            if( playlist )
            {
                loadedFlag = playlist->load(); //metadata processed for playlist (
                playlist->updateTracks(); //try to load tracks
            }
        }

        if(loadedFlag && !m_allPlaylistsLoaded)
        {
            m_allPlaylistsLoaded = true;
            emit playlistContainersReady(m_playlistList); //all playlists have been loaded; emit ready signal
        }
    }

    // Try to load searches:
    QList<Spotify::Search*> searches = m_searchHash.values();
    for( qint32 i = 0; i < searches.count(); ++i )
    {
        if( !searches.at(i)->isLoaded() )
            searches.at(i)->load();
    }

    // Try to load artist catalogues:
    QList<Spotify::ArtistCatalogue*> artistCatalogues = m_artistBrowseHash.values();
    for( qint32 i = 0; i < artistCatalogues.count(); ++i )
    {
        if( !artistCatalogues.at(i)->isLoaded() )
            artistCatalogues.at(i)->load();
    }
}

void Spotify::Session::searchDeletedSlot( QObject* object )
{
    Spotify::Search* search = static_cast< Spotify::Search* >( object );
    if( search )
        m_searchHash.remove( search->getSearch() );
    else
        qDebug() << "WARNING: Trying to delete non-existing search";
}

bool Spotify::Session::browse( Spotify::Artist* artist )
{
    if( artist )
    {
        sp_artist* spartist = artist->getArtist();
        sp_artistbrowse* ab = sp_artistbrowse_create( m_session, spartist,SP_ARTISTBROWSE_NO_TRACKS,
                                                      &Spotify::Session::artistBrowseCompleteCallback,
                                                      NULL);
        if(!ab)
           return false;
        Spotify::ArtistCatalogue* ac = new Spotify::ArtistCatalogue( ab, this );
        m_artistBrowseHash.insert( ab, ac );
        connect( ac, SIGNAL(destroyed(QObject*)),
                 this, SLOT(artistCatalogueDeletedSlot(QObject*)));
    }
    else
        return false;

    return true;
}

void Spotify::Session::artistCatalogueDeletedSlot( QObject* object )
{
    qDebug() << "An artistcatalogue was deleted";
    Spotify::ArtistCatalogue* ac = static_cast< Spotify::ArtistCatalogue* >( object );
    if( ac )
        m_artistBrowseHash.remove( ac->getArtistBrowse() );
    else
        qDebug() << "WARNING: Trying to delete non-existing artist browser";
}

bool Spotify::Session::browse( Spotify::Album* album, qint32* pCallerID )
{
    if( album )
    {
        sp_album* spalbum = album->getAlbum();
        sp_albumbrowse* ab = sp_albumbrowse_create( m_session, spalbum,
                                                    &Spotify::Session::albumBrowseCompleteCallback,
                                                    (void*)pCallerID);
        if(!ab)
            return false; //failed to create browse album request

        //error = sp_albumbrowse_error( ab );
        Spotify::AlbumBrowser* alb = new Spotify::AlbumBrowser(m_session, ab, this );
        m_albumBrowseHash.insert( ab, alb);
        connect( alb, SIGNAL(destroyed(QObject*)),
                 this, SLOT(albumBrowserDeletedSlot(QObject*)));
    }
    else
        return false;

    return true;
}

void Spotify::Session::albumBrowserDeletedSlot( QObject* object )
{
    qDebug() << "An album browser was deleted";
    Spotify::AlbumBrowser* ab = static_cast< Spotify::AlbumBrowser* >( object );
    if( ab )
        m_albumBrowseHash.remove( ab->getAlbumBrowser() );
    else
        qDebug() << "WARNING: Trying to delete non-existing album browser";
}

Spotify::Session::~Session()
{ //free allocated resources
    m_ProcessEvents = false;
    if(m_currentUser)
        m_currentUser->deleteLater();
    if(m_pAudioThread)
    {//stop the audio playback thread in a controlled manner
        m_pAudioThread->stop();
        m_pAudioThread->wait(); //wait for thread execution to finish
    }
    sp_session_release(m_session); //free current session
    qDebug() << "Session deleted";
}


/********************* CALLBACKS *****************/

void Spotify::Session::artistBrowseCompleteCallback( sp_artistbrowse* ab, void* userdata )
{
    Spotify::Session::callingSession->artistBrowseCompleteWrapper( ab, userdata );
}

void Spotify::Session::artistBrowseCompleteWrapper( sp_artistbrowse* ab, void* userdata )
{
    Q_UNUSED(userdata);
    emit onArtistBrowseComplete(ab);
}

void Spotify::Session::OnArtistBrowseComplete( sp_artistbrowse* ab )
{
    if( ab && m_artistBrowseHash.contains( ab ) )
    {
        qDebug() << "An artist was just browsed!";
        Spotify::ArtistCatalogue* ac = m_artistBrowseHash.value( ab );
        emit artistCatalogueReady( ac );
    }
}

void Spotify::Session::albumBrowseCompleteCallback( sp_albumbrowse* ab, void* userdata )
{
    Spotify::Session::callingSession->albumBrowseCompleteWrapper( ab, userdata );
}

void Spotify::Session::albumBrowseCompleteWrapper( sp_albumbrowse* ab, void* userdata )
{
    qint32* pCallerID = (qint32*)userdata;
    emit onAlbumBrowserReady(ab, pCallerID);
}

void Spotify::Session::OnAlbumBrowserReady(sp_albumbrowse *ab, qint32* pCallerID)
{
    qint32 callerID = -1;

    if(pCallerID)
    {
        callerID = *(qint32*)pCallerID;
    }
    if( ab && m_albumBrowseHash.contains( ab ) )
    {
        qDebug() << "Album browse requeste completed!";
        Spotify::AlbumBrowser* alb = m_albumBrowseHash.value( ab );
        emit albumBrowserReady( alb, callerID);
    }
}

void SP_CALLCONV Spotify::Session::playlistAddedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                                     int position, void* userdata )
{ //playlist added to container pc
    Spotify::Session::callingSession->playlistAddedWrapper( pc, playlist, position, userdata);
}

void Spotify::Session::playlistAddedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist, int position, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( userdata );
    emit onPlaylistAdded(playlist,position);
}

void Spotify::Session::OnPlayListAdded(sp_playlist* playlist, qint32 position)
{
    //Playlist has been added to root playlist container

   qDebug()<< "Playlist added";
   qint32 insertPos;

   if(!m_playlistHash.contains(playlist))
   {//playlist not already added
       Spotify::Playlist* pl = new Spotify::Playlist(m_session, PLAYLIST_TYPE_USER, playlist);
       pl->setParent( this );
       pl->setPosition( position );
       m_playlistHash.insert( playlist, pl );
       m_playlistList.insert( position, pl ); //insert playlist at its position
       emit playlistAdded( pl, position ); //signal that playlist intance has been added
   }
}

void SP_CALLCONV Spotify::Session::playlistRemovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                     int position, void* userdata )
{ //playlist removed from container pc
    Spotify::Session::callingSession->playlistRemovedWrapper( pc, playlist, position, userdata );
}

void Spotify::Session::playlistRemovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int position, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( userdata );

    emit onPlaylistRemoved(playlist, position);
}

void Spotify::Session::OnPlayListRemoved(sp_playlist* playlist, qint32 position)
{
    qDebug() << "A playlist was removed";

    if( m_playlistHash.contains( playlist ) )
    {
        Spotify::Playlist* pl = m_playlistHash.take( playlist );
        if( pl && m_playlistList.contains( pl ) && position < m_playlistList.count() )
        {
            m_playlistList.takeAt( position );
            emit playlistRemoved( position );
        }
    }
}

void SP_CALLCONV Spotify::Session::playlistMovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                              int oldPosition, int newPosition, void* userdata )
{
   // Spotify::Session::callingSession->playlistMovedWrapper( pc, playlist, oldPosition, newPosition, userdata );
}

void Spotify::Session::playlistMovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                             int oldPosition, int newPosition, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( playlist );
    Q_UNUSED( userdata );
    qDebug() << "playlist moved..";
    // libspotify workaround:
    if( newPosition > oldPosition ) newPosition -= 1;
    m_playlistList.move( oldPosition, newPosition );
    emit playlistMoved( oldPosition, newPosition );
    qDebug() << "A playlist was moved.";
}

void Spotify::Session::playListContainerLoadedCallback(sp_playlistcontainer *pc, void* userdata)
{
    qDebug()<<"Container loaded callback..";
   Spotify::Session::callingSession->playListContainerLoadedWrapper(pc, userdata);
}

void Spotify::Session::playListContainerLoadedWrapper(sp_playlistcontainer *pc, void* userdata)
{
   Q_UNUSED( userdata );
   qDebug()<< "Playlist container loaded";

   emit rootListContainerLoaded(pc);
}

void Spotify::Session::OnRootListContainerLoaded(sp_playlistcontainer* pc)
{
     qDebug() << "Number of playlists:" << sp_playlistcontainer_num_playlists( pc );

     m_playlistcontainer = pc;

     m_playlistList.clear(); //clear list of playlists
     //add initially empty playlist container instances
     for(qint32 pos = 0; pos < sp_playlistcontainer_num_playlists( pc ); pos++)
        {
         Spotify::Playlist* playlist = new Spotify::Playlist(m_session, PLAYLIST_TYPE_USER, sp_playlistcontainer_playlist(pc, pos), this);
		 m_playlistList.insert(pos, playlist); //insert container instance into list
	 }

}

void Spotify::Session::imageLoadedCallback( sp_image* image, void* userdata )
{
    Spotify::Session::callingSession->imageLoadedWrapper( image, userdata );
}

void Spotify::Session::imageLoadedWrapper( sp_image* image, void* userdata )
{
    emit onImageLoaded(image, userdata);
}

void Spotify::Session::OnImageLoaded(sp_image* image, void* userdata)
{

    if( !image || !sp_image_is_loaded( image ) )
    {
        qDebug() << "Image not loaded..";
        return;
    }

    QString format = "Unknown";
    uchar* data = NULL;
    sp_imageformat spformat = sp_image_format(image);
    size_t len = 0;

    // Get image format:
    switch (spformat)
    {
    case SP_IMAGE_FORMAT_JPEG:
        format = "JPEG";
        break;
    case SP_IMAGE_FORMAT_UNKNOWN:
        format = "UNKNOWN";
        break;
    default:
        format = "UNKNOWN";
        break;
    }

    if( format == "UNKNOWN" )
    {
        qDebug()<< "Unknown format!";
        return;
    }

    // Get actual image data
    data = (uchar*)sp_image_data(image,&len);

    // Create QImage instance, for wrapping the image data

    QImage img;

    if(!img.loadFromData(data, (int)len, format.toAscii()))
    {
        qDebug() << "Loading of image data failed!";
    }

    if(userdata)
    { //assign image to associated container
        Spotify::ImageContainer* pContainer = (Spotify::ImageContainer*)userdata;
        pContainer->setImage(img);
    }

    // Release sp_image
    sp_image_release(image);

    //emit coverArtReady(img, callerID);
}

void Spotify::Session::searchCompleteCallback( sp_search* search, void* userdata )
{
    Spotify::Session::callingSession->searchCompleteWrapper( search, userdata );
}

void Spotify::Session::searchCompleteWrapper( sp_search* search, void* userdata )
{
    Q_UNUSED( userdata )

    emit onSearchComplete(search);
}

void Spotify::Session::OnSearchComplete(sp_search* search)
{
      Spotify::Error error;
      error = sp_search_error( search );

     if(search && m_searchHash.contains( search ) )
     {
         Spotify::Search* s = m_searchHash.value( search );
         s->load(); //try to load search results
         emit searchComplete( s );
     }
     else
         qDebug() << "Error performing search: " << error.toString();
}

void Spotify::Session::loggedInCallback( sp_session* session, sp_error error )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->loggedInWrapper( Spotify::Error( error ) );
}

void Spotify::Session::loggedInWrapper( Spotify::Error error )
{
    // Let application know we're logged in (or not)
    if(error.isError())
    {
        qDebug() << "Logged in with error: " << error.toString();
    }
    else
        qDebug() << "Logged in OK..";

    if(!m_currentUser)
    {
        m_currentUser = new Spotify::User( sp_session_user( m_session ) );
        if(m_currentUser->isLoaded())
            qDebug() << "User loaded OK..";
    }
    if(!m_playlistcontainer && !error.isError())
    {
        qDebug() << "Registering container callbacks..";
        m_playlistcontainer = sp_session_playlistcontainer(m_session); //get root playlist container
        sp_playlistcontainer_add_callbacks(m_playlistcontainer,&pc_callbacks,NULL); //register callbacks
        qDebug() << "Container callbacks registered..";
        //allocate the playlist containing currenly logged-in user's starred tracks
        sp_playlist* starred_pl = sp_session_starred_create(m_session);
        if(starred_pl)
        {
            m_pStarredPlaylist = new Spotify::Playlist(m_session, PLAYLIST_TYPE_STARRED, starred_pl, this);
            if(m_pStarredPlaylist)
            {
                m_pStarredPlaylist->setHasRef(true); //corresponding sp_playlist needs to be explicitly released
            }
        }
    }

    emit loggedIn( error );
}

void Spotify::Session::loggedOutCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->loggedOutWrapper();
}

void Spotify::Session::loggedOutWrapper()
{
    qDebug() << "Logged out";
    if(m_userReqLogout)
    {
        m_userReqLogout = false;
        emit loggedOut();
    }
}


void Spotify::Session::metadataUpdatedCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->metadataUpdatedWrapper();
}

void Spotify::Session::metadataUpdatedWrapper()
{
    emit metadataUpdated();
}

void Spotify::Session::connectionErrorCallback( sp_session* session, sp_error error )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->connectionErrorWrapper( Spotify::Error( error ) );
}

void Spotify::Session::connectionErrorWrapper( Spotify::Error error )
{
    Q_UNUSED( error );

    qDebug() << "Connection error: " << error.toString();
}

void Spotify::Session::messageToUserCallback( sp_session* session, const char* message )
{
    Q_UNUSED( session );
   // Spotify::Session::callingSession->messageToUserWrapper( QString( message ) );
}

void Spotify::Session::messageToUserWrapper( QString message )
{
    //qDebug() << "MESSAGE (user): " << message.trimmed();
    //emit messageToUser( message.trimmed() );
}

void Spotify::Session::notifyMainThreadCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->notifyMainThreadWrapper();
}

void Spotify::Session::notifyMainThreadWrapper()
{
    //notification that there event processing needs to be done in libspotify.

   if(!m_ProcessEvents)
       m_ProcessEvents = true;

   emit onNotifyMainThread();
}

int Spotify::Session::musicDeliveryCallback( sp_session* session, const sp_audioformat* format,
                                             const void* frames, int num_frames )
{ //Push-callback delivering audio data from Spotify backend
    Q_UNUSED( session );
    return Spotify::Session::callingSession->musicDeliveryWrapper( format, frames, num_frames );
}

int Spotify::Session::musicDeliveryWrapper( const sp_audioformat* format, const void* frames, int num_frames )
{

    // No data or invalid format/frames:
    if (num_frames == 0 || !frames || !format || !m_buffering)
        return 0;

    m_audioFragmentBuffer += QByteArray((const char*)frames,num_frames * sizeof(int16_t) * format->channels); //audio data appended to temp. buffer until enough samples are available
    m_framesConsumed += num_frames;

    if(m_framesConsumed>=SAMPLE_RATE)
    { //one sec. of audio buffered; append to playback thread FIFO
        emit onMusicDelivery(m_framesConsumed,format,m_audioFragmentBuffer);
        m_audioFragmentBuffer = QByteArray();
        m_framesConsumed = 0;
    }

    return num_frames; //return the number of frames that have been buffered
}

void Spotify::Session::OnMusicDelivery(int num_frames, const sp_audioformat* format, QByteArray buffer)
{
    //Audio data buffer received, add buffered data at end of audio playback queue

    m_pAudioThread->getMutex()->lock();

    if(m_preBufActive)
    {
        if(m_pAudioThread->getNewTrackStarted())
        {
            qDebug() << "Pre-buffering active; next track started...";
            m_pAudioThread->setNewTrackStarted(false); //reset track started state
            m_pAudioThread->setNextTrackStartPos();
            m_bufferedAmount = 0; //reset buffering amount (newly loaded track)
        }
    }

    double tmpAmount = (double)((double)num_frames * 1000.00 / (double)SAMPLE_RATE);
    m_bufferedAmount += (quint32)(tmpAmount + 0.5); //increase the currently buffered amount
    emit trackBufferingAmountUpdated( m_bufferedAmount );

    m_pAudioThread->FIFOAppend(buffer);
    m_pAudioThread->incFIFOSize(buffer.size());
    if(m_pAudioThread->getFIFOSize()>=(m_pAudioThread->getMinBufferSize()+5)*SAMPLE_RATE*sizeof(int16_t)*format->channels)
    {
        m_pAudioThread->setBufferingPaused(true);
        m_pAudioThread->getMutex()->unlock();
        pauseResumeBuffering();
    }
    else
    {
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::OnResumeBuffering()
{
    //signal from playback thread that more buffered data is needed
    pauseResumeBuffering();
}

void Spotify::Session::playTokenLostCallback( sp_session* session )
{
    Spotify::Session::callingSession->playTokenLostWrapper( session );
}

void Spotify::Session::playTokenLostWrapper( sp_session* session )
{
    //emit signal when playback token has been lost (e.g., same account logs in somewhere else)
    Q_UNUSED( session );
    emit playTokenLost();
    m_buffering = false;
}

void Spotify::Session::logMessageCallback( sp_session* session, const char* message )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->logMessageWrapper( QString( message ) );
}

void Spotify::Session::logMessageWrapper( QString message )
{
    qDebug() << "MESSAGE (log):  " << message.trimmed();
}

void Spotify::Session::endOfTrackCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->endOfTrackWrapper();
}

void Spotify::Session::endOfTrackWrapper()
{
    qDebug()<<"End of track callback..";
    if(m_currentTrack)
    { //unload current track
        // sp_session_player_unload(m_session);
        //m_currentTrack = NULL;
    }
    if(!m_preBufActive)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setEndOfTrack(true);
        m_pAudioThread->getMutex()->unlock();
    }
    else
    {
        emit bufferNextTrack(); //pre-buffering active; signal that buffering of following track can be initiated
    }
}

void Spotify::Session::streamingError( sp_session* session, sp_error error )
{
    Q_UNUSED(session);
    Spotify::Session::callingSession->streamingErrorWrapper(error);
}

void Spotify::Session::streamingErrorWrapper(sp_error error)
{
    Spotify::Error err(error);

    qDebug() << "Streaming error: " << err.toString();
}

void Spotify::Session::setPlaylistOfflineMode(Spotify::Playlist* pPlaylist, bool offline)
{
    //Set whether the specified playlist should be synchronised for offline usage (in case offline is set to true
    //the playlist will be made available offline; in case offline is false, the playlist will be made unavailable in offline mode)

    if(pPlaylist)
    {
        sp_playlist_set_offline_mode(m_session, pPlaylist->getPlaylist(), offline);
    }
}

void Spotify::Session::setAllowMobileSync(bool allow)
{
    //Set whether offline syncing is allowed over mobile (e.g., 2G/3G) connections
    setConnectionRule(SP_CONNECTION_RULE_ALLOW_SYNC_OVER_MOBILE, allow);
}

void Spotify::Session::offlineStatusUpdated( sp_session* session )
{
    //offline status updated callback
    Q_UNUSED(session);
    Spotify::Session::callingSession->offlineStatusUpdatedWrapper();
}

void Spotify::Session::offlineSynchronistationError( sp_session* session, sp_error error)
{//offline synchronisation errror status callback
    Q_UNUSED(session);
    Spotify::Session::callingSession->offlineSynchronisationErrorWrapper(error);
}

void Spotify::Session::offlineSynchronisationErrorWrapper( sp_error error )
{//offline synchronisation errror status updated
    QString strError;
    if(error!=SP_ERROR_OK)
    {//some sync. error has occured
        strError = sp_error_message(error);
        emit offlineSyncError(strError);
    }
}


void Spotify::Session::offlineStatusUpdatedWrapper()
{
    emit onOfflineStatusUpdated();
}

void Spotify::Session::OnOfflineStatusUpdated()
{
    //offline sync status updated
    qint32 tracksToSync;
    double progress;
    bool newSync = false;
    sp_offline_sync_status offlineStatus;

    if(sp_offline_sync_get_status(m_session,&offlineStatus)) //get offline sync status
    {
        tracksToSync = sp_offline_tracks_to_sync(m_session); //update the total number of tracks that need to be synchronised
        if(tracksToSync)
        {
            if(!m_offlineSyncing)
            {
                newSync = true; //sync status changed; new  sync operation
                m_offlineSyncing = true;
            }
            progress = (double)(((double)offlineStatus.copied_tracks / (double)(offlineStatus.copied_tracks+tracksToSync))*100.0);
            emit offlineSyncStatusUpdated(newSync,(qint32)progress); //emit progress notification (percentage completed)
        }
    }
    else
    {
        emit offlineSyncStatusUpdated(newSync,100); //sync completed
        m_offlineSyncing = false;  //reset sync operation state
    }
}

 void Spotify::Session::OnOnlineStateChanged(bool online)
 {
     //current online connectivity state has changed, e.g., online -> offline (boolean specifies new state e.g.,  true = online).
     setNetworkState(); //check / set network / connectivity state
 }

