//lastfmscrobbler.cpp: Implementation of the LastFMScrobbler class (according to the Last.FM submissions API 2.0).

#include "lastfmscrobbler.h"

LastFMScrobbler::LastFMScrobbler(QObject *parent) :
    QObject(parent)
{
    //set up the scrobbling interface / networking etc.

    m_pNetworkManager = new QNetworkAccessManager(this);

    m_pAuthReply = NULL;

    m_bCachingEnabled = false;

    //setup signal / slot connections
    connect(m_pNetworkManager,SIGNAL(finished(QNetworkReply*)),SLOT(OnRequestFinished(QNetworkReply*)));
}

LastFMScrobbler::~LastFMScrobbler()
{
}

bool LastFMScrobbler::isSessionValid()
 {
     //return true in case session valid (i.e., Last.fm session key retrieved)

    return(m_strSessionKey!="");
 }

bool LastFMScrobbler::getErrorResponse(const QByteArray& response, QString& strErrorDesc, qint32& code)
{
    //in case of error response, retrieve the associated last.fm error description and code. Returns TRUE in case error response can be parsed ok.

    QDomDocument xmlContent;

    xmlContent.setContent(response);

    QDomNodeList errorNodeList = xmlContent.elementsByTagName("error"); //get the error tag

    if(!errorNodeList.isEmpty())
    {
        if(errorNodeList.length()>0)
        {
            QDomNode errorNode = errorNodeList.item(0); //we expect only one error node in the response
            QDomNamedNodeMap attribNodes = errorNode.attributes(); //get attributes of the error node element
            QDomNode codeNode = attribNodes.namedItem("code");
            if(!codeNode.isNull())
            {
               QDomAttr codeAttr = codeNode.toAttr();
               code = codeAttr.nodeValue().toInt();
               qDebug() << "Code: " << code;
               QDomElement errorElem = errorNode.toElement();
               strErrorDesc = errorElem.text(); //get text of the error tag element
               return true;
            }   
        }
    }

    return false;
}


void LastFMScrobbler::OnRequestFinished(QNetworkReply *reply)
{
    //a network request has finished

    QByteArray responseData;
    qint32 keyTagStartPos, keyTagEndPos;
    QString strErrorDesc = "";
    qint32 errorCode = -1;

    responseData = reply->readAll(); //read response data

    qDebug() << "Received response data: " << responseData;

    if(reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt()==200)
    {
        //request successful (HTTP statuscode 200 OK received)
        if(reply==m_pAuthReply) //response to authentication request
        {
            qDebug() << "Reply to session authentication request";
            //retrieve and store the user's session key to be used in subsequent Last.fm API calls
            keyTagStartPos = responseData.indexOf("<key>");
            if(keyTagStartPos!=-1)
            {
              keyTagEndPos = responseData.indexOf("</key>");
              if(keyTagEndPos!=-1)
              {
                  m_strSessionKey = responseData.mid(keyTagStartPos+5,(keyTagEndPos-keyTagStartPos-5));
              }
            }
            emit authenticateSessionStatus(true,"");
        }
        else if(reply==m_pNowPlayingReply)
        {
            //reply to Now Playing request received
            emit updateNowPlayingStatus(true,"");
        }
        else if(reply==m_pScrobbleTracksReply)
        {//reply to scrobble tracks request received
            m_ScrobblesList.clear(); //clear the current scrobbles queue (sent to Last.fm successufully)
            cacheScrobbles(); //update cache status
            emit scrobbleTracksStatus(true,"");
        }
    }
    else
    {
        //some error occured; parse out the error description
        getErrorResponse(responseData,strErrorDesc,errorCode);
        if(strErrorDesc=="")
        {
            strErrorDesc = reply->errorString(); //get http error description
        }
        if(reply==m_pAuthReply)
        {
            emit authenticateSessionStatus(false,strErrorDesc);
        }
        else if(reply==m_pNowPlayingReply)
        {
            emit updateNowPlayingStatus(false,strErrorDesc);
        }
        else if(reply==m_pScrobbleTracksReply)
        {
            if(errorCode!=ERROR_SERVICE_OFFLINE && errorCode!=ERROR_SERVICE_TEMPORARILY_UNAVAILABLE)
            { //request failed (and should not be retried)
                m_ScrobblesList.clear();
            }
            cacheScrobbles(); //save the current scrobble queue state to cache
            emit scrobbleTracksStatus(false,strErrorDesc);
        }
    }

    reply->deleteLater();
}


void LastFMScrobbler::setAPIKey(const QString &strAPIKey)
{
    //set API key value
    m_strAPIKey = strAPIKey;
}

void LastFMScrobbler::setSecret(const QString &strSecret)
{
    //set secret value
    m_strSecret = strSecret;
}

void LastFMScrobbler::setUserAgent(const QString& strUserAgent)
{
    //set user agent
    m_strUserAgent = strUserAgent;
}

void LastFMScrobbler::setCachePath(const QString& strPath)
{
    //set path to directory to which queued scrobbles should be cached

    m_strCachePath = strPath;
}

void LastFMScrobbler::setCache(bool enabled)
{
    //enabled / disable use of caching (note: caching path should be explicitely set)

    m_bCachingEnabled  = enabled;

    if(m_bCachingEnabled)
    {
        restoreScrobbles(); //try to restore saved scrobbles
    }
}

QByteArray LastFMScrobbler::getAPISignature(const QMap<QString,QString> &paramMap)
{
    //constructs and returns an API method signature

    QByteArray result;

    QMap<QString,QString>::const_iterator it;

    for(it = paramMap.constBegin(); it != paramMap.constEnd(); it++)
    {
      result += it.key().toUtf8();
      result += it.value().toUtf8();
    }

    //append shared secret to the result string

    result += m_strSecret.toUtf8();

    //finally, return md5 hash of the resulting string

    return QCryptographicHash::hash(result,QCryptographicHash::Md5).toHex();
}

QNetworkReply* LastFMScrobbler::postRequest( QMap<QString, QString> paramMap)
{
    //send of a request to the Last.fm webservice. Parameter paramMap contains the parameter / value pairs
    //for constructing the request. Returns pointer to QNetworkReply instance (holds reply data for the current request).

    QByteArray requestData;

    QMap<QString,QString>::const_iterator it;

    //construct method signature for the API call (request)

    QByteArray apiSignature = getAPISignature(paramMap);

    for(it = paramMap.constBegin(); it != paramMap.constEnd(); it++)
    {
        requestData += it.key().toUtf8();
        requestData += "=";
        requestData += QUrl::toPercentEncoding(it.value());
        requestData += "&";
    }

    requestData += "api_sig=";
    requestData += apiSignature;

    QUrl requestUrl;
    requestUrl.setUrl(LASTFM_SCROBBLER_URL);


    QNetworkRequest req;
    req.setUrl(requestUrl);
    if(m_strUserAgent!="")
    {
        req.setRawHeader("User-Agent", m_strUserAgent.toUtf8());
    }

    return(m_pNetworkManager->post(req,requestData)); //post the request, and return pointer to reply instance
}

void LastFMScrobbler::authenticateSession(const QString& strUserID, const QString& strPassword)
{
    //request a new session using the supplied Last.FM user credentials.

    QByteArray authToken;
    QMap<QString, QString> params;

    authToken = QCryptographicHash::hash(strUserID.toLower().toUtf8()+QCryptographicHash::hash(strPassword.toUtf8(),QCryptographicHash::Md5).toHex()
                                            ,QCryptographicHash::Md5).toHex(); //calculate authentication token as md5 hash

    params.insert("api_key",m_strAPIKey);
    params.insert("authToken",authToken);
    params.insert("method",METHOD_AUTH_SESSION);
    params.insert("username",strUserID);

    m_strSessionKey = ""; //empty the current session key

    m_pAuthReply = postRequest(params); //finally, send of the request
}

void LastFMScrobbler::updateNowPlaying(const QString& strTrackName, const QString& strArtistName, const QString& strAlbumName, const QString& strAlbumArtist)
{
    //update the client 'now playing' status

     QMap<QString, QString> params;

     m_currTrack.m_timestamp = QDateTime::currentDateTime(); //get current date / time(i.e., time when playback started)

     m_currTrack.m_strTrackName = strTrackName;
     m_currTrack.m_strArtistName = strArtistName;
     m_currTrack.m_strAlbumName = strAlbumName;
     if(m_currTrack.m_strArtistName != strAlbumArtist)
         m_currTrack.m_strAlbumArtist = strAlbumArtist;

     //define API call parameters
     params.insert("method",METHOD_UPDATE_NOW_PLAYING);
     params.insert("artist",m_currTrack.m_strArtistName);
     params.insert("track",m_currTrack.m_strTrackName);
     params.insert("album",m_currTrack.m_strAlbumName);
     if(m_currTrack.m_strAlbumArtist!="")
        params.insert("albumArtist",m_currTrack.m_strAlbumArtist);
     params.insert("sk",m_strSessionKey); //current session key for user
     params.insert("api_key",m_strAPIKey);

     m_pNowPlayingReply = postRequest(params); //finally, send of the request
}

void LastFMScrobbler::scrobbleTrack()
{
    //scrobble the current track

    if(!m_currTrack.m_timestamp.isNull())
    {
       m_ScrobblesList.append(m_currTrack); //append to scrobble list / queue
       m_currTrack.m_timestamp = QDateTime(); //reset timestamp (to null date/time)
   }
    sendScrobbles(); //send scrobbles in queue
}

void LastFMScrobbler::sendScrobbles()
{
    //private method handling the actual sending of scrobbled tracks to Last.Fm webservice

    QMap<QString, QString> params;
    TrackData track;

    params.insert("method",METHOD_SCROBBLE_TRACKS);

    for(int pos = 0; pos < m_ScrobblesList.size(); pos++)
    {
        track = m_ScrobblesList.at(pos);
        params.insert("artist["+QString::number(pos)+ "]",track.m_strArtistName);
        params.insert("track["+QString::number(pos)+ "]",track.m_strTrackName);
        params.insert("album["+QString::number(pos)+ "]",track.m_strAlbumName);
        if(track.m_strAlbumArtist!="")
            params.insert("albumArtist["+QString::number(pos)+ "]",track.m_strAlbumArtist);
        params.insert("timestamp["+QString::number(pos) + "]",QString::number(track.m_timestamp.toUTC().toTime_t())); //timestamp should be in unix-timestamp format (UTC)
    }

    params.insert("sk",m_strSessionKey);
    params.insert("api_key",m_strAPIKey);

    m_pScrobbleTracksReply = postRequest(params); //send of scrobble request
}

void LastFMScrobbler::cacheScrobbles()
{
    //handles caching of currently queued scrobbles to disk

    QFile cacheFile(m_strCachePath + SCROBBLES_CACHE_FILENAME); //file instance holding providing access to serialization data file

    cacheFile.open(QIODevice::WriteOnly); //opens backup file for writing only (the file is created if it does not already exist)

    QDataStream outStream(&cacheFile); //output stream for data serialization (attach to backup file instance)

    outStream.setVersion(QDataStream::Qt_4_6); //set stream version

    outStream << m_ScrobblesList; //serialize current scrobbles queue to disk

    cacheFile.close(); //finally, close the backup file
}

void LastFMScrobbler::restoreScrobbles()
{
     //restore cached scrobbles from disk

    QFile cacheFile(m_strCachePath + SCROBBLES_CACHE_FILENAME); //file instance (backup file from which serialized data should be restored)

    if(cacheFile.exists())
    {
        m_ScrobblesList.clear();

        cacheFile.open(QIODevice::ReadOnly); //open file in read-only mode

        QDataStream inStream(&cacheFile); //associate file instance with input data stream for de-serialization

        inStream >> m_ScrobblesList; //de-serialize data

        qDebug() << "Restored: "  << m_ScrobblesList.count();

        cacheFile.close();
    }
}

QDataStream &operator<<(QDataStream &stream, const TrackData& track)
{
    //save instance to datastream
    stream << track.m_strTrackName;
    stream << track.m_strArtistName;
    stream << track.m_strAlbumName;
    stream << track.m_strAlbumArtist;
    stream << track.m_timestamp;

    return stream;
}

QDataStream &operator>>(QDataStream &stream, TrackData &track)
{
    //restore instance from datastream
    stream >> track.m_strTrackName;
    stream >> track.m_strArtistName;
    stream >> track.m_strAlbumName;
    stream >> track.m_strAlbumArtist;
    stream >> track.m_timestamp;

    return stream;
}
