#include "converter.h"

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <iostream>

#include <QHash>
#include <QStringList>

#define QSTR(_i) (_i).toAscii().constData()
#define QPRINTF(_str, _val)	printf("%s: |%s|\n", (_str), (_val).toAscii().constData())

Converter::Converter()
{
	data.insert("_TYPES_", new type_data_t);
}

void Converter::add_file(char *file)
{
	FILE *fp;
	int len;
	char buf[256];
	char dat[256];
	QString *typestr = NULL;
	QString *unitstr = NULL;
	bool dosi = false;

	// open for read
	fp = fopen(file, "r");
	if (fp == NULL) {
		printf("oops, opening |%s| returned NULL!\n", file);
		return;
	}

	// read file
	do {
		// get line
		fgets(buf, 256, fp);

		// chomp
		len = strlen(buf) - 1;
		buf[len] = 0;

		// new unit type
		if (strncmp(buf, "TYPE", 4) == 0) {
			// get name
			strncpy(dat, buf+6, len-6-1);
			dat[len-6-1] = 0;
			typestr = new QString(dat);

			// add to types order
			data.value("_TYPES_")->order.append(*typestr);

			// create data struct
			data.insert(*typestr, new type_data_t);

		// new unit
		} else if (typestr && (strncmp(buf, "UNIT", 4) == 0)) {
			// get name
			strncpy(dat, buf+6, len-6-1);
			dat[len-6-1] = 0;
			unitstr = new QString(dat);

			// add to unit order
			data.value(*typestr)->order.append(*unitstr);

		// last unit is a 'base' unit
		} else if (unitstr && (strncmp(buf, "BASE", 4) == 0)) {
			// insert normal base unit
			if (strncmp(buf+5, "NORMAL", 6) == 0) {
				data.value(*typestr)->eqnf.insert(*unitstr, QString("val"));
				data.value(*typestr)->eqnt.insert(*unitstr, QString("val"));

			// insert si base unit
			} else if (strncmp(buf+5, "SI_PREFIX", 9) == 0) {
				data.value(*typestr)->eqnf.insert(*unitstr, QString("val"));
				data.value(*typestr)->eqnt.insert(*unitstr, QString("val"));
				add_si_units(typestr, unitstr);

			// something bad happened
			} else {
				printf("error parsing line |%s|\n", buf);
			}

			// clean up
			delete unitstr;

		// not a base unit, but has si prefixes
		} else if (unitstr && (strncmp(buf, "DOSI", 4) == 0)) {
			dosi = true;

		// got eqnf
		} else if (unitstr && (strncmp(buf, "EQNF", 4) == 0)) {
			data.value(*typestr)->eqnf.insert(*unitstr, QString(buf+5));

		// got eqnt
		} else if (unitstr && (strncmp(buf, "EQNT", 4) == 0)) {
			data.value(*typestr)->eqnt.insert(*unitstr, QString(buf+5));

			// check if we need to add si prefix
			if (dosi) {
				add_si_units(typestr, unitstr);
			}

			// clean up
			dosi = false;
			delete unitstr;
		}
	} while (!feof(fp));

	fclose(fp);
}

#define NUM_PREFIXES 20

Converter::si_prefix_t Converter::prefixes[NUM_PREFIXES] = {
	{ QString("Deka"),	1	},
	{ QString("Hecto"),	2	},
	{ QString("Kilo"),	3	},
	{ QString("Mega"),	6	},
	{ QString("Giga"),	9	},
	{ QString("Tera"),	12	},
	{ QString("Peta"),	15	},
	{ QString("Exa"),	18	},
	{ QString("Zetta"),	21	},
	{ QString("Yotta"),	24	},
	{ QString("Deci"),	-1	},
	{ QString("Centi"),	-2	},
	{ QString("Milli"),	-3	},
	{ QString("Micro"),	-6	},
	{ QString("Nano"),	-9	},
	{ QString("Pico"),	-12	},
	{ QString("Femto"),	-15	},
	{ QString("Atto"),	-18	},
	{ QString("Zepto"),	-21	},
	{ QString("Yocto"),	-24	},
};

void Converter::add_si_units(QString *type, QString *unit)
{
	int i;
	QString name;
	QString fact;
	int posn;
	QString new_formula;

	// loop over all prefixes
	for (i = 0; i < NUM_PREFIXES; i++) {
		// create new unit name and equation
		name = prefixes[i].prefix + unit->toLower();
		fact = QString::number(pow(10, prefixes[i].power));

		// add to order
		data.value(*type)->order.append(name);

		// create new eqnf
		posn = data.value(*type)->eqnf.value(*unit).indexOf("val");
		if (posn != -1) {
			new_formula = QString(data.value(*type)->eqnf.value(*unit));
			new_formula.insert(posn + 3, "," + QString(fact) + ",*");
			data.value(*type)->eqnf.insert(name, new_formula);
		}

		// create new eqnt
		posn = data.value(*type)->eqnt.value(*unit).indexOf("val");
		if (posn != -1) {
			new_formula = QString(data.value(*type)->eqnt.value(*unit));
			new_formula.insert(posn + 3, "," + QString(fact) + ",/");
			data.value(*type)->eqnt.insert(name, new_formula);
		}
	}
}

double Converter::rpn_eval(QString type, double val, QString formula)
{
	QStringList steps = formula.split(",", QString::SkipEmptyParts);
	QList<double> rpn;
	double b, a;

	QStringList::const_iterator i;
	for (i = steps.constBegin(); i != steps.constEnd(); i++) {
		if (*i == "val") {
			rpn.append(val);
		} else if (*i == "+") {
			b = rpn.takeLast();
			a = rpn.takeLast();
			rpn.append(a + b);
		} else if (*i == "-") {
			b = rpn.takeLast();
			a = rpn.takeLast();
			rpn.append(a - b);
		} else if (*i == "*") {
			b = rpn.takeLast();
			a = rpn.takeLast();
			rpn.append(a * b);
		} else if (*i == "/") {
			b = rpn.takeLast();
			a = rpn.takeLast();
			rpn.append(a / b);
		} else if ((*i).startsWith("CONVF:")) {
			a = rpn.takeLast();
			rpn.append(rpn_eval(type, a, data.value(type)->eqnf.value((*i).mid(6))));
		} else if ((*i).startsWith("CONVT:")) {
			a = rpn.takeLast();
			rpn.append(rpn_eval(type, a, data.value(type)->eqnt.value((*i).mid(6))));
		} else {
			rpn.append((*i).toDouble());
		}
	}

	if (rpn.size() > 1) {
		printf("rpn_eval(%s, %lf, %s): %d items on stack!\n", QSTR(type), val, QSTR(formula), rpn.size());
	}
	return rpn.takeLast();
}

QString Converter::convert(QString type, QString ou, QString nu, QString val)
{
	QStringList formula;
	double tmp;

	// check that the value is not blank
	if (val.size() == 0) {
		return QString('?');
	}

	// convert the 'from' unit to the base unit
	tmp = rpn_eval(type, val.toDouble(), data.value(type)->eqnf.value(ou));

	// convert the base unit to the 'to' unit
	tmp = rpn_eval(type, tmp, data.value(type)->eqnt.value(nu));

	// get rid of exponential notation
	return QString::number(tmp, 'f', 12);
}

QStringList Converter::get_order(QString type)
{
	return data.value(type)->order;
}
